#' Deletes track record including all of the related child records in PostgreSQL/PostGIS GPS database.
#'
#' Deletes track record including all of the related child records in PostgreSQL/PostGIS GPS database. Returns dataframe with number of records deleted.
#' @param Operation Name of operation.
#' @param Track_UUID UUID of track to be deleted.
#' @param Track_RawFile Name of original data file of the GPS track.
#' @param DBType Parameter for specifying for which database type to delete the track records from. Values can be 'Main' (default), 'NodeJS' or 'Old'.
#' @param UserConfirm Flag whether the user needs to actively confirm deletion. Default value is TRUE.
#' @param Verbose Switch for printing of query. Default value is FALSE.
#' @param SuppressPostgreSQLWarnings Switch for turning warnings off. Default value is TRUE.
#' @export

deleteTrackRecords <- function(Operation, Track_UUID=NA, Track_RawFile=NA, DBType="Main", UserConfirm=T, Verbose=F, SuppressPostgreSQLWarnings=T) {

  DBType <- toupper(DBType)
  
  ## Check input parameter validity
  if (!is.na(Track_RawFile) & is.na(Track_UUID)) {

    Query <- paste0("SELECT uuid FROM gps.tracks WHERE raw_file='", Track_RawFile, "'")
    Track_UUID <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$uuid[1]

  } else if (is.na(Track_RawFile) & !is.na(Track_UUID)) {

    ## Do nothing

  } else {

    stop("Parameters specified incorectly: Input parameter needs to be either a uuid or the name of the original raw data file.")

  }

  ## Check for proper UUID
  if (is.null(Track_UUID)) {

    warning("No track exists in database for this raw file value.", immediate. = T)
    return(NA)

  ## If UUID exists
  } else {

    ## Count records before delete
    DF_CountBefore <- countTrackRecords(Operation, Track_UUID=Track_UUID, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
    names(DF_CountBefore) <- c("Table", "Before")

    ## Check whether track was found in database
    if (DF_CountBefore$Before[DF_CountBefore$Table=="gps.tracks"]>0) {

      ## User confirmation
      if (UserConfirm==F) {
        UserResponse <- "Yes"
      } else {
        print(DF_CountBefore)
        UserResponse <- readline("Type 'Yes' to confirm deletion: ")
      }

      ## Deletion in response to user confirmation
      if (UserResponse=="Yes") {

        ## 1) Delete obspoints
        if (DBType!="NODEJS") {
          Query <- paste0("DELETE FROM gps.obspoints ",
                                     "USING gps.tracks, gps.tracks_days ",
                                     "WHERE gps.tracks_days.gpstracks_uuid = gps.tracks.uuid AND ",
                                           "gps.obspoints.gpstracksdays_uuid = gps.tracks_days.uuid AND ",
                                           "gps.tracks.uuid = '", Track_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        }
        
        
        ## 2) Delete runs_pausepoints
        Query <- paste0("DELETE FROM gps.runs_pausepoints ",
                                    "USING gps.runs, gps.tracks_days, gps.tracks ",
                                    "WHERE gps.runs_pausepoints.gpsruns_uuid = gps.runs.uuid AND ",
                                          "gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid AND ",
                                          "gps.tracks_days.gpstracks_uuid = gps.tracks.uuid AND ",
                                          "gps.tracks.uuid = '", Track_UUID, "'")
        sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        
        
        ## 3) Delete gis.gpsruns_array_xxxx
        if (DBType!="NODEJS") {
        
          Query <- paste0("DELETE FROM gis.gpsruns_array_coord ",
                          "USING gps.runs, gps.tracks_days, gps.tracks ",
                          "WHERE gis.gpsruns_array_coord.gpsruns_uuid = gps.runs.uuid AND ",
                          "gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid AND ",
                          "gps.tracks_days.gpstracks_uuid = gps.tracks.uuid AND ",
                          "gps.tracks.uuid = '", Track_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_aspect ",
                          "USING gps.runs, gps.tracks_days, gps.tracks ",
                          "WHERE gis.gpsruns_array_aspect.gpsruns_uuid = gps.runs.uuid AND ",
                          "gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid AND ",
                          "gps.tracks_days.gpstracks_uuid = gps.tracks.uuid AND ",
                          "gps.tracks.uuid = '", Track_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_avexpo ",
                          "USING gps.runs, gps.tracks_days, gps.tracks ",
                          "WHERE gis.gpsruns_array_avexpo.gpsruns_uuid = gps.runs.uuid AND ",
                          "gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid AND ",
                          "gps.tracks_days.gpstracks_uuid = gps.tracks.uuid AND ",
                          "gps.tracks.uuid = '", Track_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_elev ",
                          "USING gps.runs, gps.tracks_days, gps.tracks ",
                          "WHERE gis.gpsruns_array_elev.gpsruns_uuid = gps.runs.uuid AND ",
                          "gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid AND ",
                          "gps.tracks_days.gpstracks_uuid = gps.tracks.uuid AND ",
                          "gps.tracks.uuid = '", Track_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_gullies ",
                          "USING gps.runs, gps.tracks_days, gps.tracks ",
                          "WHERE gis.gpsruns_array_gullies.gpsruns_uuid = gps.runs.uuid AND ",
                          "gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid AND ",
                          "gps.tracks_days.gpstracks_uuid = gps.tracks.uuid AND ",
                          "gps.tracks.uuid = '", Track_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_incline ",
                          "USING gps.runs, gps.tracks_days, gps.tracks ",
                          "WHERE gis.gpsruns_array_incline.gpsruns_uuid = gps.runs.uuid AND ",
                          "gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid AND ",
                          "gps.tracks_days.gpstracks_uuid = gps.tracks.uuid AND ",
                          "gps.tracks.uuid = '", Track_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_overheadhzd ",
                          "USING gps.runs, gps.tracks_days, gps.tracks ",
                          "WHERE gis.gpsruns_array_overheadhzd.gpsruns_uuid = gps.runs.uuid AND ",
                          "gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid AND ",
                          "gps.tracks_days.gpstracks_uuid = gps.tracks.uuid AND ",
                          "gps.tracks.uuid = '", Track_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_veg ",
                          "USING gps.runs, gps.tracks_days, gps.tracks ",
                          "WHERE gis.gpsruns_array_veg.gpsruns_uuid = gps.runs.uuid AND ",
                          "gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid AND ",
                          "gps.tracks_days.gpstracks_uuid = gps.tracks.uuid AND ",
                          "gps.tracks.uuid = '", Track_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        }
         
         
        ## 4) Delete gpsruns_segements_elevband
        Query <- paste0("DELETE FROM gis.gpsruns_segments_elevband ",
                        "USING gps.runs, gps.tracks_days, gps.tracks ",
                        "WHERE gis.gpsruns_segments_elevband.gpsruns_uuid = gps.runs.uuid AND ",
                        "gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid AND ",
                        "gps.tracks_days.gpstracks_uuid = gps.tracks.uuid AND ",
                        "gps.tracks.uuid = '", Track_UUID, "'")
        sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        
        
        ## 5) Delete runs
        Query <- paste0("DELETE FROM gps.runs ",
                             "USING gps.tracks_days, gps.tracks ",
                             "WHERE gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid AND ",
                                   "gps.tracks_days.gpstracks_uuid = gps.tracks.uuid AND ",
                                   "gps.tracks.uuid = '", Track_UUID, "'")
        sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        
        
        ## 6) Delete potruns_skipped
        if (DBType!="NODEJS") {
          Query <- paste0("DELETE FROM gps.potruns_skipped ", 
                                  "USING gps.tracks_days, gps.tracks ",
                                  "WHERE gps.potruns_skipped.gpstracksdays_uuid = gps.tracks_days.uuid AND ",
                                        "gps.tracks_days.gpstracks_uuid = gps.tracks.uuid AND ",
                                        "gps.tracks.uuid = '", Track_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        }
        
        
        ## 7) Delete track_day
        Query <- paste0("DELETE FROM gps.tracks_days WHERE gpstracks_uuid='", Track_UUID, "'")
        sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

        
        ## 8) Delete tracK
        Query <- paste0("DELETE FROM gps.tracks WHERE uuid='", Track_UUID, "'")
        sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

        
        ## Count records after delete
        DF_CountAfter <- countTrackRecords(Operation, Track_UUID=Track_UUID, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        names(DF_CountAfter) <- c("Table", "After")

        ## Merge before and after table
        DF_Count <- merge(DF_CountBefore, DF_CountAfter)
        return(DF_Count)

      } else {

        warning("No records were deleted.", immediate. = T)
        return(NA)

      }

    } else {

      warning("No track was found for the provided information.", immediate. = T)
      return(NA)

    }

  }

}
