#' Deletes tracksday record including all of the related child records in PostgreSQL/PostGIS GPS database
#'
#' Deletes tracksday record including all of the related child records in PostgreSQL/PostGIS GPS database. Returns dataframe with number of records deleted. Input parameter needs to be either be a uuid or a date and unit.
#' @param Operation Name of operation.
#' @param TrackDay_UUID UUID of trackday to be deleted
#' @param TrackDay_Date Date of trackday to be deleted
#' @param TrackDay_Unit Unit of trackday to be deleted
#' @param DBType Parameter for specifying for which database type to delete the track records from. Values can be 'Main' (default), 'NodeJS' or 'Old'.
#' @param UserConfirm Flag whether the user needs to actively confirm deletion. True by default.
#' @param Verbose Switch for printing of query. Default value is FALSE.
#' @param SuppressPostgreSQLWarnings Switch for turning warnings off Default value is TRUE.
#' @export

deleteTrackdayRecords <- function(Operation, TrackDay_UUID=NA, TrackDay_Date=NA, TrackDay_Unit=NA, DBType="Main", UserConfirm=T, Verbose=F, SuppressPostgreSQLWarnings=T) {

  DBType <- toupper(DBType)
  
  ## Check input parameter validity
  if (is.na(TrackDay_UUID) & !is.na(TrackDay_Date) & !is.na(TrackDay_Unit)) {

    Query <- paste0("SELECT gps.tracks_days.uuid FROM gps.tracks INNER JOIN gps.tracks_days ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                    "WHERE gps.tracks.unit = '", TrackDay_Unit, "' AND ",
                    "gps.tracks_days.date_local = '", TrackDay_Date, "'")
    TrackDay_UUID <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$uuid[1]

  } else if (!is.na(TrackDay_UUID) & is.na(TrackDay_Date) & is.na(TrackDay_Unit)) {

    ## Do nothing because TrackDay_UUID already has correct values

  } else {

    stop("Parameters specified incorrectly: Input parameter needs to be either be a uuid or a date and unit.")
  }

  ## Check for proper UUID
  if (is.null(TrackDay_UUID)) {

    warning("No track day exists in database for this combination of unit and date values.", immediate.=T)
    return(NA)

  ## If UUID exists
  } else {

    ## Count records before delete
    DF_CountBefore <- countTrackdayRecords(Operation, TrackDay_UUID=TrackDay_UUID, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
    names(DF_CountBefore) <- c("Table", "Before")

    ## Check whether trackday was found
    if (DF_CountBefore$Before[DF_CountBefore$Table=="gps.tracks_days"]>0) {

      ## User confirmation
      if (UserConfirm==F) {
          UserResponse <- "Yes"
      } else {
          print(DF_CountBefore)
          UserResponse <- readline("Type 'Yes' to confirm deletion: ")
      }

      ## Deletion in response to user confirmation
      if (UserResponse=="Yes") {

        ## 1) Delete obspoints
        if (DBType!="NODEJS") {
          Query <- paste0("DELETE FROM gps.obspoints WHERE gpstracksdays_uuid='", TrackDay_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        }

        ## 2) Delete runs_pausepoints
        Query <- paste0("DELETE FROM gps.runs_pausepoints USING gps.runs WHERE gps.runs_pausepoints.gpsruns_uuid = gps.runs.uuid AND gps.runs.gpstracksdays_uuid = '", TrackDay_UUID, "'")
        sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

        
        ## 3) Delete Delete gis.gpsruns_array_xxxx
        if (DBType!="NODEJS") {
          Query <- paste0("DELETE FROM gis.gpsruns_array_coord USING gps.runs WHERE gis.gpsruns_array_coord.gpsruns_uuid = gps.runs.uuid AND gps.runs.gpstracksdays_uuid = '", TrackDay_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_aspect USING gps.runs WHERE gis.gpsruns_array_aspect.gpsruns_uuid = gps.runs.uuid AND gps.runs.gpstracksdays_uuid = '", TrackDay_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_avexpo USING gps.runs WHERE gis.gpsruns_array_avexpo.gpsruns_uuid = gps.runs.uuid AND gps.runs.gpstracksdays_uuid = '", TrackDay_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_elev USING gps.runs WHERE gis.gpsruns_array_elev.gpsruns_uuid = gps.runs.uuid AND gps.runs.gpstracksdays_uuid = '", TrackDay_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_gullies USING gps.runs WHERE gis.gpsruns_array_gullies.gpsruns_uuid = gps.runs.uuid AND gps.runs.gpstracksdays_uuid = '", TrackDay_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_incline USING gps.runs WHERE gis.gpsruns_array_incline.gpsruns_uuid = gps.runs.uuid AND gps.runs.gpstracksdays_uuid = '", TrackDay_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_overheadhzd USING gps.runs WHERE gis.gpsruns_array_overheadhzd.gpsruns_uuid = gps.runs.uuid AND gps.runs.gpstracksdays_uuid = '", TrackDay_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
          
          Query <- paste0("DELETE FROM gis.gpsruns_array_veg USING gps.runs WHERE gis.gpsruns_array_veg.gpsruns_uuid = gps.runs.uuid AND gps.runs.gpstracksdays_uuid = '", TrackDay_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        }        
        
        ## 4) Delete gpsruns_segements_elevband
        Query <- paste0("DELETE FROM gis.gpsruns_segments_elevband USING gps.runs WHERE gis.gpsruns_segments_elevband.gpsruns_uuid = gps.runs.uuid AND gps.runs.gpstracksdays_uuid = '", TrackDay_UUID, "'")
        sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        
        
        ## 5) Delete runs
        Query <- paste0("DELETE FROM gps.runs WHERE gpstracksdays_uuid='", TrackDay_UUID, "'")
        sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        
        
        ## 6) Delete potruns_skipped
        if (DBType!="NODEJS") {
          Query <- paste0("DELETE FROM gps.potruns_skipped WHERE gpstracksdays_uuid='", TrackDay_UUID, "'")
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        }
        
        
        ## 7) Delete trackday
        Query <- paste0("DELETE FROM gps.tracks_days WHERE uuid='", TrackDay_UUID, "'")
        sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

        
        ## Count records after delete
        DF_CountAfter <- countTrackdayRecords(Operation, TrackDay_UUID=TrackDay_UUID, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        names(DF_CountAfter) <- c("Table", "After")

        ## Merge before and after table
        DF_Count <- merge(DF_CountBefore, DF_CountAfter)
        return(DF_Count)

      } else {

        warning("No records were deleted.", immediate. = T)
        return(NA)

      }

    } else {

      warning("No trackday was found for the provided information.", immediate. = T)
      return(NA)
    }

  }

}
