#' Extracts flihgts from GPS ObsPoints stured in DB and write line geometries into gps.flights table
#'
#' Extracts flihgts from GPS ObsPoints stured in DB and write line geometries into gps.flights table
#' @param Operation Name of Operation. Used for accessing PostgreSQL database.
#' @param MinFlyignSpeed Minimum flying speed in km/h. Default is 75 km/h.
#' @param MinTimeDiffinMin Minimum time difference between flights in minutes . Default is 5 min.
#' 
#' @export
#' 
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "CMHGL"
#' extractFlightsFromGPSObsPoints(Operation)

extractFlightsFromGPSObsPoints <- function(Operation, MinFlyingSpeed=75, MinTimeDiffinMin=5) {

  ## Get units
  Query <- "Select unit, count(uuid) AS CountTracks from gps.tracks group by unit order by unit"
  Units <- getRecordsFromQuery(Operation, Query)
  
  ## Unit loop
  for (Index_Unit in 1:nrow(Units)) {
  
    Unit <- Units$unit[Index_Unit]
  
    ## Get track days per unit
    Query <- paste0("SELECT gps.tracks_days.uuid, gps.tracks_days.date_local ", 
                    "FROM gps.tracks_days INNER JOIN gps.tracks ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                    "where gps.tracks.unit = '", Unit, "'")
    UnitTrackDays <- getRecordsFromQuery(Operation, Query)
    
    ## Trackday loop
    for (Index_Trackday in 1:nrow(UnitTrackDays)) {
      
      print(paste0("PROCESSING UNIT ", Unit, " ON ", UnitTrackDays$date_local[Index_Trackday], " ..."))
    
      TrackDayUUID <- UnitTrackDays$uuid[Index_Trackday]
      
      ## Check whether flight alreadt exist
      Query <- paste0("SELECT COUNT(gps.flights.uuid) AS num_flights FROM gps.flights WHERE gpstracksdays_uuid ='", TrackDayUUID, "'") 
      NumExistFlights <- getRecordsFromQuery(Operation, Query)$num_flights
      
      if (NumExistFlights>0) {
        
        print(paste0("Skipping because ", NumExistFlights, " flights already exist!"))
        
      } else {
      
        ## Get flight GPS points
        Query <- paste0("SELECT uuid, ST_x(geom) AS lon, ST_y(geom) AS lat, datetime_local, hspeed_kmh FROM gps.obspoints WHERE ((gpstracksdays_uuid='", TrackDayUUID, "') AND (gpsruns_uuid IS NULL) AND (activity='Flying') AND (hspeed_kmh > ", MinFlyingSpeed, ")) order by datetime_local")
        FlightPoints <- getRecordsFromQuery(Operation, Query)
        
        if(nrow(FlightPoints)>0) {
        
          FlightPoints$gpsflights_uuid <- NA
          
          ## Calculate time diff
          FlightPoints$timediff <- NA
          for (Index_Point in 2:nrow(FlightPoints)) {
            FlightPoints$timediff[Index_Point] <- difftime(FlightPoints$datetime_local[Index_Point], FlightPoints$datetime_local[Index_Point-1], units = "mins")
          }
          
          ## Specify break points
          IndexSepPoints <- which(FlightPoints$timediff>MinTimeDiffinMin)
          
          if (length(IndexSepPoints)>0) {
          
            ## Start and EndIndex for individual flights
            IndexStart <- c(1,which(FlightPoints$timediff>MinTimeDiffinMin))
            IndexEnd <- c(which(FlightPoints$timediff>MinTimeDiffinMin), nrow(FlightPoints))
            
            ## Extract individual flight path
            for(IndexFlight in 1:length(IndexStart)) {
              
              ## Create uuid
              Flight_uuid <- createUuid()
              
              ## Assign uuid
              FlightPoints[(IndexStart[IndexFlight]:IndexEnd[IndexFlight]),]$gpsflights_uuid <- Flight_uuid
              
              ## Create spatial line
              Flight_DF <- FlightPoints[(IndexStart[IndexFlight]:IndexEnd[IndexFlight]),]
              
              if (nrow(Flight_DF)>1) {
              
                Flight_Line <- createSpatialLineFromCoord(Flight_DF, Flight_uuid, x="lon", y="lat")
                
                ## Create dataframe
                Flight_Data <- data.frame(uuid=Flight_uuid,
                                          gpstracksdays_uuid=TrackDayUUID,
                                          date_local=substr(Flight_DF$datetime_local[1], 1, 10),
                                          date_local_esri=substr(Flight_DF$datetime_local[1], 1, 10),
                                          flightnum=IndexFlight,
                                          flightnum_rev=NA,
                                          datetime_local_start=Flight_DF$datetime_local[1],
                                          datetime_local_end=Flight_DF$datetime_local[nrow(Flight_DF)],
                                          numpoints=nrow(Flight_DF),
                                          timediff_min=NA,
                                          flyspd_avg=mean(Flight_DF$hspeed_kmh),
                                          flyspd_max=max(Flight_DF$hspeed_kmh),
                                          length_m=NA)
              
                row.names(Flight_Data) <- Flight_uuid
              
                ## Putting together SPDF  
                if (exists("SPDF_Flight")) {
                  SPDF_Flight <- spRbind(SPDF_Flight, SpatialLinesDataFrame(Flight_Line, Flight_Data))
                } else {
                  SPDF_Flight <- SpatialLinesDataFrame(Flight_Line, Flight_Data)
                }
                rm(Flight_Line, Flight_Data)
                
              } ## end of if condition checking out whether line has more than 1 point.
              
              rm(Flight_DF)
            
            } ## end of for loop for flight creation
            
            ## Finalize SPDF
            SPDF_Flight$timediff_min <- difftime(SPDF_Flight$datetime_local_end, SPDF_Flight$datetime_local_start, units = "mins")
            SPDF_Flight$length_m <- round(gLength(spTransform(SPDF_Flight, CRS(getLocalSRDI(Operation)$proj4text)), byid=T),0)
            SPDF_Flight$flightnum_rev <- rev(SPDF_Flight$flightnum)
          
            print(paste0(nrow(SPDF_Flight), " flights detected."))
            
            ## Updating database: SPDF
            writeSPDFToGPSDB(SPDF_Flight, Operation, "gps", "flights", ShowCounter=25)
            
            ## Updating database: PointObs
            FlightPoints <- FlightPoints[(FlightPoints$gpsflights_uuid %in% SPDF_Flight$uuid),c("uuid", "gpsflights_uuid")]
            updateTblInGPSDB(FlightPoints, Operation, "gps", "obspoints", WriteToHistTbl = F, ShowCounter=250)
            
            ## Cleanup
            rm(SPDF_Flight)
            rm(FlightPoints)
              
          } else {  ## end of if condition checking if any flight detected
          
            print("No flights detected.")
            
          } ## end of if condition checking if any flight detected: no flights detected
          
        } else {  ## end of check whether any flight points retrieved
          
          print("No flight points retrieved.")
          
        } ## end of check whether any flight points retrieved: no flight points
        
      } ## end of check whether flight tracks already exist
        
      print("") # Separator
      
    } ## end of loop for tracks days
  
  } ## end of loop for units
    
}