#' Extract coordinates of raster cells along ski run
#'
#' Extract coordinates of raster cells within a given buffer along GPS ski run.
#' @param Operation Name of operation
#' @param RunUUID UUID of single run to be processed. Can only contain either a RunUUID or a RunLine.
#' @param RunLine Single object of class to be processed. Can only contain either a RunUUID or a RunLine.
#' @param Raster Object of class 'RasterLayer' (package raster)
#' @param BufferDist Size of buffer around GPS ski run in metres. Can be single value or array of values. Default value is 0.
#' @return Dataframe with coordinates, buffer distance and distance from starting and end point of run.
#'
#' @export

extractRasterCoordAlongGPSRun <- function(Operation, RunUUID, Raster, Local_proj4text, BufferDist=0) {
  
  ## Check input parameters
  if(class(Raster)[1]!="RasterLayer") {stop("Parameter 'Raster' needs to be of class 'RasterLayer'!!")}
  
  ## Get spatial object
  RunLine <- getSpatialObjectsFromUUID(Operation, "gps", "runs", UUID=RunUUID)
  RunPoints <- getSpatialObjectsDataFrameFromUUID(Operation, "gps", "runs_pausepoints", ForeignUUID=RunUUID, ForeignUUIDCol="gpsruns_uuid")
  RunStartCoord <- RunPoints[RunPoints@data$activity=="GettingReady",]@coords
  RunEndCoord <- RunPoints[RunPoints@data$activity=="PackingUp",]@coords
  
  RasterCellPoints <- data.frame(lon=numeric(0),
                                 lat=numeric(0),
                                 buffer=numeric(0))
  
  ## Check for record
  if(length(RunLine)==0) {
    
    warning(paste0("Unable to get run line record for run with uuid '", RunUUID, "'!"),immediate. = T)
    
  } else {
  
    ## Process buffers
    for (Index_Buffer in 1:length(BufferDist)) {
      
      if (BufferDist[Index_Buffer]==0) {
        
        CellNumbers <- cellFromLine(Raster, RunLine)[[1]]
        
      } else {
        
        RunLineProjected <- spTransform(RunLine, Local_proj4text)
        BufferProjected <- buffer(RunLineProjected, BufferDist[Index_Buffer])
        Buffer <- spTransform(BufferProjected, proj4string(RunLine))
        
        CellNumbers <- cellFromPolygon(Raster, Buffer)[[1]]
        
      }
      
      RasterCellPointsTemp <- data.frame(lon=xFromCell(Raster, CellNumbers),
                                         lat=yFromCell(Raster, CellNumbers),
                                         buffer=BufferDist[Index_Buffer]) 
      
      RasterCellPoints <- rbind(RasterCellPoints, 
                                RasterCellPointsTemp[!(paste(RasterCellPointsTemp$lon, RasterCellPointsTemp$lat) %in% paste(RasterCellPoints$lon, RasterCellPoints$lat)), ])
      
      rm(RasterCellPointsTemp)
          
    }
    
    if(nrow(RunStartCoord)==1) {
      RasterCellPoints$dist_start <- round(distHaversine(RasterCellPoints[,c(1,2)], RunStartCoord), 0)
    } else {
      warning(paste0("No 'GettingReady' pause point available for run with uuid '", RunUUID, "'!"),immediate. = T)
      RasterCellPoints$dist_start <- NA
    }
    
    if(nrow(RunEndCoord)==1) {
      RasterCellPoints$dist_end   <- round(distHaversine(RasterCellPoints[,c(1,2)], RunEndCoord), 0)
    } else {
      warning(paste0("No 'PackingUp' pause point available for run with uuid '", RunUUID, "'!"),immediate. = T)
      RasterCellPoints$dist_end   <- NA
    }
  
  }
    
  return(RasterCellPoints)  

}
