#' Extract coordinates of raster cells within polygons
#'
#' Extract coordinates of raster cells within polygons.
#' @param Polygon Object of class SpatialPolygons with a single polygon.
#' @param Raster Object of class 'RasterLayer' (package raster)
#' @param BufferDist Size of buffer around polygon in metres. Can be single value or array of values increasing values. Default value is 0.
#' @return Dataframe with coordinates, buffer distance and distance from starting and end point of run.
#'
#' @export

extractRasterCoordWithinPolygon <- function(Polygon, Raster, Local_proj4text, BufferDist=0) {
  
  ## Check input parameters
  if(class(Raster)[1]!="RasterLayer") {stop("Parameter 'Raster' needs to be of class 'RasterLayer'!!")}
  
  ## Preparing output dataframe
  RasterCellPoints <- data.frame(lon=numeric(0),
                                 lat=numeric(0),
                                 buffer=numeric(0))
  
  ## Check
  if (class(Polygon)[1] != "SpatialPolygons") {
    stop("Parameter 'Polygon' needs to be of class 'SpatialPolygons'!!")
  } else if (length(Polygon) != 1) {
    stop("Parameter 'Polygon' can only contain a single polygon!!")
  } else {
  
  ## Process buffers
    for (Index_Buffer in 1:length(BufferDist)) {
      
      if (BufferDist[Index_Buffer]==0) {
        
        CellNumbers <- cellFromPolygon(Raster, Polygon)[[1]]
        
      } else {
        
        PolygonProjected <- spTransform(Polygon, Local_proj4text)
        BufferProjected  <- buffer(PolygonProjected, BufferDist[Index_Buffer])
        
        if(!is.null(BufferProjected)) {
          Buffer <- spTransform(BufferProjected, proj4string(Polygon))
          CellNumbers <- cellFromPolygon(Raster, Buffer)[[1]]
        } else {
          print(paste0("No raster cells available with buffer value of ", BufferDist[Index_Buffer], "m."))
          CellNumbers <- NA
        }
        
      }
      
      if(!is.null(CellNumbers)) {
        
        if(!is.na(CellNumbers[1])) {
          RasterCellPointsTemp <- data.frame(lon=xFromCell(Raster, CellNumbers),
                                             lat=yFromCell(Raster, CellNumbers),
                                             buffer=BufferDist[Index_Buffer]) 
          
          RasterCellPoints <- rbind(RasterCellPoints, 
                                    RasterCellPointsTemp[!(paste(RasterCellPointsTemp$lon, RasterCellPointsTemp$lat) %in% paste(RasterCellPoints$lon, RasterCellPoints$lat)), ])
          
          rm(RasterCellPointsTemp)
        }
        
      }
            
    }

  }
    
  return(RasterCellPoints)  

}
