#' Extract values from raster along ski runs
#'
#' Extract values from raster within a given buffer along GPS ski runs and stores them in PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param User Name of user. Required for local root folder for location of raster.
#' @param RasterNames Name of raster(s) to be processed. Can be single string or array of strings.
#' @param Method Method for extraction of raster values. Must be 'simple' or 'bilinear'. See help for raster::extract for more detail.
#' @param Buffer size of buffer around GPS ski run in metres. Default value is 20 m. See help for raster::extract for more detail.
#' @param DecimalDigits Decimal digits for extracted values. Default value is 0 (zero). Can be single number of array of numbers. If an array, it needs to be os same size as RasterNames.
#' @param OnlyQCRuns Switch to specify whether to process only runs that are quality controlled (default) or all runs. Default value is TRUE
#' @param Overwrite Switch to specify whether to erase all records from the table before (re-) processing the runs. Default value id FALSE.
#' @param Test Only the first 10 runs are processed in Test mode. Default value is FALSE.
#' @param TimeDiffToDBServer Time difference to database server. Default value is 7 hrs (PST -> GMT)
#' @param UserConfirm Flag whether the user needs to actively confirm deletion. True by default.
#' @param DBType Parameter for specifying which database the extracted values should be written to. Values can be 'Main' (default), 'NodeJS' or 'Old'.
#' @param Verbose Switch for printing of query. Default value is FALSE.
#' @param SuppressPostgreSQLWarnings Switch for turning warnings off. Default value is TRUE.
#'
#' @export

extractRasterValuesAlongGPSRuns <- function(Operation, User, RasterNames, Method="simple", Buffer=20, DecimalDigits=0, OnlyQCRuns=T, Overwrite=F, Test=F, TimeDiffToDBServer=7, UserConfirm=T, DBType="Main", Verbose=F, SuppressPostgreSQLWarnings=T) {

  stop("THIS FUNCTION IS DEPRECATED!!!")

  ## Validate input parameters
  ## =========================

  ## Lenth of RasterNames and DecimalDigits arrays
  if (length(RasterNames)>1) {
    if (length(DecimalDigits)==1) {
      ## ok
    } else if (length(DecimalDigits)==length(RasterNames)) {
      ## ok
    } else {
      stop("The parameter DecimalDigits needs to be either a single value or an array of the same length as RasterNames!")
    }
  }

  ## Methods
  if (Method!="simple" & Method!="bilinear") {
    stop("Value of Method parameter must be 'simple' or 'bilinear'! See help for raster::extract for more detail.")
  }

  ## Get raster background information from DB
  RasterInfoAll <- listRasterRefAndArrayTbl(Operation)

  ## Loop for rasternames
  ## ====================
  for (Index_Raster in 1:length(RasterNames)) {

    RasterInfo <- subset(RasterInfoAll, name==RasterNames[Index_Raster])

    if (nrow(RasterInfo)==1) {

      ## Get basic information and raster
      ## *******************************
      RasterName <- RasterInfo$name
      DBTbl <- paste0("gpsruns_array_", RasterInfo$gis_array_name_ext)

      DecimalDigitsForRaster <- ifelse(length(DecimalDigits)>1, DecimalDigits[Index_Raster], DecimalDigits)

      Raster <- getRaster(Operation, User, RasterName)


      ## Delete existing records if Overwirte==T
      ## ***************************************
      if (Overwrite) {

        if (UserConfirm==F) {
          UserResponse <- "Yes"
        } else {
          UserResponse <- readline(paste0("Type 'Yes' to confirm deletion of ", RasterInfo$NumRecords, " existing records:"))
        }

        ## Deletion in response to user confirmation
        if (UserResponse=="Yes") {
          Query <- paste0("DELETE FROM gis.", DBTbl)
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        }
      }

      ## Retrieve UUIDs of runs to be processed
      ## **************************************
      if (OnlyQCRuns) {
        Query <- paste0("SELECT gps.qc_runs.uuid FROM gps.qc_runs LEFT JOIN gis.", DBTbl, " ON gis.", DBTbl, ".gpsruns_uuid = gps.qc_runs.uuid WHERE gis.", DBTbl, ".gpsruns_uuid is NULL")
      } else {
        Query <- paste0("SELECT gps.runs.uuid FROM gps.runs LEFT JOIN gis.", DBTbl, " ON gis.", DBTbl, ".gpsruns_uuid = gps.runs.uuid WHERE gis.", DBTbl, ".gpsruns_uuid is NULL")
      }
      GPSRuns <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

      NumRuns <- nrow(GPSRuns)
      print(paste0("Number of runs to be processed for ", RasterName, ": ", NumRuns))
      RunIndexMax <- ifelse(Test, min(10, NumRuns), NumRuns)

      ## Processing GPS runs
      ## ===================
      if (RunIndexMax == 0) {

        print(paste0("No GPS runs to be processed for ", RasterName, "!"))

      } else {

        ## Establish DB Connection: Faster to do it once outside of the loop!
        DBCon <- connectToGPSDB(Operation, ReadOnly=F, DBType=DBType)

        ## Loop for GPS runs
        for (RunIndex in 1:RunIndexMax) {

          # User feedback
          print(paste0(Sys.time(), " ", RasterName, " - Processing Run ", RunIndex, " of ", RunIndexMax, " ..."))

          # Create output DF
          OutputDF <- data.frame(gpsruns_uuid = GPSRuns$uuid[RunIndex], buffer = NA, valuearray = NA, dtcre=Sys.time()+TimeDiffToDBServer*60*60)

          # Get spatial line from run
          Run_SL <- getSpatialObjectsFromUUID(Operation, "gps", "runs", OutputDF$gpsruns_uuid[1], DBType=DBType)

          # Extract raster values and round
          OutputDF$valuearray[1] <- list(round(unlist(extract (Raster, Run_SL, method=Method, buffer=Buffer)), DecimalDigitsForRaster))
          OutputDF$buffer[1] <- Buffer

          # Writing to database
          Query <- paste0("INSERT INTO gis.", DBTbl, " (gpsruns_uuid, buffer, valuearray, dtcre) ",
                            "VALUES (", convertStringToSQL(OutputDF$gpsruns_uuid[1]), ", ",
                            convertNumericToSQL(OutputDF$buffer[1]), ", ",
                            convertVectorToSQL(unlist(OutputDF$valuearray[1])), ", ",
                            convertDateTimeToSQL(OutputDF$dtcre[1]), ")")
          res <- dbSendQuery(DBCon, Query)
          dbClearResult(res)
          rm(Query, res)

        } ## end of run loop

        ## Disconnect from DB
        dbDisconnect(DBCon)
        rm(DBCon)

      } ## end of if condition: RunIndexMax > 0

    } else {

      warning(paste0("No raster reference available for '", RasterNames[Index_Raster], "'!"))

    }

  }

}
