#' Filters raster value dataframe by daily quantile range.
#'
#' Filters raster value dataframe by daily quantile range. The raster value dataframe needs to be created with the getRasterValueDfFromRunUUID function.
#' @param RasterValueDf The raster value dataframe (class RasterValueDf) created with the getRasterValueDfFromRunUUID function.
#' @param ColName Name of the column over which the filter should be applied.
#' @param QuantileBottom Value between 0 and 1 to specify the bottom quantile boundary for the extracted data. A value of 0.8 extracts the records with the top 20 percent incline values for each day.
#' @param QuantileTop Value between 0 and 1 to specify the top quantile boundary for the extracted data. Default value is 1.
#' 
#' @return List with two data frames: 1) DataFrame: filtered dataframe; 2) DayStats: Daily processing statistics

#' @seealso \code{\link{getRasterValueDfFromRunUUID}}
#' 
#' @export

filterRasterValueDfByDailyQuantiles <- function (RasterValueDf, ColName, QuantileBottom, QuantileTop=1) {
  
  ## Check for required type
  if(!("RasterValueDf" %in% class(RasterValueDf))) {
    stop("RasterValueDf not of class 'RasterValueDf'! Create dataframe with function 'getRasterValueDfFromRunUUID'.")
  }

  ## Check required column
  DFNames <- names(RasterValueDf)
  if (!(ColName %in% names(RasterValueDf))) {
    stop(paste0("Input dataframe is missing required '", ColName, "' column!"))
  }
  
  ## Check if top boundary is larger than bottom
  if (!(QuantileBottom<QuantileTop)) {
    stop("The value of QuantileBottom must be smaller than the value of QuantileTop!")
  }

  ## Initialize list for day dataframes
  List_DayDF <- list(NA)

  ## Unique list of dates
  Dates <- sort(unique(RasterValueDf$date))

  ## Loop through days
  for (Index_Day in 1:length(Dates)) {

    DayDF <- RasterValueDf[RasterValueDf$date==Dates[Index_Day],]
    NumObsOrig <- nrow(DayDF)

    QuantileBottomValue <- quantile(DayDF[,ColName], QuantileBottom, names=F)
    QuantileTopValue    <- quantile(DayDF[,ColName], QuantileTop, names=F)

    DayDF <- DayDF[((DayDF[,ColName]>=QuantileBottomValue) & (DayDF[,ColName]<=QuantileTopValue)),]
    NumObsFilt <- nrow(DayDF)

    List_DayDF[[Index_Day]] <- DayDF

    if(Index_Day==1) {

      DayStats <- data.frame(Date=Dates[Index_Day],
                             NumObsOrig=NumObsOrig,
                             QuantBottom=QuantileBottom,
                             QuantBottomValue=QuantileBottomValue,
                             QuantTop=QuantileTop,
                             QuantTopValue=QuantileTopValue,
                             NumObsFilt=NumObsFilt)

    } else {

      DayStats <- rbind(DayStats, data.frame(Date=Dates[Index_Day],
                                             NumObsOrig=NumObsOrig,
                                             QuantBottom=QuantileBottom,
                                             QuantBottomValue=QuantileBottomValue,
                                             QuantTop=QuantileTop,
                                             QuantTopValue=QuantileTopValue,
                                             NumObsFilt=NumObsFilt))

    }

  }

  ## Combining the all the RunDF into single DF via data.table::rbindlist
  invisible(lapply(List_DayDF, setattr, name = "class", value = c("data.table",  "data.frame")))
  OutputDF <- rbindlist(List_DayDF)
  setattr(OutputDF, "class", c("data.frame", "RasterValueDf"))

  ## Output
  Output <- list(RasterValueDf=OutputDF, DayStats=DayStats)
  return(Output)

}