#' Formats Run character assessment dataframe for analysis
#'
#' Formats Run character assessment dataframe for analysis
#' @param Operation Name of operation
#' @param AssessmentDF Run character assessment dataframe
#' @param Progress Switch for printing progess (default = TRUE)
#' @param Verbose Switch for printing queries (default = FALSE)
#' @return Formatted dataframe.
#' @export
#' 
#' @examples 
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "NEH"
#' RunCharAssess <- getRecordsFromQuery(Operation, "SELECT * FROM runchar.assessments")
#'
#' RunCharAssessFormatted <- formatRunCharAssessmentDF(Operation, RunCharAssess)


formatRunCharAssessmentDF <- function(Operation, AssessmentDF, Progress=TRUE, Verbose=FALSE) {

  
  ## FUNCTION FOR FORMATING INDIVIDUAL RUN CHARACTER QUESTION ASSESSMENTS
  ## ********************************************************************
  formatRunCharAssessmentQuestions <- function(Operation, AssessmentDF, QuestionCode, Progress=FALSE, Verbose=FALSE) {
    
    ## Retrieve question record
    QuestionDetail <- getRecordsFromQuery(Operation, paste0("Select * from runchar.questions where code='", QuestionCode, "'"), Verbose = Verbose)
    
    if (nrow(QuestionDetail)!=1) {
    
      stop(paste0("Incorrect question code: ", QuestionCode))
    
    } else {
    
      ## Extract question details
      QuestionRespType <- toupper(QuestionDetail$responsetype)
      QuestionLineLevel <- convertArrayFromSQL(QuestionDetail$linelevels, Numeric = F)
      QuestionElevLevel <- convertArrayFromSQL(QuestionDetail$elevlevels, Numeric = F)

      ## Check whether question is included
      if (is.na(QuestionLineLevel[1])) {
        ColumnIncluded <- ifelse(QuestionCode %in% names(AssessmentDF), T, F)
      } else if (is.na(QuestionElevLevel[1])) {
        ColumnIncluded <- ifelse(paste0(QuestionCode, "_", QuestionLineLevel[1]) %in% names(AssessmentDF), T, F)
      } else {
        ColumnIncluded <- ifelse(paste0(QuestionCode, "_", QuestionLineLevel[1], "_", QuestionElevLevel[1]) %in% names(AssessmentDF), T, F)
      }
      
      ## If column is not present
      if(!ColumnIncluded) {
        
        ## Progress report
        if(Progress) {print(paste("Skipping", QuestionCode, "because not included in dataframe ..."))}
        
      } else {
      
        ## Progress report
        if(Progress) {print(paste("Formatting", QuestionCode, "..."))}
        
        ## Extract question enumeration details
        if(grepl("mitig_explo", QuestionCode)) {
          EnumTblName <- "enum_mitig_explo"
        } else {
          EnumTblName <- paste0("enum_", QuestionCode)
        }
        
        QuestionResponses <- getRecordsFromUUID(Operation, "runchar", EnumTblName, Verbose=Verbose)
        QuestionEnum <- QuestionResponses$code[order(QuestionResponses$order)]
      
        ## Process Single choice nominal
        if (QuestionRespType=="SC") {
        
          if (is.na(QuestionLineLevel[1])) {    
            AssessmentDF[QuestionCode] <- factor(AssessmentDF[[QuestionCode]], levels=QuestionEnum)
          } else if (is.na(QuestionElevLevel[1])) {
            for (index_linelevel in 1:length(QuestionLineLevel)) {
              AssessmentDF[paste0(QuestionCode, "_", QuestionLineLevel[index_linelevel])] <- factor(AssessmentDF[[paste0(QuestionCode, "_", QuestionLineLevel[index_linelevel])]], levels=QuestionEnum)
            }
          } else {
            for (index_linelevel in 1:length(QuestionLineLevel)) {
              for (index_elevlevel in 1:length(QuestionElevLevel)) {
                AssessmentDF[paste0(QuestionCode, "_", QuestionLineLevel[index_linelevel], "_", QuestionElevLevel[index_elevlevel])] <- factor(AssessmentDF[[paste0(QuestionCode, "_", QuestionLineLevel[index_linelevel], "_", QuestionElevLevel[index_elevlevel])]], levels=QuestionEnum)
              }
            }
          }
          
        ## Process Single choice ordinal
        } else if (QuestionRespType=="SCO") {
      
          if (is.na(QuestionLineLevel[1])) {
            AssessmentDF[QuestionCode] <- ordered(AssessmentDF[[QuestionCode]], levels=QuestionEnum)
          } else if (is.na(QuestionElevLevel[1])) {
            for (index_linelevel in 1:length(QuestionLineLevel)) {
              AssessmentDF[paste0(QuestionCode, "_", QuestionLineLevel[index_linelevel])] <- ordered(AssessmentDF[[paste0(QuestionCode, "_", QuestionLineLevel[index_linelevel])]], levels=QuestionEnum)
            }
          } else {
            for (index_linelevel in 1:length(QuestionLineLevel)) {
              for (index_elevlevel in 1:length(QuestionElevLevel)) {
                AssessmentDF[paste0(QuestionCode, "_", QuestionLineLevel[index_linelevel], "_", QuestionElevLevel[index_elevlevel])] <- ordered(AssessmentDF[[paste0(QuestionCode, "_", QuestionLineLevel[index_linelevel], "_", QuestionElevLevel[index_elevlevel])]], levels=QuestionEnum)
              }
            }
          }
          
        ## Process Multiple choice  
        } else if (QuestionRespType=="MC") {
          
          if (is.na(QuestionLineLevel[1])) {    
    
            ColNameSrc  <- QuestionCode
            
            for (index_row in 1:nrow(AssessmentDF)) {
              
              ## Update column content
              if (!is.na(AssessmentDF[index_row, ColNameSrc])) {
                AssessmentDF[index_row, ColNameSrc] <- paste(convertArrayFromSQL(AssessmentDF[index_row, ColNameSrc], Numeric = F), collapse = ",")
              }
              
              ## Create an fill boolean columns
              for (index_enum in 1:length(QuestionEnum)) {
                ColNameDest <- paste0(ColNameSrc, "_", QuestionEnum[index_enum])
                AssessmentDF[index_row, ColNameDest] <- ifelse(QuestionEnum[index_enum] %in% unlist(strsplit(AssessmentDF[index_row, ColNameSrc], split=",")), T, F)
              }
              
            }
            
          } else if (is.na(QuestionElevLevel[1])) {
    
            for (index_linelevel in 1:length(QuestionLineLevel)) {
    
              ColNameSrc  <- paste0(QuestionCode, "_", QuestionLineLevel[index_linelevel])
              
              for (index_row in 1:nrow(AssessmentDF)) {
                
                ## Update column content
                if (!is.na(AssessmentDF[index_row, ColNameSrc])) {
                  AssessmentDF[index_row, ColNameSrc] <- paste(convertArrayFromSQL(AssessmentDF[index_row, ColNameSrc], Numeric = F), collapse = ",")
                }
                
                ## Create an fill boolean columns
                for (index_enum in 1:length(QuestionEnum)) {
                  ColNameDest <- paste0(ColNameSrc, "_", QuestionEnum[index_enum])
                  AssessmentDF[index_row, ColNameDest] <- ifelse(QuestionEnum[index_enum] %in% unlist(strsplit(AssessmentDF[index_row, ColNameSrc], split=",")), T, F)
                }
                
              }
    
            }
            
          } else {
          
            for (index_linelevel in 1:length(QuestionLineLevel)) {
              
              for (index_elevlevel in 1:length(QuestionElevLevel)) {
              
                ColNameSrc  <- paste0(QuestionCode, "_", QuestionLineLevel[index_linelevel], "_", QuestionElevLevel[index_elevlevel])

                for (index_row in 1:nrow(AssessmentDF)) {
                  
                  ## Update column content
                  if (!is.na(AssessmentDF[index_row, ColNameSrc])) {
                    AssessmentDF[index_row, ColNameSrc] <- paste(convertArrayFromSQL(AssessmentDF[index_row, ColNameSrc], Numeric = F), collapse = ",")
                  }
                  
                  ## Create an fill boolean columns
                  for (index_enum in 1:length(QuestionEnum)) {
                    ColNameDest <- paste0(ColNameSrc, "_", QuestionEnum[index_enum])
                    AssessmentDF[index_row, ColNameDest] <- ifelse(QuestionEnum[index_enum] %in% unlist(strsplit(AssessmentDF[index_row, ColNameSrc], split=",")), T, F)
                  }
                  
                }
                
              }
              
            }
            
          }
        
        ## Incorrect response type  
        } else {
          
          stop(paste0("Unknown responsetype code for question ", QuestionCode, ": ", QuestionRespType))
          
        }
        
      }
      
      ## Return DF
      return(AssessmentDF)
    
    }
    
  }


  ## FORMATTING OF TABLE
  ## *******************
  
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "familiarity", Verbose=Verbose, Progress=Progress)
  
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "acc_gen_feel", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "acc_best_land", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "acc_land_feat", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "acc_land_feat_winddir", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "acc_pickup_opt", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "acc_pickup_feat", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "acc_incident", Verbose=Verbose, Progress=Progress)
  
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "ski_terrain", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "ski_spec", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "ski_diff", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "ski_chall", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "ski_overall", Verbose=Verbose, Progress=Progress)
  
  ## Number of tracks does not require formatting
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "usab_oprole", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "usab_conflict", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "usab_conflict_type", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "usab_suncrust", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "usab_reset", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "usab_wildlife", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "usab_wildlife_type", Verbose=Verbose, Progress=Progress)
  
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "hzd_steep", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "hzd_expo", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "hzd_terrainhzd", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "hzd_snowcond", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "hzd_otherhzd", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "hzd_friendly", Verbose=Verbose, Progress=Progress)
  
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "mitig_guideabil", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "mitig_explo_run", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "mitig_explo_oh", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "mitig_explo_pickup", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "mitig_skiertraf", Verbose=Verbose, Progress=Progress)
  AssessmentDF <- formatRunCharAssessmentQuestions(Operation, AssessmentDF, "mitig_incident", Verbose=Verbose, Progress=Progress)
  
  ## Additional comments does not require formatting
  
  
  ## RETURNING DATAFRAME
  ## *******************
  return(AssessmentDF)
  
}