#' Gets DEM for Operation
#'
#' Gets low-res (7.5 arc sec) or high-res (20m) DEM for operation either from local path or working directory (default).
#' If file does not exist locally, it is downloaded from http://data.avalancheresearch.ca.
#' 
#' @param Oper Abbreviation of operation
#' @param higres Switch for high-res DEM. Default is FALSE
#' @param localpath Local path for source of DEM tif file. Default is NA, which uses working directory.
#' @param baseurl URL for remote source of DEM files. Default is http://data.avalancheresearch.ca/.
#'
#' @examples
#' require(SarpGPSTools)
#'
#' Oper <- "CMHGL"
#' 
#' ## Low resolution
#' DEM <- getDEM(Oper)
#' plot(DEM)
#' 
#' ## High resolution stored in working directory
#' DEM <- getDEM(Oper, highres=T)
#' plot(DEM)
#' 
#' ## High resolution stored in local directory
#' DEM <- getDEM(Oper, highres=T, localpath="C:/Temp/")
#' plot(DEM)
#'
#' @export

getDEM <- function(Oper, highres=F, localpath=NA, baseurl="http://data.avalancheresearch.ca/") {
  
  ## Retrieve (if necessary) and load highres DEM
  if (highres) {
    
    filename <- paste0("cdsm_elevation_", Oper, ".tif")
    
    ## Create full source path
    if(substr(baseurl, nchar(baseurl), nchar(baseurl))!="/") {baseurl <- paste0(baseurl,"/")}
    source <- paste0(baseurl, filename)
    
    ## Retrived from local path and downloaded if necessary
    if (!is.na(localpath)) {
      
      if(substr(localpath, nchar(localpath), nchar(localpath))!="/") {localpath <- paste0(localpath,"/")}
      target <- paste0(localpath, filename)
      if(!file.exists(target)) {
        warning(paste0("The DEM file '", filename, "' is downloaded to ", localpath, "."), immediate. = T)
        download.file(source, target, mode="wb")
      }
    
    ## Retrived from working directory and downloaded if necessary  
    } else {
      
      target <- filename
      if(!file.exists(target)) {
        warning(paste0("The DEM file '", filename, "' is downloaded to your working directory."), immediate. = T)
        download.file(source, target, mode="wb")
      }
      
    }
    
    DEM <- raster(target)
    
  ## Load lowres DEM directly into R without downloading file  
  } else {
    load(url(paste0(baseurl, "gmted_mea075_", Oper, ".rda")))
  }
  
  ##Return
  return(DEM)
  
}