#' Extracts distance skied in three NEH terrain classes from GPS runs. 
#'
#' Extracts distance skied in three NEH terrain classes from GPS runs. 
#' @param Operation Name of operation.
#' @param GPSRunsUUIDs Array of uuids of gps runs
#' @param ShowCounter Numeric value that specifies whether counter for queries should be shown (>0) and at what interval
#' @param Verbose Switch for printing of query
#' @return Dataframe with distances in metre.
#' 
#' @examples 
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Oper <- "NEH"
#' RunName <- c("White Rabbit", "Jabber Walky")
#' 
#' ## Get  location catalog
#' LocCat <- getLocCatlog(Oper)
#' 
#' ## Get all GPS runs from these two runs
#' GPSRuns_UUID <- getGPSRunsUUIDsFromLoccatSkiRunLineUUIDsNames(Oper, Names=RunName)
#' 
#' ## Calculates skied distance in difference terrain classes
#' DistInTerrainClasses <- getDistanceInTerrainClassesFromGPSRunsUUID(Oper, GPSRuns_UUID)
#' 
#' ## Base map
#' plot(LocCat, ZoomArea=RunName, ShowSkiRunSegments=T, main=paste0("Base map for ", paste(RunName, collapse = ", ")))
#' 
#' ## Runs that did not include any black terrain
#' GPSRunsNoBlack_UUID <- as.character(DistInTerrainClasses$gpsruns_uuid[DistInTerrainClasses$black==0])
#' GPSRunsNoBlack <- getRunsObjectFromUUIDs(Oper, GPSRunsNoBlack_UUID, WithObsPoints = F, WithPausePoints = F)
#' 
#' plot(LocCat, ZoomArea=RunName, ShowSkiRunSegments=T, main="Only runs outside of black terrain")
#' plot(GPSRunsNoBlack$Runs, col="blue", add=T)
#' 
#' ## Runs that did include black terrain
#' GPSRunsBlack_UUID <- as.character(DistInTerrainClasses$gpsruns_uuid[DistInTerrainClasses$black>0])
#' GPSRunsBlack <- getRunsObjectFromUUIDs(Oper, GPSRunsBlack_UUID, WithObsPoints = F, WithPausePoints = F)
#' 
#' plot(LocCat, ZoomArea=RunName, ShowSkiRunSegments=T, main="Runs that include black terrain")
#' plot(GPSRunsBlack$Runs, col="black", add=T)
#' 
#' ## Combinded plot
#' plot(LocCat, ZoomArea=RunName, ShowSkiRunSegments=T, main="Both types of runs")
#' plot(GPSRunsBlack$Runs, col="black", add=T)
#' plot(GPSRunsNoBlack$Runs, col="blue", add=T)
#' 
#' @export

getDistanceInTerrainClassesFromGPSRunsUUID <- function(Operation, GPSRuns_UUID, ShowCounter=1, Verbose=F) {

  ## Create output dataframe
  Output <- data.frame(gpsruns_uuid=as.character(GPSRuns_UUID),
                       dist=0,
                       green=0,
                       blue=0,
                       black=0)
  
  ## Loop for processing
  for (index in 1:length(GPSRuns_UUID)) {
    
    ## GPS run uuid
    UUID <- GPSRuns_UUID[index]
    
    ## Feedback
    if (ShowCounter>0) {
      if (index==nrow(Output)) {
        print(paste0("Processing GPS run ", index, " of ", nrow(Output), "."))
      } else if ((index==1) | (index %% ShowCounter == 0)) {
        print(paste0("Processing GPS run ", index, " of ", nrow(Output), " ..."))
      }
    }
    
    ## Check whether there are ski run segements for the ski run associated with this GPS run
    Query <- paste0("SELECT loccat.skirun_segments.uuid ", 
                    "FROM gps.runs ",
                    "INNER JOIN loccat.skirun_segments ON gps.runs.loccatskiruns_uuid = loccat.skirun_segments.loccatskiruns_uuid ",
                    "WHERE gps.runs.uuid = '", UUID, "'")
    
    Check <- getRecordsFromQuery(Operation=Operation, Query=Query, Verbose=Verbose)
    
    if (nrow(Check)>0) {

      ## Determine distances within run segments    
      Query <- paste0("SELECT gpsruns_uuid, MIN(gpsruns_length_m) AS gpsruns_length_m, terrain_rating, SUM(length_within_m) AS length_within_m ",
                      "FROM (SELECT gpsruns_uuid, gpsruns_length_m, loccatskirunsegments_uuid, terrain_rating, round(ST_length(ST_Transform(TblIntersect.geom, ", 
                                   "(SELECT srdi FROM misc.local_srdi LIMIT 1)))) AS length_within_m ",
                            "FROM (", 
                                   "SELECT TblRunSegmentsSelect.uuid AS loccatskirunsegments_uuid, ",
                                          "gps.runs.uuid AS gpsruns_uuid, ",
                                          "gps.runs.length_m AS gpsruns_length_m, ",
                                          "TblRunSegmentsSelect.terrain_rating AS terrain_rating, ", 
                                          "ST_Intersection (gps.runs.geom, TblRunSegmentsSelect.geom) AS geom ",
                                    "FROM gps.runs, ",
                                         "(SELECT loccat.skirun_segments.uuid, loccat.skirun_segments.terrain_rating, loccat.skirun_segments.geom, loccat.skirun_segments.flag_delete FROM loccat.skirun_segments INNER JOIN gps.runs ON gps.runs.loccatskiruns_uuid = loccat.skirun_segments.loccatskiruns_uuid ",
                                          "WHERE gps.runs.uuid = '", UUID, "') AS TblRunSegmentsSelect ",
                                    "WHERE gps.runs.uuid = '", UUID, "' AND TblRunSegmentsSelect.flag_delete = 'No')",
                                    "AS TblIntersect ",
                            "WHERE ST_AsText (TblIntersect.geom) != 'GEOMETRYCOLLECTION EMPTY' ",
                            "ORDER BY length_within_m DESC) ",
                            "AS TblTerrainClass ",
                       "GROUP BY gpsruns_uuid, terrain_rating")
      
      TerrainClassDist <- getRecordsFromQuery(Operation=Operation, Query=Query, Verbose=Verbose)
      
      ## Transfer results into output table
      if(nrow(TerrainClassDist)>0) {
        Output$dist[Output$gpsruns_uuid==UUID] <- TerrainClassDist$gpsruns_length_m[1]
        if (nrow(TerrainClassDist[TerrainClassDist$terrain_rating=="green",])>0) {
          Output$green[Output$gpsruns_uuid==UUID] <- TerrainClassDist$length_within_m[TerrainClassDist$terrain_rating=="green"]
        }
        if (nrow(TerrainClassDist[TerrainClassDist$terrain_rating=="blue",])>0) {
          Output$blue[Output$gpsruns_uuid==UUID] <- TerrainClassDist$length_within_m[TerrainClassDist$terrain_rating=="blue"]
        }
        if (nrow(TerrainClassDist[TerrainClassDist$terrain_rating=="black",])>0) {
          Output$black[Output$gpsruns_uuid==UUID] <- TerrainClassDist$length_within_m[TerrainClassDist$terrain_rating=="black"]
        }
      }

    } else {
      
      ## Fill distance
      Query <- paste0("SELECT length_m FROM gps.runs WHERE uuid='", UUID, "'")
      
      RunDist <- getRecordsFromQuery(Operation=Operation, Query=Query, Verbose=Verbose)
      Output$dist[Output$gpsruns_uuid==UUID] <- RunDist$length_m[1]
      
    }
    
  } ## End of for loop for GPS runs
  
  ## Return output
  return(Output)
  
}