#' Get overview of how many GPS run track with have in skirun polygons. 
#'
#' Get overview of how many GPS run track with have in skirun polygons. 
#' @param Operation Name of operation.
#' @param DateStart Start date of date range.
#' @param DateEnd End date od date range. Default is same as DateStart.
#' @param MinDistInSkirun Minimum distance that run has to be in a polygon in m. Default value is 200 m.
#' @param ShowProgress Switch whether progress messages should be printed on screen. Default value is T.
#' @param Verbose Switch whether queries should be printed on screen. Default value is F.
#' @return Dataframe with list of ski runs with summary statistics about tracks.
#' 
#' @examples 
#'
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Oper <- "CMHGL"
#' 
#' DateStart <- "2016-02-01"
#' DateEnd <- "2016-02-03"
#' 
#' GPSRunSummary <- getGPSRunsInLoccatSkirunSummary(Oper, DateStart, DateEnd)
#' 
#' @export

getGPSRunsInLoccatSkirunSummary <- function(Operation, DateStart, DateEnd=DateStart, MinDistInSkirun=200, ShowProgress=T, Verbose=F) {

  ## Getting data from database
  if(ShowProgress) {print("Retrieving data from database ...")}
  
  Query <- paste0("SELECT gpsruns_uuid, ",
                         "loccatskiruns_uuid, ", 
                         "loccatskiruns_name, ",
                         "gpsruns_date, ",
                         "round(ST_length (ST_Transform (TblIntersect.geom,(SELECT srdi	FROM misc.local_srdi LIMIT 1)))) AS length_within_m ",
                  "FROM (SELECT loccat.skiruns.uuid as loccatskiruns_uuid, ",
                               "loccat.skiruns.name as loccatskiruns_name, ",
                               "gps.runs.uuid as gpsruns_uuid, ",
                               "gps.runs.date_local as gpsruns_date, ",  
                               "ST_Intersection(gps.runs.geom, loccat.skiruns.geom) as geom ",
                        "FROM gps.runs, loccat.skiruns ",
                        "WHERE gps.runs.date_local >='", DateStart, "' and gps.runs.date_local <='", DateEnd, "' ",
                        ") as TblIntersect ",
                  "WHERE ST_AsText (TblIntersect.geom) != 'GEOMETRYCOLLECTION EMPTY'")
  
  Results <- getRecordsFromQuery(Oper, Query, Verbose=Verbose)
  
  ## Eliminating short runs
  Results <- Results[Results$length_within_m>=MinDistInSkirun,]
  
  ## Processing results
  if(ShowProgress) {print("Processing results ...")}
  
  Output <- data.frame(loccatskiruns_uuid = levels(as.factor(Results$loccatskiruns_uuid)),
                       loccatskiruns_name = NA,
                       num_tracks = NA,
                       date_first = NA,
                       date_last = NA,
                       length_min = NA,
                       length_median = NA,
                       length_max = NA)
  
  for (Index in 1:nrow(Output)) {
    
    if(ShowProgress) {print(paste0("Processing run ", Index, " of ", nrow(Output), " ..."))}
    Temp <- Results[Results$loccatskiruns_uuid==Output$loccatskiruns_uuid[Index],]
    
    Output$loccatskiruns_name[Index] <- Temp$loccatskiruns_name[1]
    Output$num_tracks[Index] <- nrow(Temp)
    Output$date_first[Index] <- format(min(Temp$gpsruns_date), "%Y-%m-%d")
    Output$date_last[Index] <- format(max(Temp$gpsruns_date), "%Y-%m-%d")
    Output$length_min[Index] <- min(Temp$length_within_m)
    Output$length_max[Index] <- max(Temp$length_within_m)
    Output$length_median[Index] <- round(median(Temp$length_within_m), 0)
    
  }
                        
  ## Return results
  return(Output)
  
}