#' Gets UUIDs of GPS ski runs that are linked to ski runs or ski lines defined in loccation catalog. 
#'
#' Gets UUIDs of GPS ski runs that are linked to ski runs or ski lines defined in loccation catalog.
#' @param Operation Name of operation.
#' @param Names Names of loccation catalog ski runs or ski lines
#' @param UUIDs UUIDs of loccation catalog ski runs or ski lines
#' @param Verbose Switch for printing of query
#' @return Array of GPS run UUIDs
#' 
#' @examples 
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Oper <- "NEH"
#' RunName <- c("White Rabbit", "Jabber Walky")
#' 
#' ## Get all GPS runs from these two runs
#' GPSRuns_UUID <- getGPSRunsUUIDsFromLoccatSkiRunLineUUIDsNames(Oper, Names=RunName)
#' 
#' @export

getGPSRunsUUIDsFromLoccatSkiRunLineUUIDsNames <- function(Operation, Names=NA, UUIDs=NA) {
  
  ## Check
  if(is.na(Names[1]) & is.na(UUIDs[1])) {
    stop("Either names or UUIDs need to be provided!")
  }
  
  if(!is.na(Names[1]) & !is.na(UUIDs[1])) {
    stop("Only either names or UUIDs can be provided!")
  }
  
  ## Conversion of names to UUIDs
  if (is.na(UUIDs[1])) {
    UUIDs <- character(0)
    UUIDs <- c(UUIDs, getUUIDsFromNames(Operation, "loccat", "skiruns", Names=Names, SuppressWarnings=T))
    UUIDs <- c(UUIDs, getUUIDsFromNames(Operation, "loccat", "skilines", Names=Names, SuppressWarnings=T))
  }
  
  ## Find relevant GPS runs
  GPSRunsUUIDs <- character(0)
  
  Query <- paste0("Select uuid from gps.runs where loccatskiruns_uuid= ANY(ARRAY['", paste(UUIDs, collapse = "', '"), "'])")
  Results <- getRecordsFromQuery(Operation, Query)
  if(nrow(Results)>0) {GPSRunsUUIDs <- c(GPSRunsUUIDs, Results$uuid)}
  
  Query <- paste0("Select uuid from gps.runs where loccatskilines_uuid= ANY(ARRAY['", paste(UUIDs, collapse = "', '"), "'])")
  Results <- getRecordsFromQuery(Operation, Query)
  if(nrow(Results)>0) {GPSRunsUUIDs <- c(GPSRunsUUIDs, Results$uuid)}
  
  ## Return
  if (length(GPSRunsUUIDs)>0) {
    return(GPSRunsUUIDs)
  } else {
    return(NA)
    warning("No GPS runs for for these locations! Check whether spelling of name is correct!", immediate. = T)
  }

}