#' Get InfoEx hazard assessments from PostgreSQL/PostGIS GPS database.
#'
#' Get InfoEx hazard assessments from PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS
#' @param DateStart Start date in YYYY-MM-DD format. Default value is NA. 
#' @param DateEnd End date in YYYY-MM-DD format. Default value is DateStart.
#' @param Time For specifing time of day of hazard assessment. Values can be 'All' (default), 'Am' or 'PM'.
#' @param LocationName Single or array of location names to be associated with hazard assessments (default value is NA)

#' @export

getInfoExHzdAssessObject <- function(Operation, DBType="Main", DateStart=NA, DateEnd=DateStart, Time='All', LocationName=NA, Verbose=F, SuppressPostgreSQLWarnings=T) {
  
  Time <- toupper(Time)
  
  ## Base query
  if (Time=="ALL") {
    Query <- paste("SELECT * from infoex.hzdassess where date_assess >= '", DateStart, "' and date_assess <= '", DateEnd, "'")
  } else if (Time=="AM") {
    Query <- paste("SELECT * from infoex.hzdassess where date_assess >= '", DateStart, "' and date_assess <= '", DateEnd, "' and extract(hour from infoex.hzdassess.datetime_assess) < 12")
  } else if (Time=="PM") {
    Query <- paste("SELECT * from infoex.hzdassess where date_assess >= '", DateStart, "' and date_assess <= '", DateEnd, "' and extract(hour from infoex.hzdassess.datetime_assess) >= 12")
  } else {
    stop("Value for Time parameter not supported!")
  }

  HzdAssess <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
 
  ## Location
  Loc <- getRecordsFromUUID(Operation, "infoex", "hzdassess_loc_x", DBType=DBType, 
                            ForeignUUID=HzdAssess$uuid, ForeignUUIDCol="infoexhzdassess_uuid")
  
  ## Filtering out location names
  if (!is.na(LocationName)) {
    
    Loccat <- getRecordsFromUUID(Operation, "loccat", "all_infoex_locations", DBType=DBType, 
                                 UUID=Loc$loccat_uuid)
    
    Loccat <- Loccat[Loccat$name %in% LocationName]
    Loc <- Loc[Loc$loccat_uuid %in% Loccat$uuid]
    HzdAssess <- HzdAssess[HzdAssess$uuid %in% Loc$infoexhzdassess_uuid]
  
  }
  
  ## Avalanche Problems
  AvProb <- getRecordsFromUUID(Operation, "infoex", "hzdassess_avprob", DBType=DBType, 
                               ForeignUUID=HzdAssess$uuid, ForeignUUIDCol="infoexhzdassess_uuid")
  
  
  ## Output
  Output <- list(HzdAssess=HzdAssess,
                 Loc=Loc,
                 AvProb=AvProb)
  
  class(Output) <- append(class(Output), "HzdAssessObject")
  
  return(Output)

}