#' Get InfoEx mindset from PostgreSQL/PostGIS GPS database.
#'
#' Get InfoEx mindset from PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS
#' @param DateStart Start date in YYYY-MM-DD format. Default value is NA. 
#' @param DateEnd End date in YYYY-MM-DD format. Default value is DateStart.
#' @param Time For specifing time of day of mindset. Values can be 'All' (default), 'Am' or 'PM'.
#' @param LocationName Single or array of location names to be associated with mindset (default value is NA)
#' 
#' @examples 
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Oper <- "CMHGL"
#' 
#' DateStart_Data <- "2015-12-04"
#' DateEnd_Data   <- "2016-04-01"
#' 
#' Mindset <- getInfoExMindsetObject(Oper, DateStart=DateStart_Data, DateEnd=DateEnd_Data, Time="AM")
#' 
#' @export

getInfoExMindsetObject <- function(Operation, DBType="Main", DateStart=NA, DateEnd=DateStart, Time='All', LocationName=NA, Verbose=F, SuppressPostgreSQLWarnings=T) {
  
  Time <- toupper(Time)
  
  ## Base query
  if (Time=="ALL") {
    Query <- paste("SELECT * from infoex.mindset where date_assess >= '", DateStart, "' and date_assess <= '", DateEnd, "'")
  } else if (Time=="AM") {
    Query <- paste("SELECT * from infoex.mindset where date_assess >= '", DateStart, "' and date_assess <= '", DateEnd, "' and extract(hour from infoex.mindset.datetime_assess) < 12")
  } else if (Time=="PM") {
    Query <- paste("SELECT * from infoex.mindset where date_assess >= '", DateStart, "' and date_assess <= '", DateEnd, "' and extract(hour from infoex.mindset.datetime_assess) >= 12")
  } else {
    stop("Value for Time parameter not supported!")
  }

  Mindset <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
 
  ## Location
  Loc <- getRecordsFromUUID(Operation, "infoex", "mindset_loc_x", DBType=DBType, 
                            ForeignUUID=Mindset$uuid, ForeignUUIDCol="infoexmindset_uuid")
  
  ## Filtering out location names
  if (!is.na(LocationName)) {
    
    Loccat <- getRecordsFromUUID(Operation, "loccat", "all_infoex_locations", DBType=DBType, 
                                 UUID=Loc$loccat_uuid)
    
    Loccat <- Loccat[Loccat$name %in% LocationName]
    Loc <- Loc[Loc$loccat_uuid %in% Loccat$uuid]
    Mindset <- Mindset[Mindset$uuid %in% Loc$infoexmindset_uuid]
  
  }
  
  ## Turn into factor
  Mindset$mindset <- factor(Mindset$mindset)
  
  ## Output
  Output <- list(Mindset=Mindset,
                 Loc=Loc)
  
  class(Output) <- append(class(Output), "MindsetObject")
  
  return(Output)

}