#' Get location catalog
#'
#' Get location catalog
#' @param Operation Name of operation
#' @param DBType Name of database. Can only be 'Main' (default) or 'NodeJS'
#' @param LocalProjection Switch whether location catalog should be projected into local UTM grid. Default is T.
#' @param WithoutDelete Switch to specify whether location marked as deleted should be omitted. Default is T.
#' @param AsCentroidsSkiRuns Switch to specify whether ski runs should be retrieved as point locations at their centroids. Default is F.
#' @param AsCentroidsSkiLines Switch to specify whether ski lines should be retrieved as point locations at their centroids. Default is F.
#' @param CombineSkiLinesRunsCentroids Switch for combining the centroid of ski lines and ski runs into a single SPDF called SkiAll. Default is F. Is only applied if both AsCentroidsSkiRuns and AsCentroidsSkiLine are TRUE.
#' 
#' @return List with spatial dataframes for each location type in the location catalog.
#' 
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "CMHGL"
#' LocCat <- getLocCatlog(Operation)
#'
#' @export

getLocCatlog <- function(Operation, DBType="Main", LocalProjection=T, WithoutDeleted=T, AsCentroidsSkiRuns=F, AsCentroidsSkiLines=F, CombineSkiLinesRunsCentroids=F) {

  ## Retrieve locations
  OpArea <- getSpatialObjectsDataFrameFromUUID(Operation, DBType=DBType, "loccat", "oparea", LocalProjection=LocalProjection, WithoutDeleted=WithoutDeleted)
  Zones <- getSpatialObjectsDataFrameFromUUID(Operation, DBType=DBType, "loccat", "zones", LocalProjection=LocalProjection, WithoutDeleted=WithoutDeleted)
  SkiRuns <- getSpatialObjectsDataFrameFromUUID(Operation, DBType=DBType, "loccat", "skiruns", LocalProjection=LocalProjection, WithoutDeleted=WithoutDeleted, AsCentroid = AsCentroidsSkiRuns)
  SkiRunSegments <- getSpatialObjectsDataFrameFromUUID(Operation, DBType=DBType, "loccat", "skirun_segments", LocalProjection=LocalProjection, WithoutDeleted=WithoutDeleted)
  SkiLines <- getSpatialObjectsDataFrameFromUUID(Operation, DBType=DBType, "loccat", "skilines", LocalProjection=LocalProjection, WithoutDeleted=WithoutDeleted, AsCentroid = AsCentroidsSkiLines)
  SkiLineSegments <- getSpatialObjectsDataFrameFromUUID(Operation, DBType=DBType, "loccat", "skiline_segments", LocalProjection=LocalProjection, WithoutDeleted=WithoutDeleted)
  WxSites <- getSpatialObjectsDataFrameFromUUID(Operation, DBType=DBType, "loccat", "wxsites", LocalProjection=LocalProjection, WithoutDeleted=WithoutDeleted)
  CatRoads <- getSpatialObjectsDataFrameFromUUID(Operation, DBType=DBType, "loccat", "catroads", LocalProjection=LocalProjection, WithoutDeleted=WithoutDeleted)
  Landings <- getSpatialObjectsDataFrameFromUUID(Operation, DBType=DBType, "loccat", "landings", LocalProjection=LocalProjection, WithoutDeleted=WithoutDeleted)
  Cruxes <- getSpatialObjectsDataFrameFromUUID(Operation, DBType=DBType, "loccat", "cruxes", LocalProjection=LocalProjection, WithoutDeleted=WithoutDeleted)
  Clusters <- getSpatialObjectsDataFrameFromUUID(Operation, DBType=DBType, "loccat", "clusters", LocalProjection=LocalProjection, WithoutDeleted=WithoutDeleted)

  ## Combine skilines and ski run centroids
  if ((!AsCentroidsSkiRuns | !AsCentroidsSkiLines) & CombineSkiLinesRunsCentroids) {
    warning("CombineSkiLinesRunsCentroids is only applied if both AsCentroidsSkiRuns and AsCentroidsSkiLines are TRUE!", immediate. = T)
  }
  
  if (AsCentroidsSkiRuns & AsCentroidsSkiLines & CombineSkiLinesRunsCentroids) {
    SkiAll <- rbind(SkiRuns, SkiLines)  
  } else {
    SkiAll <- NULL
  }
  
  ## Put list together for output
  Output <- list(OpArea=OpArea,
                 Zones=Zones,
                 SkiRuns=SkiRuns,
                 SkiRunSegments=SkiRunSegments,
                 SkiLines=SkiLines,
                 SkiLineSegments=SkiLineSegments,
                 SkiAll= SkiAll,
                 WxSites=WxSites,
                 CatRoads=CatRoads,
                 Landings=Landings,
                 Cruxes=Cruxes,
                 Clusters=Clusters)
  
  class(Output) <- append(class(Output), "LocCatalog")
  
  return(Output)
  
}