#' Gets name of new GPS track files and check their validity.
#'
#' Gets name of new GPS track files and check their validity. Emails if error detected.
#' @param Operation Name of operation
#' @param Folder Complete pathname of folder with GPS track files.
#' @param EmailAddressees Array of email addressees that should be notified. 
#' @param CorrectYear Last two digits of correct year
#' @export

getNewGPSFileNameListAndCheckValidity <- function (Operation, Folder, CorrectYear, EmailAddressees=NA,  SMTPSettings=SARPSmtpSettings, GPSTestFiles=c("161224_02_PH.csv", "171224_02_PH.csv", "181224_02_PH.csv")) {

  ## Get name of new files
  GPSTrackFileListFolder <- list.files(path = Folder, pattern = ".csv")
  GPSTrackFileListProcessed <- getRecordsFromQuery(Operation, Query="SELECT raw_file from gps.tracks")$raw_file
  
  ## Delete names of file that have already been processed
  GPSTrackFileList <- GPSTrackFileListFolder
  GPSTrackFileList <- GPSTrackFileList[!(toupper(GPSTrackFileList) %in% toupper(GPSTrackFileListProcessed))]
  
  ## Delete names of test files
  GPSTrackFileList <- GPSTrackFileList[!(toupper(GPSTrackFileList) %in% toupper(GPSTestFiles))]
  
  if(length(GPSTrackFileList)>0) {
    
    ## Splitting of file names
    Split <- strsplit(GPSTrackFileList,"[.]")
    GPSTrackFileList_Name  <- unlist(Split)[2*(1:length(GPSTrackFileList))-1]
    
    Split <- strsplit(GPSTrackFileList_Name,"[_]")
    GPSTrackFileList_Date  <- unlist(Split)[3*(1:length(GPSTrackFileList))-2]
    GPSTrackFileList_Unit  <- unlist(Split)[3*(1:length(GPSTrackFileList))-1]
    GPSTrackFileList_Guide <- unlist(Split)[3*(1:length(GPSTrackFileList))]
    
    ## Initialize check DF
    CheckDF <- data.frame(FileName = GPSTrackFileList,
                          Yr = T,
                          Mth = T,
                          Day = T,
                          Unit = T,
                          Guide = T)
    
    # CheckDF <<- CheckDF
    
    ## Check 
    for (Index_File in 1:nrow(CheckDF)) {
    
      # print(as.character(CheckDF$FileName[Index_File]))
      # print(paste(GPSTrackFileList_Date[Index_File], "-", GPSTrackFileList_Unit[Index_File], "-", GPSTrackFileList_Guide[Index_File]))
      
      if(nchar(GPSTrackFileList_Date[Index_File])==8) {GPSTrackFileList_Date[Index_File] <- substr(GPSTrackFileList_Date[Index_File], 3, 8)}

      if(!(((as.numeric(substr(GPSTrackFileList_Date[Index_File], 3, 4)) > 7) & (substr(GPSTrackFileList_Date[Index_File], 1, 2) == CorrectYear-1)) | 
           ((as.numeric(substr(GPSTrackFileList_Date[Index_File], 3, 4)) <= 7) & (substr(GPSTrackFileList_Date[Index_File], 1, 2) == CorrectYear)))){
        CheckDF$Yr[Index_File] <- F
      }
      if((as.numeric(substr(GPSTrackFileList_Date[Index_File], 3, 4)) < 0) | (as.numeric(substr(GPSTrackFileList_Date[Index_File], 3, 4)) > 12)) {
        CheckDF$Mth[Index_File] <- F
      }
      if((as.numeric(substr(GPSTrackFileList_Date[Index_File], 5, 6)) < 0) | (as.numeric(substr(GPSTrackFileList_Date[Index_File], 5, 6)) > 31)) {
        CheckDF$Day[Index_File] <- F
      } 
      if(!is.na(as.numeric(GPSTrackFileList_Unit[Index_File]))) {
        if((as.numeric(GPSTrackFileList_Unit[Index_File]) < 0) & (as.numeric(GPSTrackFileList_Unit[Index_File]) > 100)) {
          CheckDF$Unit[Index_File] <- F
        }
      } else {
        CheckDF$Unit[Index_File] <- F
      }
      if(is.na(GPSTrackFileList_Guide[Index_File])) {CheckDF$Guide[Index_File] <- F}
    }
    
    ## Notify if error detected
    if(!is.na(EmailAddressees[1])) {
      
      for (Index_File in 1:nrow(CheckDF)) {
        
        if (!(CheckDF$Yr[Index_File] & CheckDF$Mth[Index_File] & CheckDF$Day[Index_File] & CheckDF$Unit[Index_File] & CheckDF$Guide[Index_File])) {
          
          EmailMessage <- paste0("GPS Processing: Issue with ", Operation, " file name: ", CheckDF$FileName[Index_File])
          sendNotificationEmail(EmailAddressees, EmailMessage)
          warning(EmailMessage, immediate. = T)
          
        }
        
      }
      
    }
    
    ## Remove invalid names
    Output <- GPSTrackFileList[CheckDF$Yr & CheckDF$Mth & CheckDF$Day & CheckDF$Unit & CheckDF$Guide]
    
  } else {
    
    Output <- character()
    
  }
  
  ## Return valid names
  return(Output)
  
}
