#' Combines values from specified raster value arrays for list of runs and combines into dataframe.
#'
#' Combines values from specified raster value arrays for list of runs and combines into dataframe.
#' @param Operation Name of Operation. Used for accessing PostgreSQL database.
#' @param RunUUIDList Array of run uuids. Default is NA, which retrieved all records.
#' @param RasterNames Array of raster names to be included. By default, all available raster value arrays are included.
#' @param AddElevBand Switch to specify whether elevation band labels (BTL, TL, ALP) should be included as additional data frame column.
#' @param DBType Optional parameter to specify which database type to get the RunObjects from. Values can be 'Main' (default) or 'NodeJS'.
#' @param SuppressProgressMsgs Switch to turn off progress messages. Default is FALSE.
#' 
#' @return Object of class 'RasterValueDf': Dataframe with all value pairs for specified runs.
#' 
#' @examples 
#'  require(SarpGPSTools)
#'  require(SarpGPSToolsPrivate)
#'  
#'  Operation <- "MWHS"
#'  RunUUIDList <- getRecordsFromQuery(Operation, "Select UUID from gps.qc_runs")$uuid
#'  RunUUIDList <- RunUUIDList[1:100]
#'  
#'  ## Get raster data for all raster arrays
#'  RasterData <- getRasterValueDfFromRunUUID(Operation, RunUUIDList)
#'  
#'  ## Get raster data for a specific raster array
#'  RasterData <- getRasterValueDfFromRunUUID(Operation, RunUUIDList, "Convexities")
#'    
#' @export

getRasterValueDfFromRunUUID <- function (Operation, RunUUIDList=NA, RasterNames=NA, AddElevBand=T, DBType="Main", SuppressProgressMsgs=F) {

  ## Initialize process variables
  RunsSkippedBecauseArrayDiffSize <- 0
  RunsSkippedBacauseMissingArrays <- 0
  RunsProcessed_Index <- 0
  List_RunDF <- list(NA)

  ## Get raster information
  RasterList <- listRasterRefAndArrayTbl(Operation)
  options(warn=-1)
  if (!is.na(RasterNames)){RasterList <- RasterList[RasterList$name %in% RasterNames,]}
  options(warn=0)
  if (nrow(RasterList)==0) (stop("No valid rasters selected!"))

  ## Get coordinate information
  if(!SuppressProgressMsgs) {print(paste0(Sys.time(), " - Retrieving coordinate information ..."))}
  ArrayDF_coord <- getRecordsFromUUID(Operation, "gis", "gpsruns_array_coord", RunUUIDList, UUIDCol = "gpsruns_uuid", DBType=DBType, SuppressOrderWarning = T)
  NumOfRuns <- nrow(ArrayDF_coord)
  
  
  ## Initialize names for df and array holding raster information
  Names_ArrayDFs <- character(0)
  Names_Arrays <- character(0)

  ## Retrieving raster information
  for (Index_Raster in 1:nrow(RasterList)) {
    if(!SuppressProgressMsgs) {print(paste0(Sys.time(), " - Retrieving raster array for ", RasterList$name[Index_Raster], " ..."))}

    Names_ArrayDFs <- c(Names_ArrayDFs, paste0("ArrayDF_", RasterList$gis_array_name_ext[Index_Raster]))
    Names_Arrays   <- c(Names_Arrays, paste0("Array_", RasterList$gis_array_name_ext[Index_Raster]))

    assign(Names_ArrayDFs[Index_Raster], getArrayTblFromUUID(Operation, RasterList$name[Index_Raster], RunUUIDList, DBType=DBType))

  }

  ## Check whether ArrayDF have any values and delete from RasterList
  ArrayDF_nrow <- numeric(0)
  for (Index_Raster in 1:nrow(RasterList)) {
    ArrayDF_nrow <- c(ArrayDF_nrow, nrow(get(Names_ArrayDFs[Index_Raster])))
  }
  
  if(any(ArrayDF_nrow==0)) {
    warning(paste0("Raster value array '", paste0(RasterList$name[which(ArrayDF_nrow==0)], collapse = "', '"), "' deleted because no values included!"), immediate. = T)
    ## rm(as.character(Names_ArrayDFs[which(ArrayDF_nrow==0)]))
    
    RasterList <- RasterList[-which(ArrayDF_nrow==0),]
    Names_ArrayDFs <- Names_ArrayDFs[-which(ArrayDF_nrow==0)]
    Names_Arrays <- Names_Arrays[-which(ArrayDF_nrow==0)]
  }
  
  ## Check whether all value arrays have records for all runs
  ArrayDF_nrow <- numeric(0)
  for (Index_Raster in 1:nrow(RasterList)) {
    ArrayDF_nrow <- c(ArrayDF_nrow, nrow(get(Names_ArrayDFs[Index_Raster])))
  }
  
  if(min(ArrayDF_nrow) < NumOfRuns) {
   stop(paste0("Raster arrays are not available for all runs (", min(ArrayDF_nrow), " instead of ", length(RunUUIDList), ").", 
               "\n  Please adjust your selection. -> Likely changing from gps.runs to gps.qc_runs will fix the problem!)"))
  }

  ## Retrieving date information for run
  if(!SuppressProgressMsgs) {print(paste0(Sys.time(), " - Retrieving dates ..."))}
  RunAttributes <- getRecordsFromUUID(Operation, "gps", "runs", RunUUIDList, ResultCol="date_local", DBType=DBType, SuppressOrderWarning=T)
  
  
  ## Extracting information for runs and splicing together into master df
  ## ********************************************************************
  
  ## Explicit list of run uuids to ensure that the merging is done correctly.
  RunUUIDList <- ArrayDF_coord$gpsruns_uuid
  
  for (Index_Run in 1:NumOfRuns) {
    
    # Index_Run <- 1

    if (Index_Run==1 | Index_Run==NumOfRuns | (Index_Run %% 500) == 0) {
      if(!SuppressProgressMsgs) {print(paste0(Sys.time(), " - Processing Run ", Index_Run))}
    }

    ## Initialize array for length of arrays
    ArrayLength <- numeric(0)
    
    ## Get coordinate information
    Array_lon <- convertArrayFromSQL(ArrayDF_coord$lon[ArrayDF_coord$gpsruns_uuid==RunUUIDList[Index_Run]])
    Array_lat <- convertArrayFromSQL(ArrayDF_coord$lat[ArrayDF_coord$gpsruns_uuid==RunUUIDList[Index_Run]])
    Array_buffer <- convertArrayFromSQL(ArrayDF_coord$buffer[ArrayDF_coord$gpsruns_uuid==RunUUIDList[Index_Run]])
    Array_dist_start <- convertArrayFromSQL(ArrayDF_coord$dist_start[ArrayDF_coord$gpsruns_uuid==RunUUIDList[Index_Run]])
    Array_dist_end <- convertArrayFromSQL(ArrayDF_coord$dist_end[ArrayDF_coord$gpsruns_uuid==RunUUIDList[Index_Run]])
    ArrayLength <- c(ArrayLength, length(Array_lon))

    ## Extract arrays for correct run
    for (Index_Raster in 1:nrow(RasterList)) {

      # Index_Raster <- 1
      ArrayDF_Temp <- get(Names_ArrayDFs[Index_Raster])
      assign(Names_Arrays[Index_Raster], ArrayDF_Temp[ArrayDF_Temp$gpsruns_uuid==RunUUIDList[Index_Run],2][[1]])
      ArrayLength <- c(ArrayLength, length(get(Names_Arrays[Index_Raster])))
    }
    rm(ArrayDF_Temp)

    ## Only process run is all arrays are of the same length
    if ((min(ArrayLength) == max(ArrayLength)) & min(ArrayLength)>0) {

      ## Initializing dataframe
      RunDF <- data.frame(gpsruns_uuid=rep(RunUUIDList[Index_Run], ArrayLength[1]),
                          date=rep(RunAttributes$date_local[RunAttributes$uuid==RunUUIDList[Index_Run]], ArrayLength[1]),
                          lon=Array_lon,
                          lat=Array_lat,
                          buffer=Array_buffer,
                          dist_start=Array_dist_start,
                          dist_end=Array_dist_end)

      ## Creating dataframe for single run
      for (Index_Raster in 1:nrow(RasterList)) {
        RunDF <- cbind(RunDF, get(Names_Arrays[Index_Raster]))
        names(RunDF)[Index_Raster+7] <- RasterList$gis_array_name_ext[Index_Raster]
      }

      ## Adding RunDF to list
      RunsProcessed_Index <- RunsProcessed_Index + 1
      List_RunDF[[RunsProcessed_Index]] <- RunDF

    } else if (min(ArrayLength)==0) {

      RunsSkippedBecauseMissingArrays <- RunsSkippedBecauseMissingArrays+1

    } else {

      RunsSkippedBecauseArrayDiffSize <- RunsSkippedBecauseArrayDiffSize+1

    }

    ## Clean up
    rm(ArrayLength)

  }

  if (length(List_RunDF)==0) {stop("No run data frames in list!")}

  ## Combining the all the RunDF into single DF via data.table::rbindlist
  if(!SuppressProgressMsgs) {print(paste0(Sys.time(), " - Combining RunDF into single data.frame"))}
  invisible(lapply(List_RunDF, setattr, name = "class", value = c("data.table",  "data.frame")))
  OutputDF <- rbindlist(List_RunDF)

  ## Converting data:table back to data.frame
  setattr(OutputDF, "class", "data.frame")

  ## Add elevation band
  if(AddElevBand & ("Elevation" %in% RasterList$name)) {

    if(!SuppressProgressMsgs) {print(paste0(Sys.time(), " - Adding elevation band labels"))}

    ## Check for elevation column
    if(!("elev" %in% names(OutputDF))) {stop("OutputDF does not have a 'elev' column!")}

    Btl2Tl <- getElevBandBoundary(Operation, "TL", "Bottom")
    Tl2Alp <- getElevBandBoundary(Operation, "TL", "Top")

    OutputDF$elevband <- ordered(ifelse(OutputDF$elev<=Btl2Tl, "BTL",
                                 ifelse(OutputDF$elev<=Tl2Alp, "TL", "ALP")), levels = c("BTL", "TL", "ALP"))

  }

  ## Warning about skipped runs
  if (RunsSkippedBecauseArrayDiffSize > 0) {
    warning(paste0(RunsSkippedBecauseArrayDiffSize, " run(s) skipped because value arrays were not of same size!"))
  }
  if (RunsSkippedBacauseMissingArrays > 0) {
    warning(paste0(RunsSkippedBacauseMissingArrays, " run(s) skipped because missing value arrays!"))
  }

  if(!SuppressProgressMsgs) {print(paste0(Sys.time(), " - Processing complete"))}
  
  ## Add class label
  class(OutputDF) <- append(class(OutputDF), "RasterValueDf")

  ## Return Output
  return(OutputDF)

}
