#' Gets record from database based on list of UUID
#'
#' Gets record from database based on list of UUID
#' @param Operation Name of operation
#' @param Schema Name of schema that contains DBTbl
#' @param DBTbl Name of database table with geometry
#' @param DBType Type of database to query from. Can be 'Main' (default) or 'NodeJS' for database that feeds online viewer.
#' @param UUID Single UUID or array of UUIDs. Entire table is returned if no UUID value is provided.
#' @param UUIDCol Name of column with UUID in database table
#' @param ForeignUUID Single or array of foreign key UUID values for querying table over ForeignUUIDCol.
#' @param ForeignUUIDCol Name of foreign key UUID column against which the ForeignUUID values are applied.
#' @param ResultCol Names of columns included in results table (array of names). UUID column is added automatically. No additional columns are added when an empy string ("") is provided. All columns are returned if no value is provided.
#' @param AddRowNames Switch whether uuids should be added as row names
#' @param GeomCol Name of geometry column to delete it from results dataframe
#' @param Verbose Switch for printing of query
#' @param SuppressPostgreSQLWarnings Switch for suppressing warning messages from postgresqlExecStatement. Turn on for debugging!
#' @param SuppressOrderWarning Switch for suppressing warning message about oder of databframe being not the same as the uuid list.
#' @return Dataframe with the specified columns for the provided UUIDs
#' 
#' @export

getRecordsFromUUID <- function(Operation, Schema, DBTbl, DBType="Main", UUID=NA, UUIDCol="uuid", ForeignUUID=NA, ForeignUUIDCol=NA, GeomCol="geom", ResultCol=NA, AddRowNames=T, Verbose=F, SuppressPostgreSQLWarnings=T, SuppressOrderWarning=T) {

  if(length(UUID)>1 & !SuppressOrderWarning) {warning("Does not necessarily provide records in same order as uuid array!", immediate.=T)}

  ## Build query - Select
  if(length(ResultCol)==1) {
    if(is.na(ResultCol)) {
      Query <- paste0("SELECT *")
    } else {
      if (is.na(ForeignUUIDCol)) {
        if (ResultCol=="") {
          Query <- paste0("SELECT \"", UUIDCol, "\"")
        } else {
          Query <- paste0("SELECT \"", UUIDCol, "\", \"", ResultCol, "\"")
        }
      } else {
        if (ResultCol=="") {
          Query <- paste0("SELECT \"", UUIDCol, "\", \"", ForeignUUIDCol, "\"")
        } else {
          Query <- paste0("SELECT \"", UUIDCol, "\", \"", ForeignUUIDCol, "\", \"", ResultCol, "\"")
        }
      }
    }
  } else {
    if (is.na(ForeignUUIDCol)) {
      Query <- paste0("SELECT \"", UUIDCol, "\", \"", paste(ResultCol, collapse = "\", \""), "\"")
    } else {
      Query <- paste0("SELECT \"", UUIDCol, "\", \"", ForeignUUIDCol, "\", \"", paste(ResultCol, collapse = "\", \""), "\"")
    }
  }

  ## Build query - Table reference
  if (Schema=="") {
    Query <- paste0(Query, " FROM \"", DBTbl, "\"")
  } else {
    Query <- paste0(Query, " FROM \"", Schema, "\".\"", DBTbl, "\"")
  }

  ## Add where if necessary
  if ((!is.na(UUID[1])) | (!is.na(ForeignUUID[1]))) {
    Query <- paste0(Query, " WHERE ")
  }
  
  ## Build query - UUID selection
  if(length(UUID)==1) {
    if(!is.na(UUID)) {
      Query <- paste0(Query, "(\"", UUIDCol, "\"=", convertStringToSQL(UUID), ")")
    }
  } else {
    Query <- paste0(Query, "(")
    for (i in 1:length(UUID)) {
      Query <- paste0(Query, "\"", UUIDCol, "\"=", convertStringToSQL(UUID[i]), " OR ")
    }
    Query <- paste0(substr(Query, 1, nchar(Query)-4), ")")
  }
  
  ## Add AND if necessary
  if ((!is.na(UUID[1])) & (!is.na(ForeignUUID[1]))) {
    Query <- paste0(Query, " AND ")
  }
  
  ## Building query - ForeignUUIDCol values
  if(length(ForeignUUID)==1) {
    if (!is.na(ForeignUUID)) {
      if (typeof(ForeignUUID)=="character") {Query <- paste0(Query, "(\"", ForeignUUIDCol, "\"=", convertStringToSQL(ForeignUUID), ")")}
      else if (typeof(ForeignUUID)=="double") {Query <- paste0(Query, "(\"", ForeignUUIDCol, "\"=", convertNumericToSQL(ForeignUUID), ")")}
      else {stop(paste0("Data type '", typeof(ForeignUUID), "' not supported!"))}
    }
  } else {
    Query <- paste0(Query, "(")
    for (Index_Value in 1:length(ForeignUUID)) {
      if (typeof(ForeignUUID)=="character") {Query <- paste0(Query, "\"", ForeignUUIDCol, "\"=", convertStringToSQL(ForeignUUID[Index_Value]), " OR ")}
      else if (typeof(ForeignUUID)=="double") {Query <- paste0(Query, "\"", ForeignUUIDCol, "\"=", convertNumericToSQL(ForeignUUID[Index_Value]), " OR ")}
      else {stop(paste0("Data type '", typeof(ForeignUUID), "' not supported!"))}
    }
    Query <- paste0(substr(Query, 1, nchar(Query)-4), ")")
    rm(Index_Value)
  }

  ## Retrieve results from DB
  Results <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

  ## Set rownames to UUIDS
  if (AddRowNames) {row.names(Results) <- Results$uuid}

  ## Deletes geometry column if it is there
  if (GeomCol %in% colnames(Results)) {
    Results <- deleteDFColumnsByName(Results, GeomCol)
  }

  ## Return dataframe
  return(Results)
}
