#' Get run character assessments
#'
#' Get run character assessments
#' @param Operation Name of operation
#' @param SkirunNames Single or array of skirun names
#' @param SkirunUUIDs Single or array of skirun UUIDs
#' @param ZoneNames Single or array of zone names
#' @param ZoneUUIDs Single or array of skirun UUIDs
#' @param ClusterNames Single or array of cluster names
#' @param ClusterUUIDs Single or array of cluster UUIDs
#' @param FormatAssessTbl Swtich for whether assessment table should be formatted. Default is TRUE.
#' @param AddNames Switch for adding run, cluster and zone names. Default is TRUE.
#' @param FromSurveyDB Switch for specifying source of run assessments to be survey database
#' @param Verbose Switch for printing of query.
#' @return Dataframe with run character assessments
#' 
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "CMHGL"
#' 
#' ## Specific runs
#' SkirunNames <- c("North Goat", "Alexis")
#' RunChar_GPSDB <- getRunCharAssessments(Operation, SkirunNames=SkirunNames)
#' RunChar_Survey <- getRunCharAssessments(Operation, SkirunNames=SkirunNames, FromSurveyDB=T)
#' 
#' ## Entire cluster
#' ClusterNames <- "Wagner Bowls"
#' RunChar_GPSDB <- getRunCharAssessments(Operation, ClusterNames=ClusterNames)
#' RunChar_SurveyDB <- getRunCharAssessments(Operation, ClusterNames=ClusterNames, FromSurveyDB=T)
#' 
#' ## Entire zone from name
#' ZoneNames <- "Lake Creek"
#' RunChar_GPSDB <- getRunCharAssessments(Operation, ZoneNames=ZoneNames)
#' RunChar_SurveyDB <- getRunCharAssessments(Operation, ZoneNames=ZoneNames, FromSurveyDB=T)
#' 
#' ## Entire zone from uuid
#' ZoneUUID <- "ce345152-10b3-427c-bc8f-4a367b22637e"
#' RunChar_GPSDB <- getRunCharAssessments(Operation, ZoneUUIDs=ZoneUUID)
#' RunChar_SurveyDB <- getRunCharAssessments(Operation, ZoneUUIDs=ZoneUUID, FromSurveyDB=T)
#' 
#' @export


getRunCharAssessments <- function(Operation, SkirunNames=NA, SkirunUUIDs=NA, ClusterNames=NA, ClusterUUIDs=NA, ZoneNames=NA, ZoneUUIDs=NA, FormatAssessTbl=T, AddNames=T, FromSurveyDB=F, Verbose=F) {
  
  ## Error check
  Check <- !is.na(SkirunNames[1]) + !is.na(SkirunUUIDs[1]) + 
           !is.na(ClusterNames[1]) + !is.na(ClusterUUIDs[1]) + 
           !is.na(ZoneNames[1]) + !is.na(ZoneUUIDs[1])
  
  ## More than one variable specified
  if (Check>1) {
    stop("Only one of the name parameters can be used!")
  }
  
  print("Retreiving data ...")
  
  ## Get assessment for ski run names     
  if (!is.na(SkirunNames[1]) | !is.na(SkirunUUIDs[1])) {
    
    if (!is.na(SkirunNames[1])) {
      SkirunUUIDs <- getUUIDsFromNames(Operation=Operation, Schema="loccat", DBTbl="skiruns", Names=SkirunNames, Verbose=Verbose)
    }
    
    if (is.na(SkirunUUIDs[1])) {
      Assess <- NULL
    } else {
      if (!FromSurveyDB) {
        Assess <- getRecordsFromUUID(Operation=Operation, Schema="runchar", DBTbl="assessments", ForeignUUID = SkirunUUIDs,
                                     ForeignUUIDCol = "loccatskiruns_uuid", Verbose=Verbose)
      } else {
        Query <- paste0("SELECT * FROM assessments_output WHERE loccatskiruns_uuid IN ('", paste(SkirunUUIDs, collapse = "', '"), "')")
        Assess <- getRecordsFromQueryTerrainSurvey(Query)
      }
    }
    
    ## Get assessments for ski runs within zones    
  } else if (!is.na(ZoneNames[1]) | !is.na(ZoneUUIDs[1])) {
    
    if(!is.na(ZoneNames[1])){
      ZoneUUIDs <- getUUIDsFromNames(Operation=Operation, Schema="loccat", DBTbl="zones", Names=ZoneNames, Verbose=Verbose)
    }
    
    if(is.na(ZoneUUIDs[1])) {
      Assess <- NULL
    } else {
      SkirunUUIDs <- getRecordsFromUUID(Operation=Operation, Schema="loccat", DBTbl="skiruns", ForeignUUID = ZoneUUIDs,
                                        ForeignUUIDCol="loccatzone_uuid", ResultCol = "", Verbose=Verbose)$uuid
      
      if (!FromSurveyDB) {
        Assess <- getRecordsFromUUID(Operation=Operation, Schema="runchar", DBTbl="assessments", ForeignUUID = SkirunUUIDs,
                                     ForeignUUIDCol = "loccatskiruns_uuid", Verbose=Verbose)
      } else {
        Query <- paste0("SELECT * FROM assessments_output WHERE loccatskiruns_uuid IN ('", paste(SkirunUUIDs, collapse = "', '"), "')")
        Assess <- getRecordsFromQueryTerrainSurvey(Query)
      }
    }
    
    ## Get assessments for ski runs within clusters    
  } else if (!is.na(ClusterNames[1]) | !is.na(ClusterUUIDs[1])) {
    
    if (!is.na(ClusterNames[1])) {
      ClusterUUIDs <- getUUIDsFromNames(Operation=Operation, Schema="loccat", DBTbl="clusters", Names=ClusterNames, Verbose=Verbose)
    }
    
    if(is.na(ClusterUUIDs[1])) {
      Assess <- NULL
    } else {
      SkirunUUIDs <- getRecordsFromUUID(Operation=Operation, Schema="loccat", DBTbl="skiruns", ForeignUUID = ClusterUUIDs,
                                        ForeignUUIDCol="loccatclusters_uuid", ResultCol = "", Verbose=Verbose)$uuid
      
      if (!FromSurveyDB) {
        Assess <- getRecordsFromUUID(Operation=Operation, Schema="runchar", DBTbl="assessments", ForeignUUID = SkirunUUIDs,
                                     ForeignUUIDCol = "loccatskiruns_uuid", Verbose=Verbose)
      } else {
        Query <- paste0("SELECT * FROM assessments_output WHERE loccatskiruns_uuid IN ('", paste(SkirunUUIDs, collapse = "', '"), "')")
        Assess <- getRecordsFromQueryTerrainSurvey(Query)
      }
    }
  
  ## Get all assessments  
  } else {
    
    if (!FromSurveyDB) {
      Assess <- getRecordsFromUUID(Operation=Operation, Schema="runchar", DBTbl="assessments", Verbose=Verbose)
    } else {
      Query <- paste0("SELECT * FROM assessments_output WHERE operation='", Operation, "'")
      Assess <- getRecordsFromQueryTerrainSurvey(Query)
    }

  }
  
  if(is.null(Assess)) {
    
    warning("No assessment records found! Returning NULL.", immediate. = T)
    
  } else {
    
    print(paste0("Retreived ", nrow(Assess), " character assessment(s)."))
    
    ## Format output df
    if (FormatAssessTbl) {
      if(!FromSurveyDB) {
        Assess <- formatRunCharAssessmentDF(Operation=Operation, AssessmentDF=Assess, Verbose=Verbose)
      } else {
        warning("Formatting of assessment table from survey DB not implemented yet!", immediate. = T)
      }
    }
    
    ## Adding names
    if(AddNames) {
      
      print("Adding run, zone and cluster names ...")
      
      RunNames <- getRecordsFromUUID(Operation=Operation, Schema="loccat", DBTbl="skiruns", ResultCol = c("name", "loccatzone_uuid", "loccatclusters_uuid"))
      names(RunNames)[2] <- "loccatskiruns_name"
      ClusterNames <- getRecordsFromUUID(Operation=Operation, Schema="loccat", DBTbl="clusters", ResultCol = "name")
      names(ClusterNames)[2] <- "loccatclusters_name"
      ZoneNames <- getRecordsFromUUID(Operation=Operation, Schema="loccat", DBTbl="zones", ResultCol = "name")
      names(ZoneNames)[2] <- "loccatzone_name"
      
      Assess <- merge(Assess, RunNames, by.x="loccatskiruns_uuid", by.y="uuid", all.x=T)
      Assess <- merge(Assess, ClusterNames, by.x="loccatclusters_uuid", by.y="uuid", all.x=T)
      Assess <- merge(Assess, ZoneNames, by.x="loccatzone_uuid", by.y="uuid", all.x=T)
      
      Assess$loccatskirun_name <- factor(Assess$loccatskiruns_name)
      Assess$loccatcluster_name <- factor(Assess$loccatclusters_name)
      Assess$loccatzone_name <- factor(Assess$loccatzone_name)
    
    }
    
  }
  
  ## Return df
  return(Assess)
  
}