#' Get run list ratings for an operation
#'
#' Get run list ratings for an operation.
#' @param Operation Name of Operation. Used for accessing PostgreSQL database.
#' @param DateStart Start date in YYYY-MM-DD format. Default value is NA.
#' @param DateEnd End date in YYYY-MM-DD format. Default value is DateStart.
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS
#' @return dataframe with run list codings.
#'
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' require(SarpGPSToolsVis)
#'
#' Operation <- "CMHGL"
#' DateStart <- "2014-01-01"
#' DateEnd   <- "2014-03-31"
#'
#' ## get RunList
#' RunList <- getRunListData(Operation, DateStart = DateStart, DateEnd = DateEnd)
#'
#' @export

getRunListData <- function(Operation, DateStart, DateEnd=DateStart, StandardizedCoding=T, ShowSummary=T, DBType="Main", Verbose=F){

  ## BUILD QUERY AND GET DATA
  if (StandardizedCoding) {
    Query <- paste0("SELECT * from runlist.get_runlist_data('", DateStart, "', '", DateEnd, "')")
  } else {
    Query <- paste0("SELECT * from runlist.get_runlist_data('", DateStart, "', '", DateEnd, "', _standard:=FALSE)")
  }

  RunListData <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose)

  ## PRINT DATA SUMMARY
  if (ShowSummary) {
    ## PRINT DATA SUMMARY
    print("=================================================", quote=F)
    print("RUN LIST DATA SUMMARY", quote=F)
    print("-------------------------------------------------", quote=F)
    print(paste0("- Number of days in the run list: ", length(unique(RunListData$date))), quote=F)
    print(paste0("- Number of ski runs/lines in the run list: ", length(unique(RunListData$loc_uuid))), quote=F)
    print(paste0("- From date: ",min(RunListData$date)), quote=F)
    print(paste0("- To date: ", max(RunListData$date)), quote=F)
    table(RunListData$code_general)
    print("=================================================", quote=F)
  }

  if (nrow(RunListData) > 0){

    if (Operation=="NEH"){

      makeCodeAndColorGeneral <- function(RunListData){
        
        ## DEFAULT: NOT DISCUSSED
        RunListData$code_general <- "nd"
        
        for(i in 1:nrow(RunListData)){

          numSkiLines  <- sum(!is.na(RunListData$code_class1[i]), !is.na(RunListData$code_class2[i]), !is.na(RunListData$code_class3[i]))
          numDiscussed <- 3 - sum(is.na(RunListData$code_class1[i]=="nd"), is.na(RunListData$code_class2[i]=="nd"), is.na(RunListData$code_class3[i]=="nd"))

          ## ARE LINES ON THE RUN DISCUSSED
          if (numDiscussed > 0){
            ## AT LEAST ONE LINE ON THE RUN IS OPEN
            if ("green" %in% c(RunListData$code_class1[i], RunListData$code_class2[i], RunListData$code_class3[i])){
              RunListData$code_general[i] <- "green"
            ## CHECK FOR CODE THAT IS RESPONSIBLE FOR RUN NOT OPEN
            ## CHECK CODE ON GREEN SKI LINES IF EXIST
            } else if (!is.na(RunListData$code_class1[i])){
              if (RunListData$code_class1[i]=="red"){RunListData$code_general[i] <- "red"}
              else if (RunListData$code_class1[i]=="black"){RunListData$code_general[i] <- "black"}
              ## CHECK CODE ON BLUE SKI LINES IF EXIST
            } else if (!is.na(RunListData$code_class2[i])){
              if (RunListData$code_class2[i]=="red"){RunListData$code_general[i] <- "red"}
              else if (RunListData$code_class2[i]=="black"){RunListData$code_general[i] <- "black"}
            ## CHECK CODE ON BLACK SKI LINES IF EXIST
            } else if (!is.na(RunListData$code_class3[i])){
              if (RunListData$code_class3[i]=="red"){RunListData$code_general[i] <- "red"}
              else if (RunListData$code_class3[i]=="black"){RunListData$code_general[i] <- "black"}
            }
          }    
            # } else if ("red" %in% c(RunListData$code_class1[i], RunListData$code_class2[i], RunListData$code_class3[i])){
            #   RunListData$code_general[i] <- "red"
            # } else if ("black" %in% c(RunListData$code_class1[i], RunListData$code_class2[i], RunListData$code_class3[i])){
            #   RunListData$code_general[i] <- "red"
            # } else {
            #   RunListData$code_general[i] <- "nd"
            # }


        }#end for


        RunListData$code_general_clr[RunListData$code_general=="green"] <- "#006600"
        RunListData$code_general_clr[RunListData$code_general=="red"]   <- "#cc0000"
        RunListData$code_general_clr[RunListData$code_general=="black"] <- "#000000"
        RunListData$code_general_clr[RunListData$code_general=="nd"]    <- "#ffffff"

        return(RunListData)

      }#end function

      RunListData <- makeCodeAndColorGeneral(RunListData)

    }#end if

  }#end if




  ## RETURN DATA
  return(RunListData)

}
