#' Get names of closest ski line or ski run for gps run from uuid. 
#'
#' Get names of closest ski line or ski run for gps run from uuid.
#' @param Operation Name of operation.
#' @param GPSRunsUUIDs Array of uuids of gps runs
#' @param MinDistInSkirun Minimum distance that run has to be in a polygon in m. Default value is 200 m.
#' @param OnlyLongest Switch for whether only longest section in polygon should be retained. Default value is F.
#' @return Dataframe with list of skirun/skiline uuids and names, and calculated distances.
#' 
#' @examples 
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "CMHGL"
#' 
#' ## Get GPS run uuids for GPS unit and day
#' GPSRunsUUIDs <- getRunUUIDsFromTrackerUnitAndDate(Operation, GpsUnit = '29', DateStart = '2016-01-01')
#' 
#' ## Get run usage list with different defaults
#' RunNamesDefault <- getRunNamesFromGPSRunsUUID(Operation, GPSRunsUUIDs)
#' RunNamesAll <- getRunNamesFromGPSRunsUUID(Operation, GPSRunsUUIDs, MinDistInSkirun = 0)
#' RunNamesOnlyOne <- getRunNamesFromGPSRunsUUID(Operation, GPSRunsUUIDs, OnlyLongest = T)
#' 
#' @export

getRunNamesFromGPSRunsUUID <- function(Operation, GPSRunsUUIDs, MinDistInSkirun=200, OnlyLongest=F, Verbose=F, ShowProgress=T) {

  warning("Check out approach in getGPSRunsInLoccatSkirunSummary for doing this more efficiently!!", immediate. = T)
  
  if (is.null(GPSRunsUUIDs)) {

    print("No GPS runs to be processed!")
    return(NULL)
  
  } else {
  
    ## Check loccat
    NumLoccatSkiruns  <- getRecordsFromQuery(Operation, "select count(uuid) AS num from loccat.skiruns")$num
    NumLoccatSkilines <- getRecordsFromQuery(Operation, "select count(uuid) AS num from loccat.skilines")$num
    
    ## Loop over runs
    for (Index_Runs in 1:length(GPSRunsUUIDs)) {
      
      if(ShowProgress) {print(paste0("Processing GPS track ", Index_Runs, " of ", length(GPSRunsUUIDs), "..."))}
      
      ## Compare with ski run polygons if available
      if (NumLoccatSkiruns>0) {
    
        Query <- paste0("SELECT '", GPSRunsUUIDs[Index_Runs], "' AS gpsruns_uuid, ",
                               "uuid as loccatskiruns_uuid, ", 
                               "name as loccatskiruns_name, ",
                               "round(ST_length(ST_Transform(TblIntersect.geom, (Select srdi from misc.local_srdi limit 1)))) AS length_within_m, ",
                               "null as loccatskilines_uuid, ",
                               "null as loccatskilines_name, ",
                               "null as dist_to_m ",
                        "FROM (select uuid, ",
                                     "name, ",
                                     "ST_intersection((select geom from gps.runs where uuid='", GPSRunsUUIDs[Index_Runs], "' limit 1), geom) AS geom ",
                                     "from loccat.skiruns) AS TblIntersect ", 
                        "WHERE ST_AsText(TblIntersect.geom)!='GEOMETRYCOLLECTION EMPTY'",
                        "ORDER BY length_within_m DESC")  
        
        RunNames <- getRecordsFromQuery(Operation, Query, Verbose=Verbose)
        
        ## Replace table with null if empty
        if (nrow(RunNames)==0) {
          RunNames <- NULL
        
        ## Eliminate sections that are too short
        } else {
          if (OnlyLongest) {
            RunNames <- RunNames[1,]
          } else {
            RunNames <- RunNames[RunNames$length_within_m >= MinDistInSkirun,]
          }
        }
        
      } else {
        RunNames <- NULL
      }
      
      ## Compare with ski lines if avaiable and necessary
      if (is.null(RunNames) & NumLoccatSkilines>0) {
        
        RunNames <- NULL
        warning("Not yet implemented for ski lines!", immediate. = T)
        
        Query <- paste0("")
        
        
        
        
      }
      
      ## Combine results
      if (!is.null(RunNames)) {
        if(exists("RunNamesAll")) {
          RunNamesAll <- rbind(RunNamesAll, RunNames)
        } else {
          RunNamesAll <- RunNames
        }
      }
    }
   
    ## Return results
    return(RunNamesAll)
  
  }
}
