#' Gets array of run UUIDs form GPS tracker unit and date/date range.
#'
#' Gets array of run UUIDs form GPS tracker unit and date/date range.
#' @param Operation Name of operation.
#' @param GpsUnit Name of GPS unit.
#' @param DateStart Starting date of date range.
#' @param DateEnd End date of date range (equal to DateStart by default).
#' @param GuideTypesIncl Guides types to include in list (e.g., 'Lead guide', 'Snow safety').
#' @param GuideTypesExcl Guides types to include in list (e.g., 'Lead guide', 'Snow safety').
#' @param DBType Operational parameter for specifying from which database the run uuids should be retrieved. Values can be 'Main' (default) or 'NodeJS'.
#' @param QCOnly Switch for only including quality controlled runs.
#' @param Verbose Switch for printing of query
#' @param SuppressPostgreSQLWarnings Switch for suppressing warning messages from postgresqlExecStatement. Turn on for debugging!
#' @return Array with run UUIDs
#' @export

getRunUUIDsFromTrackerUnitAndDate <- function(Operation, GpsUnit=NULL, DateStart, DateEnd=DateStart, GuideTypesIncl=NULL, GuideTypesExcl=NULL, DBType="Main", QCOnly = T, Verbose=F, SuppressPostgreSQLWarnings=T, SuppressOrderWarning=F) {

  ## Base query
  if (QCOnly) {
    
      Query <- paste0("SELECT gps.qc_runs.uuid ",
                      "FROM gps.qc_runs ",
                      "INNER JOIN gps.tracks_days ON gps.tracks_days.uuid = gps.qc_runs.gpstracksdays_uuid ",
                      "INNER JOIN gps.tracks ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                      "WHERE gps.qc_runs.date_local >= '", DateStart, "' and  gps.qc_runs.date_local <= '", DateEnd, "'")

  } else {

      Query <- paste0("SELECT gps.runs.uuid ",
                      "FROM gps.runs ",
                      "INNER JOIN gps.tracks_days ON gps.tracks_days.uuid = gps.runs.gpstracksdays_uuid ",
                      "INNER JOIN gps.tracks ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                      "WHERE gps.runs.date_local >= '", DateStart, "' and  gps.runs.date_local <= '", DateEnd, "'")
      
  }
  
  ## Adding GPS units
  if (!is.null(GpsUnit)) {
    Query <- paste0(Query, " and (")
    for(Index in 1:length(GpsUnit)) {
      if (Index==1) {
        Query <- paste0(Query, "gps.tracks.unit='", GpsUnit[Index], "'")
      } else {
        Query <- paste0(Query, " OR gps.tracks.unit='", GpsUnit[Index], "'")
      }
    }
    Query <- paste0(Query, ")")
  }
  
  ## Adding guide types to include
  if (!is.null(GuideTypesIncl)) {
    Query <- paste0(Query, " and (")
    for(Index in 1:length(GuideTypesIncl)) {
      if (Index==1) {
        Query <- paste0(Query, "gps.tracks_days.guide_type='", GuideTypesIncl[Index], "'")
      } else {
        Query <- paste0(Query, " OR gps.tracks_days.guide_type='", GuideTypesIncl[Index], "'")
      }
    }
    Query <- paste0(Query, ")")
  }
  
  ## Adding guide types to exclude
  if (!is.null(GuideTypesExcl)) {
    Query <- paste0(Query, " and (")
    for(Index in 1:length(GuideTypesExcl)) {
      if (Index==1) {
        Query <- paste0(Query, "gps.tracks_days.guide_type!='", GuideTypesExcl[Index], "'")
      } else {
        Query <- paste0(Query, " AND gps.tracks_days.guide_type!='", GuideTypesExcl[Index], "'")
      }
    }
    Query <- paste0(Query, ")")
  }

  ## Retrieve results from DB
  Results <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

  ## Return dataframe
  return(Results$uuid)
}
