#' Get run use list for a date and GPS unit
#'
#' Get run use list for a date and GPS unit. The name associated with the run is the name of the closest ski line.
#' @param Operation Name of operation.
#' @param Date Date for requested run usage list.
#' @param Unit Unit label for requested run usage list (default=NULL).
#' @param MaxDist_M Maximum distance for proximity calculation in meters.
#' @param WithAltitude Flag whether altitude values should be added. Can have values NA, 'dem' or 'gps'.
#' @param WithTime Flag whether times should be included. Can be true or false (default).
#' @param WithUUID Flag whether UUID of run should be included. Can be true (default) or false.
#' @param DBType Optional parameter to specify which database type to create the run usage list from. Values can be 'Main' (default) or 'NodeJS'. 
#' @param Verbose Switch for printing SQL query. Default value is FALSE.
#' @param SuppressPostgreSQLWarnings Switch for turning warnings off
#' @return Dataframe with list of runs, names of closest ski line and distance to ski line.
#' @export

getRunUsageList <- function(Operation, Date, Unit=NULL, MaxDist_M=1000, WithAltitude=NA, WithTime=F, WithUUID=T, DBType="Main", Verbose=F, SuppressPostgreSQLWarnings=T) {

  if (is.na(WithAltitude) & WithTime==F) {

    QueryStart <- "SELECT gps.tracks_days.date_local, gps.tracks.unit, gps.runs.runnum AS num, gps.runs.uuid, tbl_dist.closest_skiline, tbl_dist.dist_deg "

  } else if (tolower(WithAltitude)=="dem" & WithTime==F) {

    QueryStart <- "SELECT gps.tracks_days.date_local, gps.tracks.unit, gps.runs.runnum AS num, gps.runs.uuid, gps.runs.dem_altitude_max AS alt_max, gps.runs.dem_altitude_min AS alt_min, null AS alt_diff, tbl_dist.closest_skiline, tbl_dist.dist_deg "

  } else if (tolower(WithAltitude)=="gps" & WithTime==F) {

    QueryStart <- "SELECT gps.tracks_days.date_local, gps.tracks.unit, gps.runs.runnum AS num, gps.runs.uuid, gps.runs.gps_altitude_max AS alt_max, gps.runs.gps_altitude_min AS alt_min, null AS alt_diff, tbl_dist.closest_skiline, tbl_dist.dist_deg "

  } else if (is.na(WithAltitude) & WithTime==T) {

    QueryStart <- "SELECT gps.tracks_days.date_local, gps.tracks.unit, gps.runs.runnum AS num, gps.runs.datetime_local_start AS start, gps.runs.datetime_local_end AS end, gps.runs.uuid, tbl_dist.closest_skiline, tbl_dist.dist_deg "

  } else if (tolower(WithAltitude)=="dem" & WithTime==T) {

    QueryStart <- "SELECT gps.tracks_days.date_local, gps.tracks.unit, gps.runs.runnum AS num, gps.runs.datetime_local_start AS start, gps.runs.datetime_local_end AS end, gps.runs.uuid, gps.runs.dem_altitude_max AS alt_max, gps.runs.dem_altitude_min AS alt_min, null AS alt_diff, tbl_dist.closest_skiline, tbl_dist.dist_deg "

  } else if (tolower(WithAltitude)=="gps" & WithTime==T) {

    QueryStart <- "SELECT gps.tracks_days.date_local, gps.tracks.unit, gps.runs.runnum AS num, gps.runs.datetime_local_start AS start, gps.runs.datetime_local_end AS end, gps.runs.uuid, gps.runs.gps_altitude_max AS alt_max, gps.runs.gps_altitude_min AS alt_min, null AS alt_diff, tbl_dist.closest_skiline, tbl_dist.dist_deg "

  } else {

    stop("Incorrect specification of WithAltitude and WithTime parameters.")
  }

  ## Retrieve run use list
  if(!is.null(Unit)){
    Query <- paste0(QueryStart,
                    "FROM gps.tracks ",
                    "INNER JOIN gps.tracks_days ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                    "INNER JOIN gps.runs ON gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid ",
                    "INNER JOIN (SELECT DISTINCT ON (gps.runs.uuid) ",
                                "gps.runs.uuid AS gpsruns_uuid, ",
                                "loccat.skilines.name AS closest_skiline, ",
                                "st_distance(gps.runs.geom, loccat.skilines.geom)::FLOAT AS dist_deg ",
                                "FROM gps.runs ",
                                "LEFT JOIN loccat.skilines ON st_dwithin(gps.runs.geom, loccat.skilines.geom, ", convertMetersToDeg(MaxDist_M), ") ",
                                "ORDER BY gps.runs.uuid, dist_deg) tbl_dist ON tbl_dist.gpsruns_uuid = gps.runs.uuid ",
                    "WHERE gps.tracks_days.date_local='", Date, "' AND gps.tracks.unit='", Unit, "' ",
                    "ORDER BY  gps.tracks_days.date_local, gps.tracks.unit, gps.runs.runnum ")
  }else{
    Query <- paste0(QueryStart,
                    "FROM gps.tracks ",
                    "INNER JOIN gps.tracks_days ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                    "INNER JOIN gps.runs ON gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid ",
                    "INNER JOIN (SELECT DISTINCT ON (gps.runs.uuid) ",
                    "gps.runs.uuid AS gpsruns_uuid, ",
                    "loccat.skilines.name AS closest_skiline, ",
                    "st_distance(gps.runs.geom, loccat.skilines.geom)::FLOAT AS dist_deg ",
                    "FROM gps.runs ",
                    "LEFT JOIN loccat.skilines ON st_dwithin(gps.runs.geom, loccat.skilines.geom, ", convertMetersToDeg(MaxDist_M), ") ",
                    "ORDER BY gps.runs.uuid, dist_deg) tbl_dist ON tbl_dist.gpsruns_uuid = gps.runs.uuid ",
                    "WHERE gps.tracks_days.date_local='", Date, "' ",
                    "ORDER BY  gps.tracks_days.date_local, gps.tracks.unit, gps.runs.runnum ")
  }
  
  ResultDF <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

  ## Add altitude difference column
  if (!is.na(WithAltitude)) {
    ResultDF$alt_diff <- ResultDF$alt_max - ResultDF$alt_min
  }

  ## Convert distance value from deg to m
  ResultDF$dist_m <- round(convertDegToMeters(ResultDF$dist_deg), 0)
  ResultDF <- deleteDFColumnsByName(ResultDF, "dist_deg")
  
  ## Eliminate UUIF is required
  if(!WithUUID) {
    ResultDF <- deleteDFColumnsByName(ResultDF, "uuid")
  }

  return(ResultDF)
}
