#' Gets RunsObject for single or multiple run UUIDs.
#'
#' Gets RunsObject for single or multiple run UUIDs. RunObject object that includes a list with geometries and data frames for runs(line), observation points (points) and pause points (points) for requested runs.
#' @param Operation name of operation
#' @param RunUUIDs Single or array of RunUUIDs
#' @param DBType Optional parameter to specify which database type to get the RunObjects from. Values can be 'Main' (default) or 'NodeJS'.
#' @param WithObsPoints Switch for whether ObsPoints should be included. Default is TRUE.
#' @param WithPausePoints Switch for whether PausePoints should be included. Default is TRUE.
#' @param WithRasterValueArrays Switch for whether RasterValueArray should be included. Default is FALSE.
#' @param LocalProjection Switch whether location catalog should be projected into local UTM grid. Default is T.
#' 
#' @return RunsObject object that includes a list with geometries and data frames for runs(line), observation points (points) and pause points (points) for requested runs.
#'
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "NEH"
#' RunUUIDs <- getRunUUIDsFromTrackerUnitAndDate(Operation, "13", "2015-02-02")
#' Runs <- getRunsObjectFromUUIDs(Operation, RunUUIDs)
#' 
#' @export

getRunsObjectFromUUIDs <- function(Operation, RunUUIDs, DBType="Main", WithObsPoints=T, WithPausePoints=T, WithRasterValueArrays=F, LocalProjection=T) {
  
  ## RETRIEVING INFORMATION
  ## **********************
  
  print(paste0("Retrieving geometries for ", length(RunUUIDs), " run(s):"))
  
  print(" - retrieving run geometries and attributes  ...")
  Runs <- getSpatialObjectsDataFrameFromUUID(Operation, "gps", "runs", UUID=RunUUIDs, DBType=DBType, LocalProjection=LocalProjection)
  
  if(WithPausePoints) {
    print(" - retrieving pause point geometries and attributes ...")
    PausePoints <- getSpatialObjectsDataFrameFromUUID(Operation, "gps", "runs_pausepoints", DBType=DBType, ForeignUUID=RunUUIDs, ForeignUUIDCol="gpsruns_uuid", LocalProjection=LocalProjection)
  } else {
    print(" - pause point geometries and attributes skipped ...")
    PausePoints <- NA
  }

  if(WithObsPoints) {
    print(" - retrieving observation point geometries and attributes  ...")
    ObsPoints <- getSpatialObjectsDataFrameFromUUID(Operation, "gps", "view_obspoints", DBType=DBType, ForeignUUID=RunUUIDs, ForeignUUIDCol="gpsruns_uuid", LocalProjection=LocalProjection)
  } else {
    print(" - observation point geometries and attributes skipped ...")
    ObsPoints <- NA
  }
  
  if(WithRasterValueArrays) {
    print(" - retrieving raster value arrays  ...")
    RasterValueDF <- getRasterValueDfFromRunUUID(Operation, RunUUIDs, DBType=DBType, SuppressProgressMsgs=T)
  } else {
    RasterValueDF <- NA
  }
  
  ## CREATING RUN OBJECT FOR OUTPUT
  ## ******************************
  
  Output <- list(Runs=Runs,
                 PausePoints=PausePoints,
                 ObsPoints=ObsPoints,
                 RasterValueDF=RasterValueDF)
  
  class(Output) <- append(class(Output), "RunsObject")
  
  return(Output)
  
}