#' Gets spatial data frame from database based on list of UUID
#'
#' Conveninece function for directly getting a spatial polygons dataframe from database based on list of UUID
#' @param Operation Name of operation
#' @param Schema Name of schema that contains table with geometry
#' @param DBTbl Name of database table with geometry
#' @param UUID single UUID or array of UUIDs. If NA is provided, the entire table is returned.
#' @param UUIDCol Name of column with UUID in database table
#' @param ForeignUUID Single or array of foreign key UUID values for querying table over ForeignUUIDCol.
#' @param ForeignUUIDCol Name of foreign key UUID column against which the ForeignUUID values are applied.
#' @param GeomCol Name of column with geometry in database table
#' @param AsCentroid Switch for whether returned geometry should be the centroid. Default value is F.
#' @param LocalProjection Switch for whether geometry should be transformed into local UTM coordinate system. Default value is FALSE.
#' @param WithoutDeleted Switch to eleiminate locations that are considered deleted. Default is False.
#' @param ResultCol Names of columns included in results table (array of names). uuid column is added automatically.
#' @param DBType Optional parameter for specifying which database to get the spatial object from
#' @param Verbose Switch for printing of query
#' @param SuppressPostgreSQLWarnings Switch for supressing warning messages from postgresqlExecStatement. Turn on for debugging!
#' @return Appropriate sp spatial objects (SpatialPoints, SpatialLines, SpatialPolygons)
#' @export

getSpatialObjectsDataFrameFromUUID <- function(Operation, Schema, DBTbl, DBType="Main", UUID=NA, UUIDCol="uuid", ForeignUUID=NA, ForeignUUIDCol=NA, GeomCol="geom", AsCentroid=F, LocalProjection=F, WithoutDeleted=F, ResultCol=NA, Verbose=F, SuppressPostgreSQLWarnings=T) {

  ## Retrieve spatial objects
  SpatialObjects <- getSpatialObjectsFromUUID(Operation=Operation, Schema=Schema, DBTbl=DBTbl, DBType=DBType, UUID=UUID, UUIDCol=UUIDCol, ForeignUUID=ForeignUUID, ForeignUUIDCol=ForeignUUIDCol, GeomCol=GeomCol, AsCentroid=AsCentroid, LocalProjection=LocalProjection, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

  if (!is.null(SpatialObjects)) {
  
    ## Retrieve attributes
    AttributesDF <- getRecordsFromUUID(Operation=Operation, Schema=Schema, DBTbl=DBTbl, DBType=DBType, UUID=UUID, UUIDCol=UUIDCol, ForeignUUID=ForeignUUID, ForeignUUIDCol=ForeignUUIDCol, GeomCol=GeomCol, ResultCol=ResultCol, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings, SuppressOrderWarning=T)
    # Order warning is suppressed because it does not matter in this application.
  
    ## Create spatial objects dataframe
    if (class(SpatialObjects)=="SpatialPoints") {
      SPDF <- SpatialPointsDataFrame(SpatialObjects, AttributesDF)
    } else if (class(SpatialObjects)=="SpatialLines") {
      SPDF <- SpatialLinesDataFrame(SpatialObjects, AttributesDF)
    } else if (class(SpatialObjects)=="SpatialPolygons") {
      SPDF <- SpatialPolygonsDataFrame(SpatialObjects, AttributesDF)
    } else {
      SPDF <- NULL
      warning(paste0("Class '", class(SpatialObjects), "' not supported."), immediate. = T)
    }
    
  } else {
    
    SPDF <- NULL
    
  }
  
  ## Eliminate deleted records
  if(!is.null(SPDF) & WithoutDeleted) {
    
    SPDF <- tryCatch(
      {SPDF <- SPDF[SPDF$flag_delete != "Yes",]},
      error=function(cond) {
        #message("Comparison results in error message --> likely empty spatial dataframe!")
        return(NULL)
      }
    )
    
  }

  ## Return SPDF
  return(SPDF)
}
