#' Retrieves spatial objects from PostgreSQl/PostGIS GPS database.
#'
#' Retrieves spatial objects from PostgreSQl/PostGIS GPS database and turns them into sp spatial objects.
#' @param Operation Name of operation.
#' @param Schema Name of schema that contains table with geometry.
#' @param DBTbl Name of database table with geometry.
#' @param DBType DBType Type of database to create connection with. Can be 'Main' (default) or 'NodeJS' for database that feeds online viewer.
#' @param UUID single UUID or array of UUIDs. If NA is provided, the entire table is returned.
#' @param UUIDCol Name of column with UUID in database table.
#' @param ForeignUUID Single or array of foreign key UUID values for querying table over ForeignUUIDCol.
#' @param ForeignUUIDCol Name of foreign key UUID column against which the ForeignUUID values are applied.
#' @param GeomCol Name of column with geometry in database table.
#' @param AsCentroid Switch for whether returned geometry should be the centroid. Default value is F.
#' @param LocalProjection Switch for whether geometry should be transformed into local UTM coordinate system. Default value is FALSE.
#' @param Verbose Switch for printing of query. Default value is FALSE.
#' @param SuppressPostgreSQLWarnings Switch for supressing warning messages from postgresqlExecStatement. Default value is TRUE. Turn on for debugging!
#' @return Appropriate sp spatial objects (SpatialPoints, SpatialLines, SpatialPolygons)
#' @export

getSpatialObjectsFromUUID <- function(Operation, Schema, DBTbl, DBType="Main", UUID=NA, UUIDCol="uuid", ForeignUUID=NA, ForeignUUIDCol=NA, GeomCol="geom", AsCentroid=F, LocalProjection=F, Verbose=F, SuppressPostgreSQLWarnings=T) {

  ## Build query - Base
  if (AsCentroid) {
    if (is.na(ForeignUUIDCol)) {
      if (Schema=="") {
        Query <- paste0("SELECT ", UUIDCol, ", ST_AsText(ST_Centroid(", GeomCol, ")) AS wkt, ST_SRID(", GeomCol, ") AS srid FROM \"", DBTbl, "\"")
      } else {
        Query <- paste0("SELECT ", UUIDCol, ", ST_AsText(ST_Centroid(", GeomCol, ")) AS wkt, ST_SRID(", GeomCol, ") AS srid FROM \"", Schema, "\".\"", DBTbl, "\"")
      }
    } else {
      if (Schema=="") {
        Query <- paste0("SELECT ", UUIDCol, ", ", ForeignUUIDCol, ", ST_AsText(ST_Centroid(", GeomCol, ")) AS wkt, ST_SRID(", GeomCol, ") AS srid FROM \"", DBTbl, "\"")
      } else {
        Query <- paste0("SELECT ", UUIDCol, ", ", ForeignUUIDCol, ", ST_AsText(ST_Centroid(", GeomCol, ")) AS wkt, ST_SRID(", GeomCol, ") AS srid FROM \"", Schema, "\".\"", DBTbl, "\"")
      }
    }
  } else {  
    if (is.na(ForeignUUIDCol)) {
      if (Schema=="") {
        Query <- paste0("SELECT ", UUIDCol, ", ST_AsText(", GeomCol, ") AS wkt, ST_SRID(", GeomCol, ") AS srid FROM \"", DBTbl, "\"")
      } else {
        Query <- paste0("SELECT ", UUIDCol, ", ST_AsText(", GeomCol, ") AS wkt, ST_SRID(", GeomCol, ") AS srid FROM \"", Schema, "\".\"", DBTbl, "\"")
      }
    } else {
      if (Schema=="") {
        Query <- paste0("SELECT ", UUIDCol, ", ", ForeignUUIDCol, ", ST_AsText(", GeomCol, ") AS wkt, ST_SRID(", GeomCol, ") AS srid FROM \"", DBTbl, "\"")
      } else {
        Query <- paste0("SELECT ", UUIDCol, ", ", ForeignUUIDCol, ", ST_AsText(", GeomCol, ") AS wkt, ST_SRID(", GeomCol, ") AS srid FROM \"", Schema, "\".\"", DBTbl, "\"")
      }
    }
  }
  
  ## Add where if necessary
  if ((!is.na(UUID[1])) | (!is.na(ForeignUUID[1]))) {
    Query <- paste0(Query, " WHERE ")
  }

  ## Build query - UUID selection
  if(length(UUID)==1) {
    if (!is.na(UUID)) {
      Query <- paste0(Query, "(\"", UUIDCol, "\"=", convertStringToSQL(UUID), ")")
    }
  } else {
    Query <- paste0(Query, "(")
    for (Index_UUID in 1:length(UUID)) {
      Query <- paste0(Query, "\"", UUIDCol, "\"=", convertStringToSQL(UUID[Index_UUID]), " OR ")
    }
    Query <- paste0(substr(Query, 1, nchar(Query)-4), ")")
    rm(Index_UUID)
  }
  
  ## Add AND if necessary
  if ((!is.na(UUID[1])) & (!is.na(ForeignUUID[1]))) {
    Query <- paste0(Query, " AND ")
  }

  ## Building query - ForeignUUIDCol values
  if(length(ForeignUUID)==1) {
    if (!is.na(ForeignUUID)) {
      if (typeof(ForeignUUID)=="character") {Query <- paste0(Query, "(\"", ForeignUUIDCol, "\"=", convertStringToSQL(ForeignUUID), ")")}
      else if (typeof(ForeignUUID)=="double") {Query <- paste0(Query, "(\"", ForeignUUIDCol, "\"=", convertNumericToSQL(ForeignUUID), ")")}
      else {stop(paste0("Data type '", typeof(ForeignUUID), "' not supported!"))}
    }
  } else {
    Query <- paste0(Query, "(")
    for (Index_Value in 1:length(ForeignUUID)) {
      if (typeof(ForeignUUID)=="character") {Query <- paste0(Query, "\"", ForeignUUIDCol, "\"=", convertStringToSQL(ForeignUUID[Index_Value]), " OR ")}
      else if (typeof(ForeignUUID)=="double") {Query <- paste0(Query, "\"", ForeignUUIDCol, "\"=", convertNumericToSQL(ForeignUUID[Index_Value]), " OR ")}
      else {stop(paste0("Data type '", typeof(ForeignUUID), "' not supported!"))}
    }
    Query <- paste0(substr(Query, 1, nchar(Query)-4), ")")
    rm(Index_Value)
  }

  ## Send query to database
  ResultDF <- getRecordsFromQuery(Operation=Operation, Query=Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

  if(nrow(ResultDF)>0) {
  
    ## Create spatial objects
    for (Index_Rows in 1:nrow(ResultDF)) {
  
      ## Convert WKT to spatial object (requires rgeos package)
      SpatialObject <- rgeos::readWKT(ResultDF$wkt[Index_Rows], id=ResultDF$uuid[Index_Rows])
  
      ## Add projection information if SRID=4617
      if (ResultDF$srid[Index_Rows]==Const_GlobalCrsSrid) {
        proj4string(SpatialObject) <- Const_GlobalCrsString
      }
  
      ## Combining spatial objects (requires maptools package)
      if (Index_Rows==1) {
        SpatialObjects <- SpatialObject
      } else {
        SpatialObjects <- maptools::spRbind(SpatialObjects, SpatialObject)
      }
    }
    rm(Index_Rows)
    
    ## Transformation into local coordinate system
    if (LocalProjection) {
      SpatialObjects <- spTransform(SpatialObjects, getLocalSRDI(Operation)$proj4text)
    }
    
  } else {
    
    SpatialObjects <- NULL
    
  }
  
  ## Return final result
  return (SpatialObjects)

}
