#' Gets UUIDs from names
#'
#' Gets UUID from single or array of names
#' @param Operation Name of operation
#' @param Schema Name of schema that contains DBTbl
#' @param DBTbl Name of database table with geometry
#' @param DBType Type of database to query from. Can be 'Main' (default) or 'NodeJS' for database that feeds online viewer.
#' @param Names Single or array of names
#' @param Verbose Switch for printing of query.
#' @param SuppressWarnings Switch for whether warnings should be suppressed. Default is FALSE.
#' @return Single UUID or array of UUID
#' @export
#' 
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Oper <- "CMHGL"
#' SkirunNames <- c("North Goat", "Alexis")
#' 
#' SkirunUUIDs <- getUUIDsFromNames(Operation=Oper, Schema="loccat", DbTbl="skiruns", Names=SKirunNames)


getUUIDsFromNames <- function (Operation, Schema, DBTbl, DBType="Main", Names, IncludeDeleted=F, Verbose=F, SuppressWarnings=F) {

  uuids <- character(0)
  
  for (i in 1:length(Names)) {
    
    Name <- Names[i]
    
    if(IncludeDeleted) {
      Query <- paste0("SELECT uuid FROM ", Schema, ".", DBTbl, " WHERE name='", Name, "'")
    } else {
      Query <- paste0("SELECT uuid FROM ", Schema, ".", DBTbl, " WHERE name='", Name, "' AND flag_delete='No'")
    }
    
    Result <- getRecordsFromQuery(Operation, Query)
    
    if (nrow(Result)==0) {
      if(!SuppressWarnings) {warning(paste0("No uuid available for '", Name, "' in ", Schema, ".", DBTbl, "!"), immediate. = T)}
    } else if (nrow(Result)>1) {
      if(!SuppressWarnings) {warning(paste0(nrow(Result), " uuids available for '", Name, "' in ", Schema, ".", DBTbl, "!"), immediate. = T)}
      uuids <- c(uuids, Result$uuid)
    } else {
      uuids <- c(uuids, Result$uuid)
    }
    
    rm(Query, Result)
    
  }
  
  if(is.na(nchar(uuids[1]))) {
    uuids <- NA
    if(!SuppressWarnings) {warning(paste0("No objects with these name(s) found in ", Schema, ".", DBTbl, "!"), immediate. = T)}
  }
  
  return(uuids)
  
}
