#' Import contour lines for elevation band boundaries from shp file and write into PostgreSQL/PostGIS GPS database.
#'
#' Import contour lines for elevation band boundaries from shp file and write into PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param User Name of user. Required for getting correct local folder path to contour shapefile.
#' @param DBType Optional parameter for specifying which database type the cat roads should be imported to. Values can be 'Main' (default) or 'NodeJS'.
#' @param FileName Name of shape file with contour lines. Defailt avalue is 'contour'
#' @param ElevBTL2TL Elevation of below treeline - treeline elevation band boundary. If NA (default), values is extracted from database.
#' @param ElevTL2ALP Elevation of treeline - alpine elevation band boundary. If NA (default), values is extracted from database.
#' @param Overwrite Boolean flag for whether the entire catroad table should be overwritten (i.e., all records deleted first). Default is F.
#'
#' @examples
#' require(SarpGPSToolsPrivate)
#' require(SarpGPSTools)
#' 
#' Operation <- "MPS"
#' User <- "SarpLab"
#' 
#' importElevBandBoundariesFromContourShp(Operation, User)
#' 
#' @export

importElevBandBoundariesFromContourShp <- function(Operation, User, DBType="Main", FileName="contour", ElevBTL2TL=NA, ElevTL2ALP=NA, Overwrite=F) {
  
  ## Import
  print("Importing contour lines ...")
  ContourFolder <- paste0(getLocalRootFolderPath(Operation, User), "DEM\\cdsm_elevation_contour_", Operation)
  SPDF_Contours <- readOGR(ContourFolder, FileName)
  
  ## Getting elevations for elevation band boundaries if not provided
  if(is.na(ElevBTL2TL)) {
    ElevBTL2TL <- getElevBandBoundary(Operation, "TL", "Bottom")
  }
  if(is.na(ElevTL2ALP)) {
    ElevTL2ALP <- getElevBandBoundary(Operation, "TL", "Top")
  }
  
  if(is.na(ElevBTL2TL)|is.na(ElevTL2ALP)) {
    stop("Elevations for elevation band boundaries not provided in function or available from database!")
  }
  
  ## Filtering
  print("Filtering contour lines ...")
  SPDF_ElevBand <- SPDF_Contours[SPDF_Contours@data$ELEV==ElevBTL2TL|SPDF_Contours@data$ELEV==ElevTL2ALP,]
  
  ## Getting data frame ready for import
  print("Getting contourlines ready for import ...")
  SPDF_ElevBand@data$boundary <- ifelse(SPDF_ElevBand@data$ELEV==ElevBTL2TL, "BTL-TL", "TL-ALP")
  SPDF_ElevBand@data$uuid <- sapply(SPDF_ElevBand@data$boundary, function(x) createUuid())
  
  ## Changing IDs
  SPDF_ElevBand_SPL <- SPDF_ElevBand@lines
  
  for (Index in 1:nrow(SPDF_ElevBand)) {
    SPDF_ElevBand_SPL[[Index]]@ID <- SPDF_ElevBand@data$uuid[SPDF_ElevBand@data$ID==SPDF_ElevBand_SPL[[Index]]@ID]
  }
  
  SPDF_ElevBand_DF <- SPDF_ElevBand@data
  SPDF_ElevBand_DF <- deleteDFColumnsByName(SPDF_ElevBand_DF, "ID")
  row.names(SPDF_ElevBand_DF) <- SPDF_ElevBand_DF$uuid
  names(SPDF_ElevBand_DF) <- c("elevation", "boundary", "uuid")
  
  SPDF_ElevBand_New <- SpatialLinesDataFrame(SpatialLines(SPDF_ElevBand_SPL), SPDF_ElevBand_DF)
  
  ## Writing to database
  print("Writing countour lines into GPS database ...")
  writeSPDFToGPSDB(SPDF_ElevBand_New, Operation, "dem", "elevbands_boundaries", DBType=DBType, SRID=Const_GlobalCrsSrid, Overwrite=Overwrite)
  
}