#' Retrieve avalanche observations from InfoEx and write into PostgreSQL/PostGIS GPS database.
#'
#' Retrieve avalanche observations from InfoEx and write into PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS
#' @param DateStart Start date in YYYY-MM-DD format. Default value is NA. 
#' @param DateEnd End date in YYYY-MM-DD format. Default value is DateStart.
#' @param NewRecordsOnly Boolean flag for importing new records only. Default is T.
#' @param SkipMissingLoc Switch for whether records of missing locations should be skipped or produce an error. Default is F.
#' @param Overwrite Boolean flag for whether the entire avobs table should be overwritten (i.e., all records deleted first). Default is F.

#' @export

importInfoExAvObs <- function(Operation, DBType="Main", DateStart=NA, DateEnd=DateStart, NewRecordsOnly=T, SkipMissingLoc=F, Overwrite=F, Verbose=F, SuppressPostgreSQLWarnings=T) {
  
  ## Retrieve information from InfoEx
  require(SarpInfoExToolsPrivate)
  require(SarpInfoExTools)
  
  AvObs <- SarpInfoExTools::getInfoExAvObs(Operation, DateStart=DateStart, DateEnd=DateEnd, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
  
  detach("package:SarpInfoExToolsPrivate", unload=TRUE)
  detach("package:SarpInfoExTools", unload=TRUE)
  
  ## Processing
  if(nrow(AvObs$AvObs)> 0) {
    
    print("Processing records for importing into GPS database ...")
    
    ## Eliminate records that already exist in DB
    if(NewRecordsOnly & !Overwrite) {
      AvObs$AvObs <- returnNewExternalRecordsOnly(AvObs$AvObs, ImportRelevantId="uuid", ExistRelevantId="infoex_uuid", Operation=Operation, Schema="infoex", DBTbl="avobs", DBType=DBType)
    } else {
      print(paste0("Number of av obs retrieved from InfoEx: ", nrow(AvObs$AvObs)))
    }
    
    if(nrow(AvObs$AvObs)> 0) {
    
      ## Av observations
      ## ***************
      
      ## Change column names
      colnames(AvObs$AvObs)[which(colnames(AvObs$AvObs)=="ob_date")] <- "date"
      colnames(AvObs$AvObs)[which(colnames(AvObs$AvObs)=="comments")] <- "comment"
      colnames(AvObs$AvObs)[which(colnames(AvObs$AvObs)=="min_size")] <- "size_avg"
      colnames(AvObs$AvObs)[which(colnames(AvObs$AvObs)=="max_size")] <- "size_max"
      colnames(AvObs$AvObs)[which(colnames(AvObs$AvObs)=="grain_form_min")] <- "grain_form_avg"
       
      ## Create new columns
      AvObs$AvObs$infoex_uuid <- AvObs$AvObs$uuid
      AvObs$AvObs$source <- "InfoEx"
      AvObs$AvObs$date_esri <- AvObs$AvObs$date
      AvObs$AvObs$datetime <- NA
      for (Index_Record in 1:nrow(AvObs$AvObs)) {
        if(!is.na(AvObs$AvObs$ob_time[Index_Record])) {AvObs$AvObs$datetime[Index_Record] <- paste(AvObs$AvObs$date[Index_Record], AvObs$AvObs$ob_time[Index_Record])}
      }
       
      ## Remove unnecessary columns as needed
      AvObs$AvObs <- deleteDFColumnsByName(AvObs$AvObs, "state")
      AvObs$AvObs <- deleteDFColumnsByName(AvObs$AvObs, "caaml_group_id")
      AvObs$AvObs <- deleteDFColumnsByName(AvObs$AvObs, "create_user_uuid")
      AvObs$AvObs <- deleteDFColumnsByName(AvObs$AvObs, "operation_uuid")
      AvObs$AvObs <- deleteDFColumnsByName(AvObs$AvObs, "review_user_uuid")
      AvObs$AvObs <- deleteDFColumnsByName(AvObs$AvObs, "submit_user_uuid")
      AvObs$AvObs <- deleteDFColumnsByName(AvObs$AvObs, "workflow_execution_id")
      AvObs$AvObs <- deleteDFColumnsByName(AvObs$AvObs, "share_level")
      AvObs$AvObs <- deleteDFColumnsByName(AvObs$AvObs, "exact_locations")
      AvObs$AvObs <- deleteDFColumnsByName(AvObs$AvObs, "ob_interval")
      AvObs$AvObs <- deleteDFColumnsByName(AvObs$AvObs, "ob_time")
      AvObs$AvObs <- deleteDFColumnsByName(AvObs$AvObs, "reported_date")
      AvObs$AvObs <- deleteDFColumnsByName(AvObs$AvObs, "route_control")
  
        
      ## LOCATION REFERENCES
      ## *******************
      
      ## Delete location references that are not relevant anymore
      AvObs$Loc <- AvObs$Loc[(AvObs$Loc$observation_uuid %in% AvObs$AvObs$infoex_uuid),]
      
      ## Change column names
      colnames(AvObs$Loc)[which(colnames(AvObs$Loc)=="observation_uuid")] <- "infoexavobs_uuid"
      colnames(AvObs$Loc)[which(colnames(AvObs$Loc)=="location_uuid")] <- "infoexlocation_uuid"
      
      ## Getting uuids from all Infoex locations in loccat
      AllInfoExLoc <- getRecordsFromQuery(Operation, "Select * from loccat.all_infoex_locations", DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
      AllInfoExLoc <- na.omit(AllInfoExLoc)
      
      ## Update LocationTable
      AvObs$Loc$uuid <- NA
      for (Index_Loc in 1: nrow(AvObs$Loc)) {
        AvObs$Loc$uuid[Index_Loc] <- createUuid()
      }
       
      AvObs$Loc$loccat_uuid <- NA
      Unique_InfoExLocation_uuid <- unique(AvObs$Loc$infoexlocation_uuid)
      for (Index_UniqueLoc in 1:length(Unique_InfoExLocation_uuid)) {
        loccat_uuid <- AllInfoExLoc$uuid[AllInfoExLoc$infoex_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]]
        loccat_uuid <- loccat_uuid[!is.na(loccat_uuid)]
        if (length(loccat_uuid)==1) {
          AvObs$Loc$loccat_uuid[AvObs$Loc$infoexlocation_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]] <- loccat_uuid
        }
      }
      
      ## Check for missing locations
      MissingLoc <- AvObs$Loc[is.na(AvObs$Loc$loccat_uuid),]
      
      if (nrow(MissingLoc)>0) {
        if(SkipMissingLoc) {
          AvObs$Loc <- AvObs$Loc[!is.na(AvObs$Loc$loccat_uuid),]
          NumObsSkipped <- nrow(AvObs$AvObs[!(AvObs$AvObs$uuid %in% AvObs$Loc$infoexavobs_uuid),])
          AvObs$AvObs <- AvObs$AvObs[(AvObs$AvObs$uuid %in% AvObs$Loc$infoexavobs_uuid),]
          print(paste("Skipping", NumObsSkipped, "records because of missing location information in target database!"))
        } else {
          Error_MissingLoc <<- MissingLoc
          MissingLocCount <- length(unique(MissingLoc$infoexlocation_uuid))
          stop(paste(MissingLocCount, "missing locations for observations!! See data frame Error_MissingLoc"))
        }
      }
  
      ## WRITING TABLE TO GPSDB
      ## **********************
      
      ## AvObs <<- AvObs
      
      if (nrow(AvObs$AvObs)>0) {
      
        if (Overwrite) {
          print("Deleting existing records ...")
          sendQueryToGPSDB(Operation, "DELETE FROM infoex.avobs_loc_x", DBType=DBType)
          sendQueryToGPSDB(Operation, "DELETE FROM infoex.avobs", DBType=DBType)
        }
         
          writeDFToGPSDB(DF=AvObs$AvObs, Operation, "infoex", "avobs", DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "av obs")
          writeDFToGPSDB(DF=AvObs$Loc, Operation, "infoex", "avobs_loc_x", DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "av obs locations")
  
      } else {
        
        warning("No avalanche observations to import!", immediate. = T)
        
      }
      
    } else {
      
      warning("No avalanche observations to import!", immediate. = T)
      
    }
        
  } else {
    
    warning("No avalanche observations to import!", immediate. = T)
    
  }
  
  print("")
  
}  