#' Retrieve field observations from InfoEx and write into PostgreSQL/PostGIS GPS database.
#'
#' Retrieve field observations from InfoEx and write into PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS
#' @param DateStart Start date in YYYY-MM-DD format. Default value is NA. 
#' @param DateEnd End date in YYYY-MM-DD format. Default value is DateStart.
#' @param NewRecordsOnly Boolean flag for importing new records only. Default is T.
#' @param SkipMissingLoc Switch for whether records of missing locations should be skipped or produce an error. Default is F.
#' @param Overwrite Boolean flag for whether the entire avobs table should be overwritten (i.e., all records deleted first). Default is F.

#' @export

importInfoExFieldObs <- function(Operation, DBType="Main", DateStart=NA, DateEnd=DateStart, NewRecordsOnly=T, SkipMissingLoc=F, Overwrite=F, Verbose=F, SuppressPostgreSQLWarnings=T) {
  
  ## Retrieve information from InfoEx
  require(SarpInfoExToolsPrivate)
  require(SarpInfoExTools)
  
  FieldObs <- SarpInfoExTools::getInfoExFieldObs(Operation, DateStart=DateStart, DateEnd=DateEnd, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
  
  detach("package:SarpInfoExToolsPrivate", unload=TRUE)
  detach("package:SarpInfoExTools", unload=TRUE)
  

  ## Processing
  if(nrow(FieldObs$FieldObs)> 0) {
    
    print("Processing records for importing into GPS database ...")
    
    ## Field observations
    ## ******************
    
    ## Eliminate records that already exist in DB
    if(NewRecordsOnly & !Overwrite) {
      FieldObs$FieldObs <- returnNewExternalRecordsOnly(FieldObs$FieldObs, ImportRelevantId="uuid", ExistRelevantId="infoex_uuid", Operation=Operation, Schema="infoex", DBTbl="fieldobs", DBType=DBType)
    } else {
      print(paste0("Number of field obs retrieved from InfoEx: ", nrow(FieldObs$FieldObs)))
    }
    
    if(nrow(FieldObs$FieldObs)> 0) {
    
      ## Change column names
      colnames(FieldObs$FieldObs)[which(colnames(FieldObs$FieldObs)=="ob_date")] <- "date"
      colnames(FieldObs$FieldObs)[which(colnames(FieldObs$FieldObs)=="we")] <- "hsw"
      colnames(FieldObs$FieldObs)[which(colnames(FieldObs$FieldObs)=="comments")] <- "comment"
      
      ## Create new columns
      FieldObs$FieldObs$infoex_uuid <- FieldObs$FieldObs$uuid
      FieldObs$FieldObs$source <- "InfoEx"
      FieldObs$FieldObs$date_esri <- FieldObs$FieldObs$date
      FieldObs$FieldObs$datetime_start <- ifelse(is.na(FieldObs$FieldObs$ob_start_time), NA, paste(FieldObs$FieldObs$date, FieldObs$FieldObs$ob_start_time))
      FieldObs$FieldObs$datetime_end <- ifelse(is.na(FieldObs$FieldObs$ob_end_time), NA, paste(FieldObs$FieldObs$date, FieldObs$FieldObs$ob_end_time))
      
      ## Remove unnecessary columns as needed
      FieldObs$FieldObs <- deleteDFColumnsByName(FieldObs$FieldObs, "ob_start_time")
      FieldObs$FieldObs <- deleteDFColumnsByName(FieldObs$FieldObs, "ob_end_time")
      FieldObs$FieldObs <- deleteDFColumnsByName(FieldObs$FieldObs, "state")
      FieldObs$FieldObs <- deleteDFColumnsByName(FieldObs$FieldObs, "caaml_group_id")
      FieldObs$FieldObs <- deleteDFColumnsByName(FieldObs$FieldObs, "create_user_uuid")
      FieldObs$FieldObs <- deleteDFColumnsByName(FieldObs$FieldObs, "operation_uuid")
      FieldObs$FieldObs <- deleteDFColumnsByName(FieldObs$FieldObs, "review_user_uuid")
      FieldObs$FieldObs <- deleteDFColumnsByName(FieldObs$FieldObs, "submit_user_uuid")
      FieldObs$FieldObs <- deleteDFColumnsByName(FieldObs$FieldObs, "workflow_execution_id")
      FieldObs$FieldObs <- deleteDFColumnsByName(FieldObs$FieldObs, "share_level")
      
    
      ## LOCATION REFERENCES
      ## *******************
      
      ## Delete location references that are not relevant anymore
      FieldObs$Loc <- FieldObs$Loc[(FieldObs$Loc$observation_uuid %in% FieldObs$FieldObs$infoex_uuid),]
      
      ## Change column names
      colnames(FieldObs$Loc)[which(colnames(FieldObs$Loc)=="observation_uuid")] <- "infoexfieldobs_uuid"
      colnames(FieldObs$Loc)[which(colnames(FieldObs$Loc)=="location_uuid")] <- "infoexlocation_uuid"
      
      ## Getting uuids from all Infoex locations in loccat
      AllInfoExLoc <- getRecordsFromQuery(Operation, "Select * from loccat.all_infoex_locations", DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
      AllInfoExLoc <- na.omit(AllInfoExLoc)
      
      ## Update LocationTable
      FieldObs$Loc$uuid <- sapply(FieldObs$Loc$infoexfieldobs_uuid, function(x) createUuid())
       
      FieldObs$Loc$loccat_uuid <- NA
      Unique_InfoExLocation_uuid <- unique(FieldObs$Loc$infoexlocation_uuid)
      for (Index_UniqueLoc in 1:length(Unique_InfoExLocation_uuid)) {
        loccat_uuid <- AllInfoExLoc$uuid[AllInfoExLoc$infoex_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]]
        loccat_uuid <- loccat_uuid[!is.na(loccat_uuid)]
        if (length(loccat_uuid)==1) {
          FieldObs$Loc$loccat_uuid[FieldObs$Loc$infoexlocation_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]] <- loccat_uuid
        }
      }
      
      ## Check for missing locations
      MissingLoc <- FieldObs$Loc[is.na(FieldObs$Loc$loccat_uuid),]
      
      if (nrow(MissingLoc)>0) {
        if(SkipMissingLoc) {
          FieldObs$Loc <- FieldObs$Loc[!is.na(FieldObs$Loc$loccat_uuid),]
          NumObsSkipped <- nrow(FieldObs$FieldObs[!(FieldObs$FieldObs$uuid %in% FieldObs$Loc$infoexfieldobs_uuid),])
          FieldObs$FieldObs <- FieldObs$FieldObs[(FieldObs$FieldObs$uuid %in% FieldObs$Loc$infoexfieldobs_uuid),]
          print(paste("Skipping", NumObsSkipped, "records because of missing location information in target database!"))
        } else {
          Error_MissingLoc <<- MissingLoc
          MissingLocCount <- length(unique(MissingLoc$infoexlocation_uuid))
          stop(paste(MissingLocCount, "missing locations for observations!! See data frame Error_MissingLoc"))
        }
      }
      
  
      ## WRITING TABLE TO GPSDB
      ## **********************
      
      ## FieldObs <<- FieldObs
      
      if(nrow(FieldObs$FieldObs)>0) {
      
        if (Overwrite) {
          print("Deleting existing records ...")
          sendQueryToGPSDB(Operation, "DELETE FROM infoex.fieldobs_loc_x", DBType=DBType)
          sendQueryToGPSDB(Operation, "DELETE FROM infoex.fieldobs", DBType=DBType)
        }
        
         writeDFToGPSDB(DF=FieldObs$FieldObs, Operation, "infoex", "fieldobs", DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "field obs")
         writeDFToGPSDB(DF=FieldObs$Loc, Operation, "infoex", "fieldobs_loc_x", DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "field obs locations")
  
      } else {
        
        warning("No field observation summaries to import!", immediate. = T)
        
      }
  
    } else {
      
      warning("No field observation summaries to import!", immediate. = T)
      
    }
           
  } else {
    
    warning("No field observation summaries to import!", immediate. = T)
    
  }
  
  print("")
  
}  