  #' Retrieve freeform text records from InfoEx and write into PostgreSQL/PostGIS GPS database.
#'
#' Retrieve freeform text records from InfoEx and write into PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param TargetSchema Name of schema that includes target table
#' @param TargetTbl name of target table
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS'.
#' @param Name Name of free form text content that should be extracted. Default value is NA.
#' @param DateStart Start date in YYYY-MM-DD format. Default value is NA. 
#' @param DateEnd End date in YYYY-MM-DD format. Default value is DateStart.
#' @param NewRecordsOnly Boolean flag for importing new records only. Default is T.
#' @param SkipMissingLoc Switch for whether records of missing locations should be skipped or produce an error. Default is F.
#' @param Overwrite Boolean flag for whether the entire wxobs table should be overwritten (i.e., all records deleted first). Default is F.
#' @export

importInfoExFreeFromText<- function(Operation, TargetSchema, TargetTbl, DBType="Main", Name=NA, DateStart=NA, DateEnd=DateStart, NewRecordsOnly=T, SkipMissingLoc=F, Overwrite=F, UserConfirm=T, Verbose=F, SuppressPostgreSQLWarnings=T) {
  
  warning("Need to expend functionality to only import new records!", immediate. = T)
  
  ## Retrieve information from InfoEx
  require(SarpInfoExToolsPrivate)
  require(SarpInfoExTools)
  
  Result <- getInfoExFreeFormText(Operation, Name=Name, DateStart=DateStart, DateEnd=DateEnd, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
  
  detach("package:SarpInfoExToolsPrivate", unload=TRUE)
  detach("package:SarpInfoExTools", unload=TRUE)
  
  Tbl_FFT <- Result$FreeFormText
  
  if(nrow(Tbl_FFT)>0) {
    
    print("Processing records for importing into GPS database ...")
    
    ## Eliminate records that already exist in DB
    if(NewRecordsOnly & !Overwrite) {
      Tbl_FFT <- returnNewExternalRecordsOnly(Tbl_FFT, ImportRelevantId="uuid", ExistRelevantId="infoexfreeform_uuid", Operation=Operation, Schema=TargetSchema, DBTbl=TargetTbl, DBType=DBType)
    } else {
      print(paste0("Number of records retrieved from InfoEx: ", nrow(Tbl_FFT)))
    }
    
    if(nrow(Tbl_FFT)>0) {
    
      ## FREE FORM TEXT RECORDS
      ## **********************
      
      ## Change column names
      colnames(Tbl_FFT)[which(colnames(Tbl_FFT)=="uuid")] <- "infoexfreeform_uuid"
      
      colnames(Tbl_FFT)[which(colnames(Tbl_FFT)=="assessment_date")] <- "date"
      colnames(Tbl_FFT)[which(colnames(Tbl_FFT)=="assessment_time")] <- "time"
      
      colnames(Tbl_FFT)[which(colnames(Tbl_FFT)=="freeform_name")] <- "infoexfreeform_name"
      colnames(Tbl_FFT)[which(colnames(Tbl_FFT)=="freeform_content")] <- "infoexfreeform_content"
  
      ## Create new columns
      Tbl_FFT$uuid <- NA
      for (Index in 1:nrow(Tbl_FFT)) {Tbl_FFT$uuid[Index] <- createUuid()}
      Tbl_FFT$date_esri <- Tbl_FFT$date
      Tbl_FFT$datetime <- paste(Tbl_FFT$date, Tbl_FFT$time)
      Tbl_FFT$source <- "InfoEx"
      
      ## Remove unnecessary columns as needed
      Tbl_FFT <- deleteDFColumnsByName(Tbl_FFT, "submitted_on")
      Tbl_FFT <- deleteDFColumnsByName(Tbl_FFT, "location_group_uuid")
      Tbl_FFT <- deleteDFColumnsByName(Tbl_FFT, "time")
      
      
      ## LOCATION REFERENCES
      ## *******************
      
      ## Getting uuids from all Infoex locations in loccat
      AllInfoExLoc <- getRecordsFromQuery(Operation, "Select * from loccat.all_infoex_locations", DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
      AllInfoExLoc <- na.omit(AllInfoExLoc)
      
      ## Create LocationTable
      Tbl_Loc <- data.frame(uuid = character(),
                            FFTRecord_uuid = character(),
                            loccat_uuid = character(),
                            infoexlocation_uuid = character())
      
      ## Looping through FFT records
      for (Index_FFT in 1:nrow(Tbl_FFT)) {
        
        FFT_uuid <- Tbl_FFT$uuid[Index_FFT]
        
        Tbl_Loc_SingleFFTRecord <- Result$Loc[Result$Loc$workflow_execution_id==Tbl_FFT$workflow_execution_id[Index_FFT],]
        
        for (Index_Loc in 1:nrow(Tbl_Loc_SingleFFTRecord)) {
          
          Tbl_Loc <- rbind(Tbl_Loc, 
                           data.frame(uuid = createUuid(),
                                      FFTRecord_uuid = Tbl_FFT$uuid[Index_FFT],
                                      loccat_uuid = NA, 
                                      infoexlocation_uuid = Result$Loc$infoexlocation_uuid[Index_Loc]))
          
        } # end of Index_Loc
        
      } # end of Index_FFT loop
      
      ## Adding loccat uuid
      Unique_InfoExLocation_uuid <- unique(Tbl_Loc$infoexlocation_uuid)
      for (Index_UniqueLoc in 1:length(Unique_InfoExLocation_uuid)) {
        Tbl_Loc$loccat_uuid[Tbl_Loc$infoexlocation_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]] <- AllInfoExLoc$uuid[AllInfoExLoc$infoex_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]]  
      }
      
      ## Change name of column
      colnames(Tbl_Loc)[which(colnames(Tbl_Loc)=="FFTRecord_uuid")] <- paste0(TargetSchema, gsub("_", "", TargetTbl), "_uuid")
      
      ## Delete unnecessary column
      Tbl_FFT <- deleteDFColumnsByName(Tbl_FFT, "workflow_execution_id")
      
  
      ## WRITING TABLES TO DATABASE
      ## **************************
      
      if (Overwrite) {
        
        ## User confirmation
        if (UserConfirm==F) {
          UserResponse <- "Yes"
        } else {
          NumRec <- getRecordsFromQuery(Operation, paste0("SELECT count(uuid) as numrec FROM ", TargetSchema, ".", TargetTbl), DBType = DBType)$numrec
          print(paste0("Are you sure you want to delete all of the existing records: ", NumRec, " records."))
          UserResponse <- readline("Type 'Yes' to confirm deletion: ")
        }
        
        if (UserResponse=="Yes") {
          print("Deleting existing records ...")
          sendQueryToGPSDB(Operation, paste0("DELETE FROM ", TargetSchema, ".", TargetTbl, "_loc_x"), DBType=DBType)
          sendQueryToGPSDB(Operation, paste0("DELETE FROM ", TargetSchema, ".", TargetTbl), DBType=DBType)
        }
      }
      
      writeDFToGPSDB(DF=Tbl_FFT, Operation, TargetSchema, TargetTbl, DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "freeform text record")
      writeDFToGPSDB(DF=Tbl_Loc, Operation, TargetSchema, paste0(TargetTbl, "_loc_x"), DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "freeform text record locations")

    } else {
      
      warning("Not free form text records to import!", immediate. = T)
      
    }
      
  } else {
    
    warning("Not free form text records to import!", immediate. = T)
    
  }
  
  print("")
    
}  