#' Retrieve hazard assessments from InfoEx and write into PostgreSQL/PostGIS GPS database.
#'
#' Retrieve hazard assessments from InfoEx and write into PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS
#' @param DateStart Start date in YYYY-MM-DD format. Default value is NA. 
#' @param DateEnd End date in YYYY-MM-DD format. Default value is DateStart.
#' @param NewRecordsOnly Boolean flag for importing new records only. Default is T.
#' @param SkipMissingLoc Switch for whether records of missing locations should be skipped or produce an error. Default is F.
#' @param Overwrite Boolean flag for whether the entire avobs table should be overwritten (i.e., all records deleted first). Default is F.

#' @export

importInfoExHzdAssess <- function(Operation, DBType="Main", DateStart=NA, DateEnd=DateStart, NewRecordsOnly=T, SkipMissingLoc=F, Overwrite=F, UserConfirm=T, Verbose=F, SuppressPostgreSQLWarnings=T) {
  
  ## Retrieve information from InfoEx
  require(SarpInfoExToolsPrivate)
  require(SarpInfoExTools)
  
  HzdAssess <- SarpInfoExTools::getInfoExHzdAssess(Operation, DateStart=DateStart, DateEnd=DateEnd, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
  
  detach("package:SarpInfoExToolsPrivate", unload=TRUE)
  detach("package:SarpInfoExTools", unload=TRUE)
  
  ## Processing
  if(nrow(HzdAssess$HzdAssess)> 0) {
    
    print("Processing records for importing into GPS database ...")
    
    ## Eliminate records that already exist in DB
    if(NewRecordsOnly) {
      HzdAssess$HzdAssess <- returnNewExternalRecordsOnly(HzdAssess$HzdAssess, ImportRelevantId="uuid", ExistRelevantId="infoex_uuid", Operation=Operation, Schema="infoex", DBTbl="hzdassess", DBType=DBType)
    } else {
      print(paste0("Number of hzd assess retrieved from InfoEx: ", nrow(HzdAssess$HzdAssess)))
    }
    
    if(nrow(HzdAssess$HzdAssess)> 0) {
    
      ## Hazard assessments
      ## ******************
      
      ## Change column names
      colnames(HzdAssess$HzdAssess)[which(colnames(HzdAssess$HzdAssess)=="submitted_on")] <- "datetime_submission"
      colnames(HzdAssess$HzdAssess)[which(colnames(HzdAssess$HzdAssess)=="assessment_date")] <- "date_assess"
      colnames(HzdAssess$HzdAssess)[which(colnames(HzdAssess$HzdAssess)=="assessment_time")] <- "time_assess"
      colnames(HzdAssess$HzdAssess)[which(colnames(HzdAssess$HzdAssess)=="character")] <- "type"
      
      ## Create new columns
      HzdAssess$HzdAssess$infoex_uuid <- HzdAssess$HzdAssess$uuid
      HzdAssess$HzdAssess$source <- "InfoEx"
      HzdAssess$HzdAssess$date_assess_esri <- HzdAssess$HzdAssess$date_assess
      HzdAssess$HzdAssess$datetime_assess <- paste(HzdAssess$HzdAssess$date_assess, HzdAssess$HzdAssess$time_assess)
      
      ## Remove unnecessary columns as needed
      HzdAssess$HzdAssess <- deleteDFColumnsByName(HzdAssess$HzdAssess, "location_group_uuid")
      HzdAssess$HzdAssess <- deleteDFColumnsByName(HzdAssess$HzdAssess, "time_assess")
    
        
      ## LOCATION REFERENCES
      ## *******************
      
      ## Delete location references that are not relevant anymore
      HzdAssess$Loc <- HzdAssess$Loc[(HzdAssess$Loc$workflow_execution_id %in% HzdAssess$HzdAssess$infoex_uuid),]
      
      ## Change column names
      colnames(HzdAssess$Loc)[which(colnames(HzdAssess$Loc)=="workflow_execution_id")] <- "infoexhzdassess_uuid"
      
      ## Getting uuids from all Infoex locations in loccat
      AllInfoExLoc <- getRecordsFromQuery(Operation, "Select * from loccat.all_infoex_locations", DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
      AllInfoExLoc <- na.omit(AllInfoExLoc)
      
      ## Update LocationTable
      HzdAssess$Loc$uuid <- sapply(HzdAssess$Loc$infoexhzdassess_uuid, function(x) createUuid())
      
      HzdAssess$Loc$loccat_uuid <- NA
      Unique_InfoExLocation_uuid <- unique(HzdAssess$Loc$infoexlocation_uuid)
      for (Index_UniqueLoc in 1:length(Unique_InfoExLocation_uuid)) {
        loccat_uuid <- AllInfoExLoc$uuid[AllInfoExLoc$infoex_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]]
        loccat_uuid <- loccat_uuid[!is.na(loccat_uuid)]
        if (length(loccat_uuid)==1) {
          HzdAssess$Loc$loccat_uuid[HzdAssess$Loc$infoexlocation_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]] <- loccat_uuid
        }
      }
      
      ## Check for missing locations
      MissingLoc <- HzdAssess$Loc[is.na(HzdAssess$Loc$loccat_uuid),]
      
      if (nrow(MissingLoc)>0) {
        if(SkipMissingLoc) {
          HzdAssess$Loc <- HzdAssess$Loc[!is.na(HzdAssess$Loc$loccat_uuid),]
          NumObsSkipped <- nrow(HzdAssess$HzdAssess[!(HzdAssess$HzdAssess$uuid %in% HzdAssess$Loc$infoexhzdassess_uuid),])
          HzdAssess$HzdAssess <- HzdAssess$HzdAssess[(HzdAssess$HzdAssess$uuid %in% HzdAssess$Loc$infoexhzdassess_uuid),]
          print(paste("Skipping", NumObsSkipped, "records because of missing location information in target database!"))
        } else {
          Error_MissingLoc <<- MissingLoc
          MissingLocCount <- length(unique(MissingLoc$infoexlocation_uuid))
          stop(paste(MissingLocCount, "missing locations for observations!! See data frame Error_MissingLoc"))
        }
      }
      
      ## AVALANCHE PROBLEM TABLE
      ## ***********************
      
      ## Remove avalanche problems not related to new hzd assessment
      HzdAssess$AvProblems <- HzdAssess$AvProblems[(HzdAssess$AvProblems$workflow_execution_id %in% HzdAssess$HzdAssess$infoex_uuid),]
      
      ## Remove unnecessary columns as needed
      HzdAssess$AvProblems <- deleteDFColumnsByName(HzdAssess$AvProblems, "ob_date")
      HzdAssess$AvProblems <- deleteDFColumnsByName(HzdAssess$AvProblems, "state")
      HzdAssess$AvProblems <- deleteDFColumnsByName(HzdAssess$AvProblems, "caaml_group_id")
      HzdAssess$AvProblems <- deleteDFColumnsByName(HzdAssess$AvProblems, "create_user_uuid")
      HzdAssess$AvProblems <- deleteDFColumnsByName(HzdAssess$AvProblems, "operation_uuid")
      HzdAssess$AvProblems <- deleteDFColumnsByName(HzdAssess$AvProblems, "review_user_uuid")
      HzdAssess$AvProblems <- deleteDFColumnsByName(HzdAssess$AvProblems, "submit_user_uuid")
      HzdAssess$AvProblems <- deleteDFColumnsByName(HzdAssess$AvProblems, "ob_time")
      HzdAssess$AvProblems <- deleteDFColumnsByName(HzdAssess$AvProblems, "share_level")
      
      ## Change column names
      colnames(HzdAssess$AvProblems)[which(colnames(HzdAssess$AvProblems)=="workflow_execution_id")] <- "infoexhzdassess_uuid"
      colnames(HzdAssess$AvProblems)[which(colnames(HzdAssess$AvProblems)=="pwl_uuid")] <- "infoex_pwl_uuid"
      
      ## Extracting hazard chart values
      if (nrow(HzdAssess$AvProblems)>0) {
      
        HzdAssess$AvProblems$hzdchart_size_min <- NA
        HzdAssess$AvProblems$hzdchart_size_typ <- NA
        HzdAssess$AvProblems$hzdchart_size_max <- NA
        
        HzdAssess$AvProblems$hzdchart_like_min <- NA
        HzdAssess$AvProblems$hzdchart_like_typ <- NA
        HzdAssess$AvProblems$hzdchart_like_max <- NA
        
        for (Index_AvProb in 1:nrow(HzdAssess$AvProblems)) {
          ParsedJSON <- jsonlite::fromJSON(HzdAssess$AvProblems$hazard_chart[Index_AvProb])
          
          HzdAssess$AvProblems$hzdchart_size_min[Index_AvProb] <- ParsedJSON$x$min
          HzdAssess$AvProblems$hzdchart_size_typ[Index_AvProb] <- ParsedJSON$x$typ
          HzdAssess$AvProblems$hzdchart_size_max[Index_AvProb] <- ParsedJSON$x$max
          
          HzdAssess$AvProblems$hzdchart_like_min[Index_AvProb] <- ParsedJSON$y$min
          HzdAssess$AvProblems$hzdchart_like_typ[Index_AvProb] <- ParsedJSON$y$typ
          HzdAssess$AvProblems$hzdchart_like_max[Index_AvProb] <- ParsedJSON$y$max
        }
        
      } else {
        
        HzdAssess$AvProblems$hzdchart_size_min <- numeric()
        HzdAssess$AvProblems$hzdchart_size_typ <- numeric()
        HzdAssess$AvProblems$hzdchart_size_max <- numeric()
        
        HzdAssess$AvProblems$hzdchart_like_min <- numeric()
        HzdAssess$AvProblems$hzdchart_like_typ <- numeric()
        HzdAssess$AvProblems$hzdchart_like_max <- numeric()
        
      }
      
      HzdAssess$AvProblems <- deleteDFColumnsByName(HzdAssess$AvProblems, "hazard_chart")
      
      
      ## Extract location information
      if (nrow(HzdAssess$AvProblems)>0) {
        
        HzdAssess$AvProblems$loc_alp <- NA
        HzdAssess$AvProblems$loc_tl <- NA
        HzdAssess$AvProblems$loc_btl <- NA
        
        for (Index_AvProb in 1:nrow(HzdAssess$AvProblems)) {
          ParsedJSON <- jsonlite::fromJSON(HzdAssess$AvProblems$location[Index_AvProb])
          
          if (length(ParsedJSON[1][[1]])>0) {HzdAssess$AvProblems$loc_alp[Index_AvProb] <- paste(ParsedJSON[1][[1]], collapse = ",")}
          if (length(ParsedJSON[2][[1]])>0) {HzdAssess$AvProblems$loc_tl[Index_AvProb]  <- paste(ParsedJSON[2][[1]], collapse = ",")}
          if (length(ParsedJSON[3][[1]])>0) {HzdAssess$AvProblems$loc_btl[Index_AvProb] <- paste(ParsedJSON[3][[1]], collapse = ",")}
        }
        
      } else {
        
        HzdAssess$AvProblems$loc_alp <- character()
        HzdAssess$AvProblems$loc_tl <- character()
        HzdAssess$AvProblems$loc_btl <- character()
        
      }
      
      HzdAssess$AvProblems <- deleteDFColumnsByName(HzdAssess$AvProblems, "location")
      
      ## WRITING TABLE TO GPSDB
      ## **********************
      
      # HzdAssess <<- HzdAssess
      
      if(nrow(HzdAssess$HzdAssess)>0) {
      
        if (Overwrite) {
          
          ## User confirmation
          if (UserConfirm==F) {
            UserResponse <- "Yes"
          } else {
            NumRec <- getRecordsFromQuery(Operation, "SELECT count(uuid) as numrec FROM infoex.hzdassess", DBType = DBType)$numrec
            print(paste0("Are you sure you want to delete all of the existing records: ", NumRec, " records."))
            UserResponse <- readline("Type 'Yes' to confirm deletion: ")
          }
          
          ## Deleting
          if (UserResponse=="Yes") { 
            print("Deleting existing records ...")
            sendQueryToGPSDB(Operation, "DELETE FROM infoex.hzdassess_loc_x", DBType=DBType)
            sendQueryToGPSDB(Operation, "DELETE FROM infoex.hzdassess_avprob", DBType=DBType)
            sendQueryToGPSDB(Operation, "DELETE FROM infoex.hzdassess", DBType=DBType)
          }  
          
        }
        
        ## Writing to database
        writeDFToGPSDB(DF=HzdAssess$HzdAssess, Operation, "infoex", "hzdassess", DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "hzd assessment")
        writeDFToGPSDB(DF=HzdAssess$Loc, Operation, "infoex", "hzdassess_loc_x", DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "hzd assessment locations")
        if (nrow(HzdAssess$AvProblems)>0) {
          writeDFToGPSDB(DF=HzdAssess$AvProblems, Operation, "infoex", "hzdassess_avprob", DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "avalanche problem")
        }

      } else {
        
        warning("No hazard assessment to import!", immediate. = T)
        
      }
        
    } else {
    
      warning("No hazard assessment to import!", immediate. = T)
      
    }
          
  } else {
    
    warning("No hazard assessment to import!", immediate. = T)
    
  }
  
  print("")
  
}  