#' Retrieve location catalog from InfoEx and write into PostgreSQL/PostGIS GPS database.
#'
#' Retrieve location catalog from InfoEx and write into PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS
#' @param NewRecordsOnly Boolean flag for importing new records only. Default is T.
#' @param OpArea_Include Boolean flag for whether oparea table should be updated. Default is T.
#' @param OpArea_Overwrite Boolean flag for whether the entire oparea table should be overwritten (i.e., all records deleted first). Default is F.
#' @param Zone_Include Boolean flag for whether zones table should be updated. Default is T.
#' @param Zone_Overwrite Boolean flag for whether the entire zones table should be overwritten (i.e., all records deleted first). Default is F.
#' @param SkiRun_Include Boolean flag for whether skiruns table should be updated. Default is T.
#' @param SkiRun_Overwrite Boolean flag for whether the entire skiruns table should be overwritten (i.e., all records deleted first). Default is F.
#' @param SkiLine_Include Boolean flag for whether skilines table should be updated. Default is T.
#' @param SkiLine_Overwrite Boolean flag for whether the entire skilines table should be overwritten (i.e., all records deleted first). Default is F.
#' @param WxSite_Include Boolean flag for whether wxsites table should be updated. Default is T.
#' @param WxSite_Overwrite Boolean flag for whether the entire wxsites table should be overwritten (i.e., all records deleted first). Default is F.

#' @export

importInfoExLocCatalog <- function(Operation, DBType="Main", NewRecordsOnly=T, OpArea_Include=T, OpArea_Overwrite=F, Zone_Include=T, Zone_Overwrite=F, SkiRun_Include=T, SkiRun_Overwrite=F, SkiLine_Include=T, SkiLine_Overwrite=F, WxSite_Include=T, WxSite_Overwrite=F) {

  require(SarpInfoExToolsPrivate)
  require(SarpInfoExTools)

  InfoExLocCatSPDF <- getInfoExLocCatalog(Operation)

  detach("package:SarpInfoExToolsPrivate", unload=TRUE)
  detach("package:SarpInfoExTools", unload=TRUE)
  
  formatDFforGPSDB <- function(InfoExLocCatSPDF) {
    if(class(InfoExLocCatSPDF)!="logical") {
      InfoExLocCatSPDF@data <- deleteDFColumnsByName(InfoExLocCatSPDF@data, c("abbreviation", "active", "description", "type", "operation_id", "parent_id", "photos"))
      InfoExLocCatSPDF@data$infoex_uuid <- InfoExLocCatSPDF@data$uuid
      InfoExLocCatSPDF@data$infoex_name <- InfoExLocCatSPDF@data$name
      InfoExLocCatSPDF@data$external_id[InfoExLocCatSPDF@data$external_id==''] <- NA
    }
    return(InfoExLocCatSPDF)
  }
  
  InfoExLocCatSPDF$OpArea <- formatDFforGPSDB(InfoExLocCatSPDF$OpArea)
  InfoExLocCatSPDF$Zones <- formatDFforGPSDB(InfoExLocCatSPDF$Zones)
  InfoExLocCatSPDF$SkiRuns <- formatDFforGPSDB(InfoExLocCatSPDF$SkiRuns)
  InfoExLocCatSPDF$SkiLines <- formatDFforGPSDB(InfoExLocCatSPDF$SkiLines)
  InfoExLocCatSPDF$WxSites <- formatDFforGPSDB(InfoExLocCatSPDF$WxSites)
  

  if(class(InfoExLocCatSPDF$OpArea)=="SpatialPolygonsDataFrame" & OpArea_Include) {
    print("PROCESSING OPAREA...")
    if(NewRecordsOnly) {
      InfoExLocCatSPDF$OpArea <- returnNewExternalRecordsOnlySPDF(InfoExLocCatSPDF$OpArea, Operation=Operation, Schema="loccat", DBTbl="oparea", DBType=DBType)
    }
    if (!is.null(InfoExLocCatSPDF$OpArea)){
      if (nrow(InfoExLocCatSPDF$OpArea) > 0) {
        writeSPDFToGPSDB(SPDF = InfoExLocCatSPDF$OpArea, Operation, "loccat", "oparea", DBType=DBType, Overwrite=OpArea_Overwrite, Polygon2Multipolygon = T, ShowCounter=1, ObjLabel = "OpArea")
      } else {
        print("No records to be imported.")
      }
    }
  }
  
  if(class(InfoExLocCatSPDF$Zones)=="SpatialPolygonsDataFrame" & Zone_Include) {
    print("PROCESSING ZONES...")
    if(NewRecordsOnly) {
      InfoExLocCatSPDF$Zones <- returnNewExternalRecordsOnlySPDF(InfoExLocCatSPDF$Zones, Operation=Operation, Schema="loccat", DBTbl="zones", DBType=DBType)
    }
    if (!is.null(InfoExLocCatSPDF$Zones)){
      if (nrow(InfoExLocCatSPDF$Zones) > 0) {
        writeSPDFToGPSDB(SPDF = InfoExLocCatSPDF$Zones, Operation, "loccat", "zones", DBType=DBType, Overwrite=Zone_Overwrite, ShowCounter=1, ObjLabel = "Zone")
      } else {
        print("No records to be imported.")
      }
    }
  }
  
  if(class(InfoExLocCatSPDF$SkiRuns)=="SpatialPolygonsDataFrame" & SkiRun_Include) {
    print("PROCESSING SKIRUNS...")
    if(NewRecordsOnly) {
      InfoExLocCatSPDF$SkiRuns <- returnNewExternalRecordsOnlySPDF(InfoExLocCatSPDF$SkiRuns, Operation=Operation, Schema="loccat", DBTbl="skiruns", DBType=DBType)
    }
    if (!is.null(InfoExLocCatSPDF$SkiRuns)){
      if (nrow(InfoExLocCatSPDF$SkiRuns) > 0) {
        writeSPDFToGPSDB(SPDF = InfoExLocCatSPDF$SkiRuns, Operation, "loccat", "skiruns", DBType=DBType, Overwrite=SkiRun_Overwrite, ShowCounter=20, ObjLabel = "Ski Run")
      } else {
        print("No records to be imported.")
      }
    }
  }
  
  if(class(InfoExLocCatSPDF$SkiLines)=="SpatialLinesDataFrame" & SkiLine_Include) {
    print("PROCESSING SKILINES...")
    if(NewRecordsOnly) {
      InfoExLocCatSPDF$SkiLines <- returnNewExternalRecordsOnlySPDF(InfoExLocCatSPDF$SkiLines, Operation=Operation, Schema="loccat", DBTbl="skilines", DBType=DBType)
    }
    if (!is.null(InfoExLocCatSPDF$SkiLines)){
      if (nrow(InfoExLocCatSPDF$SkiLines) > 0) {
        writeSPDFToGPSDB(SPDF = InfoExLocCatSPDF$SkiLines, Operation, "loccat", "skilines", DBType=DBType, Overwrite=SkiLine_Overwrite, ShowCounter=20, ObjLabel = "Ski Line")
      } else {
        print("No records to be imported.")
      }
    }
  }
  
  if(class(InfoExLocCatSPDF$WxSites)=="SpatialPointsDataFrame" & WxSite_Include) {
    print("PROCESSING WXSITES...")
    if(NewRecordsOnly) {
      InfoExLocCatSPDF$WxSites <- returnNewExternalRecordsOnlySPDF(InfoExLocCatSPDF$WxSites, Operation=Operation, Schema="loccat", DBTbl="wxsites", DBType=DBType)
    }
    if (!is.null(InfoExLocCatSPDF$WxSites)){
      if (nrow(InfoExLocCatSPDF$WxSites) > 0) {
        writeSPDFToGPSDB(SPDF = InfoExLocCatSPDF$WxSites, Operation, "loccat", "wxsites", DBType=DBType, Overwrite=WxSite_Overwrite, ShowCounter=1, ObjLabel = "Wx Site")
      } else {
        print("No records to be imported.")
      }
    }
  }
  
  print("")
  
}