#' Retrieve run coding from InfoEx and write into PostgreSQL/PostGIS GPS database.
#'
#' Retrieve run coding from InfoEx and write into PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS
#' @param DateStart Start date in YYYY-MM-DD format. Default value is NA. 
#' @param DateEnd End date in YYYY-MM-DD format. Default value is DateStart.
#' @param NewRecordsOnly Boolean flag for importing new records only. Default is T.
#' @param SkipMissingLoc Switch for whether records of missing locations should be skipped or produce an error. Default is F.
#' @param Overwrite Boolean flag for whether the entire avobs table should be overwritten (i.e., all records deleted first). Default is F.
#'
#' @export

importInfoExRunCoding <- function(Operation, DBType="Main", DateStart=NA, DateEnd=DateStart, NewRecordsOnly=T, SkipMissingLoc=F, Overwrite=F, Verbose=F, SuppressPostgreSQLWarnings=T) {
  
  ## Retrieve information from InfoEx
  require(SarpInfoExToolsPrivate)
  require(SarpInfoExTools)
  
  TblRunCoding <- getInfoExRunCoding(Operation, DateStart=DateStart, DateEnd=DateEnd, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
  
  detach("package:SarpInfoExToolsPrivate", unload=TRUE)
  detach("package:SarpInfoExTools", unload=TRUE)
  
  ## Processing
  if(nrow(TblRunCoding)>0) {
    
    ## Eliminate records that already exist in DB
    if(NewRecordsOnly & !Overwrite) {
      TblRunCoding <- returnNewExternalRecordsOnly(TblRunCoding, ImportRelevantId="uuid", ExistRelevantId="infoex_uuid", Operation=Operation, Schema="runlist", DBTbl="codes_skiruns_skilines", DBType=DBType)
    } else {
      print(paste0("Number of run codings retrieved from InfoEx: ", nrow(TblRunCoding)))
    }
    
    if(nrow(TblRunCoding)>0) {
  
      ## Change name of columns
      colnames(TblRunCoding)[which(colnames(TblRunCoding)=="ob_date")] <- "date"
      colnames(TblRunCoding)[which(colnames(TblRunCoding)=="location_uuid")] <- "infoexlocation_uuid"
      colnames(TblRunCoding)[which(colnames(TblRunCoding)=="name")] <- "code_general"
      
      ## Adding additional columns as needed
      TblRunCoding$date_esri <- TblRunCoding$date
      TblRunCoding$infoex_uuid <- TblRunCoding$uuid
      TblRunCoding$source <- "InfoEx"
      
      ## Delete unnecessary columns
      TblRunCoding <- deleteDFColumnsByName(TblRunCoding, "ob_time")
      
          
      ## Getting uuids from loccat.skiruns and loccat.skilines
      Query <- "SELECT uuid, infoex_uuid FROM loccat.skiruns"
      TblSkiruns <- getRecordsFromQuery(Operation, Query=Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
      
      Query <- "SELECT uuid, infoex_uuid FROM loccat.skilines"
      TblSkilines <- getRecordsFromQuery(Operation, Query=Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
      
      ## Filling in loccatskiruns_uuid and loccatskilines_uuid
      TblRunCoding$loccatskiruns_uuid <- NA
      Unique_InfoExLocation_uuid <- unique(TblRunCoding$infoexlocation_uuid)
      for (Index_UniqueLoc in 1:length(Unique_InfoExLocation_uuid)) {
        loccat_uuid <- TblSkiruns$uuid[TblSkiruns$infoex_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]]
        if (length(loccat_uuid)>0) {
          TblRunCoding$loccatskiruns_uuid[TblRunCoding$infoexlocation_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]] <- loccat_uuid
        }
      }
      
      TblRunCoding$loccatskilines_uuid <- NA
      Unique_InfoExLocation_uuid <- unique(TblRunCoding$infoexlocation_uuid)
      for (Index_UniqueLoc in 1:length(Unique_InfoExLocation_uuid)) {
        loccat_uuid <- TblSkilines$uuid[TblSkilines$infoex_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]]
        if (length(loccat_uuid)>0) {
          TblRunCoding$loccatskilines_uuid[TblRunCoding$infoexlocation_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]] <- loccat_uuid
        }
      }
      
      ## Check for missing locations
      MissingLoc <- TblRunCoding[is.na(TblRunCoding$loccatskiruns_uuid) & is.na(TblRunCoding$loccatskilines_uuid),]
  
      if (nrow(MissingLoc)>0) {
        if (SkipMissingLoc) { 
          TblRunCoding <- TblRunCoding[!is.na(TblRunCoding$loccatskiruns_uuid) | !is.na(TblRunCoding$loccatskilines_uuid),]
          print(paste("Skipping", nrow(MissingLoc), "records because of missing location information in target database."))
        } else {
          Error_MissingLoc <<- MissingLoc
          MissingLocCount <- length(unique(MissingLoc$infoexlocation_uuid))
          stop(paste(MissingLocCount, "missing locations for observations!! See data frame Error_MissingLoc"))
        }
      }
      
      if(nrow(TblRunCoding)>0) {
      
        ## Creating records for daily runlist
        UniqueDates <- unique(TblRunCoding$date)
        TblRunlist <- data.frame(uuid=NA,
                                 date=UniqueDates,
                                 date_esri=UniqueDates,
                                 source='InfoEx')
        
        ## Creating uuid for daily runlist and trasnfer to code records
        TblRunCoding$runlistrunlists_uuid <- NA
        for (Index_RunList in 1:nrow(TblRunlist)) {
          TblRunlist$uuid[Index_RunList] <- createUuid()
          TblRunCoding$runlistrunlists_uuid[TblRunCoding$date==TblRunlist$date[Index_RunList]] <- TblRunlist$uuid[Index_RunList]
        }
        
        ## Writing table to GPSDB
        writeDFToGPSDB(DF=TblRunlist, Operation=Operation, Schema="runlist", DBTbl="runlists", DBType=DBType, Overwrite=Overwrite, ShowCounter=20, ObjLabel = "runlist record", Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        writeDFToGPSDB(DF=TblRunCoding, Operation=Operation, Schema="runlist", DBTbl="codes_skiruns_skilines", DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "run code record", Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        
      } else {
        
        warning("No run codings to import!", immediate. = T)
        
      }
      
    } else {
      
      warning("No run codings to import!", immediate. = T)
      
    }
      
  } else {
    
    warning("No run codings to import!", immediate. = T)
    
  }
  
  print("")
  
}  