#' Retrieve weather observations from InfoEx and write into PostgreSQL/PostGIS GPS database.
#'
#' Retrieve weather observations from InfoEx and write into PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS
#' @param DateStart Start date in YYYY-MM-DD format. Default value is NA. 
#' @param DateEnd End date in YYYY-MM-DD format. Default value is DateStart.
#' @param NewRecordsOnly Boolean flag for importing new records only. Default is T.
#' @param Overwrite Boolean flag for whether the entire wxobs table should be overwritten (i.e., all records deleted first). Default is F.
#' @export

importInfoExWxObs <- function(Operation, DBType="Main", DateStart=NA, DateEnd=DateStart, NewRecordsOnly=T, Overwrite=F, Verbose=F, SuppressPostgreSQLWarnings=T) {
  
  ## Retrieve information from InfoEx
  require(SarpInfoExToolsPrivate)
  require(SarpInfoExTools)
  
  WxObsObj <- SarpInfoExTools::getInfoExWxObs(Operation, DateStart=DateStart, DateEnd=DateEnd, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
  
  detach("package:SarpInfoExToolsPrivate", unload=TRUE)
  detach("package:SarpInfoExTools", unload=TRUE)
  
  TblWxObs <- WxObsObj$Obs
  
  if(nrow(TblWxObs)>0) {
  
    ## Eliminate records that already exist in DB
    if(NewRecordsOnly) {
      TblWxObs <- returnNewExternalRecordsOnly(TblWxObs, ImportRelevantId="uuid", ExistRelevantId="infoex_uuid", Operation=Operation, Schema="infoex", DBTbl="wxobs", DBType=DBType)
    } else {
      print(paste0("Number of wx obs retrieved from InfoEx: ", nrow(TblWxObs)))
    }
    
    if(nrow(TblWxObs)>0) {
    
      ## Adding additional columns as needed
      TblWxObs$date_esri <- TblWxObs$date
      TblWxObs$infoex_uuid <- TblWxObs$uuid
      TblWxObs$source <- "InfoEx"
      
      ## Getting uuids from loccat.wxsites
      Query <- "SELECT uuid, infoex_uuid FROM loccat.wxsites"
      TblWxSites <- getRecordsFromQuery(Operation, Query=Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
      
      ## Eliminating weather records without valid location uuid
      # TblWxObs <- TblWxObs[TblWxObs$infoexlocation_uuid %in% TblWxSites$infoex_uuid,]
      # print(paste0("Number of wx obs with existing locations: ", nrow(TblWxObs)))
      
      ## Filling in loccatwxsites_uuid
      TblWxObs$loccatwxsites_uuid <- NA
      Unique_InfoExLocation_uuid <- unique(TblWxObs$infoexlocation_uuid)
      for (Index_UniqueLoc in 1:length(Unique_InfoExLocation_uuid)) {
        loccat_uuid <- TblWxSites$uuid[TblWxSites$infoex_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]]
        loccat_uuid <- loccat_uuid[!is.na(loccat_uuid)]
        if (length(loccat_uuid)==1) {
          TblWxObs$loccatwxsites_uuid[TblWxObs$infoexlocation_uuid==Unique_InfoExLocation_uuid[Index_UniqueLoc]] <- loccat_uuid
        }
      }
      
      ## Check for missing locations
      MissingLoc <- TblWxObs[is.na(TblWxObs$loccatwxsites_uuid),]
      if (nrow(MissingLoc)>0) {
        Error_MissingLoc <<- MissingLoc
        MissingLocCount <- length(unique(Error_MissingLoc$externallocation_id))
        stop(paste(MissingLocCount, "missing locations for observations!! See data frame Error_MissingLoc"))
      }
      
      ## Writing table to GPSDB
      writeDFToGPSDB(DF=TblWxObs, Operation=Operation, Schema="infoex", DBTbl="wxobs", DBType=DBType, Overwrite=Overwrite, ShowCounter=20, ObjLabel = "wx observation", Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

    } else {
      
      warning("Not weather observations to import!", immediate. = T)
      
    }
    
  } else {
    
    warning("Not weather observations to import!", immediate. = T)
    
  }
  
  print("")
    
}  