#' Import capacity table from Roger
#'
#' Import simulated capacity table from Roger. Currently only implemented for zones_daily table
#' @param Operation Name of operation
#' @param FileName Name of csv file to be imported including full path
#' @param DBTbl Name of capacity table for data import. Only zobnes_daily implemented so far.
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS
#' @param ShowCounter Interval for showing progress counter. 
#' @param Overwrite Boolean flag for whether the entire avobs table should be overwritten (i.e., all records deleted first). Default is F.
#' @export
#' 
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "CMHGL"
#' 
#' Folder <- "C:\\Users\\Pascal\\Documents\\Pascal\\SFU\\Students-Current\\SterchiReto\\Paper1_DataCollection\\CapacityData\\"
#' File <- "Zone_Daily.csv"
#' 
#' importSnowbaseCapacity(Operation, FileName=paste0(Folder, File), Overwrite=T)



importSnowbaseCapacity <- function(Operation, FileName, DBTbl="zones_daily", DBType="Main", ShowCounter=500, Overwrite=F) {

  ## Get data
  print("Reading data ...")
  Tbl <- read.csv(file=FileName)

  if (DBTbl=="zones_daily") {
    
    print("Processing data ...")
    
    ## Adding zone uuids
    LocCat <- getLocCatlog(Operation)
    ZonesDF <- LocCat$Zones@data
    ZonesDF <- ZonesDF[,c(1, 3)]
    names(ZonesDF)[1] <- "loccatzones_uuid"
    Tbl <- merge(Tbl, ZonesDF, by.x="Zone_ID", by.y="external_id", all.x=T)
    
    ## Check for missing location uuids
    Test <- Tbl[is.na(Tbl$loccatzones_uuid),]
    if (nrow(Test)>0) {
      ErrorTbl <<- Test
      stop("No location uuids for all records. See ErrorTbl for incomplete records!")
    }
    
    ## Adding uuids
    Tbl$uuid <- apply(Tbl, 1, function(x) createUuid())
    
    ## Reformatting date
    reformatDate <- function(olddate) {
      parts <- strsplit(strsplit(as.character(olddate), " ")[[1]][1], "/")[[1]]
      newdate <- paste(parts[3], parts[1], parts[2], sep="-")
      return(newdate)
    }
    for (i in 1:nrow(Tbl)) {
      Tbl$date[i] <- reformatDate(Tbl$Zone_Date[i])
    }
    
    ## Create import table
    Import <- data.frame(uuid=Tbl$uuid,
                         loccatzones_uuid=Tbl$loccatzones_uuid,
                         date=Tbl$date,
                         date_esri=Tbl$date,
                         datetime=Tbl$date,
                         num_skiers=Tbl$Num_Skiers,
                         accessibility=Tbl$Accessibility,
                         cap_runs_all=Tbl$ZoneDailyCapacity,
                         cap_stb=Tbl$ZoneDailyStbCapacity,
                         cap_access=Tbl$ZoneDailyAccessCapacity,
                         cap_use=Tbl$ZoneDailyUseCapacity,
                         cap_stbaccessuse=Tbl$ZoneDailyStbAccessAndUseCapacity,
                         cap_solar=Tbl$ZoneDailySolarCapacity,
                         cap_runs_eonly=Tbl$ZoneDailyERunsCapacity,
                         cap_runs_gplusonly=Tbl$ZoneDailyGPlusRunsCapacity)
    
    ## Write to DB
    writeDFToGPSDB(Import, Operation=Operation, Schema="capacity", DBTbl=DBTbl, ShowCounter=ShowCounter, Overwrite = Overwrite)
    
  } else {
    
    warning("Import for this capacity database tabvle has not been implemented yet!", immediate. = T)
    
  }
  
}

