#' Retrieve hazard assessments from Snowbase and write into PostgreSQL/PostGIS GPS database.
#'
#' Retrieve hazard assessments from Snowbase and write into PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS
#' @param DateStart Start date in YYYY-MM-DD format. Default value is NA. 
#' @param DateEnd End date in YYYY-MM-DD format. Default value is DateStart.
#' @param NewRecordsOnly Boolean flag for importing new records only. Default is T.
#' @param SkipMissingLoc Switch for whether records of missing locations should be skipped or produce an error. Default is F.
#' @param Overwrite Boolean flag for whether the entire avobs table should be overwritten (i.e., all records deleted first). Default is F.

#' @export

importSnowbaseHzdAssess <- function(Operation, DBType="Main", DateStart=NA, DateEnd=DateStart, NewRecordsOnly=T, SkipMissingLoc=F, Overwrite=F, Verbose=F, SuppressPostgreSQLWarnings=T) {
  
  ## Retrieve information from InfoEx
  print("Retrieving records from Snowbase")
  HzdAssess <- SarpCMHSnowbase::getSnowbaseHzdAssess(Operation, DateStart=DateStart, DateEnd=DateEnd, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
  
  ## Eliminate records that already exist in DB
  if(NewRecordsOnly) {
    HzdAssess <- returnNewExternalRecordsOnly(HzdAssess, ImportRelevantId="STB_ID", ExistRelevantId="external_id", Operation=Operation, Schema="infoex", DBTbl="hzdassess", DBType=DBType)
  } else {
    print(paste0("Number of hzd assess retrieved from InfoEx: ", nrow(HzdAssess$HzdAssess)))
  }
  
  ## Processing
  if(nrow(HzdAssess)> 0) {
    
    print(paste0("Number of hzd assessments retrieved from Snowbase: ", nrow(HzdAssess)))
    print("Processing records for importing into GPS database ...")
    
    ## Hazard assessments
    ## ******************
    
    ## Change column names
    colnames(HzdAssess)[which(colnames(HzdAssess)=="DTMOD")] <- "datetime_submission"
    colnames(HzdAssess)[which(colnames(HzdAssess)=="STB_DATE")] <- "datetime_assess"
    colnames(HzdAssess)[which(colnames(HzdAssess)=="STB_ID")] <- "external_id"
    colnames(HzdAssess)[which(colnames(HzdAssess)=="STB_ALP")] <- "alp_stb"
    colnames(HzdAssess)[which(colnames(HzdAssess)=="STB_ALP_REMARK")] <- "alp_comment"
    colnames(HzdAssess)[which(colnames(HzdAssess)=="STB_TLINE")] <- "tl_stb"
    colnames(HzdAssess)[which(colnames(HzdAssess)=="STB_TLINE_REMARK")] <- "tl_comment"
    colnames(HzdAssess)[which(colnames(HzdAssess)=="STB_SUB")] <- "btl_stb"
    colnames(HzdAssess)[which(colnames(HzdAssess)=="STB_SUB_REMARK")] <- "btl_comment"
    colnames(HzdAssess)[which(colnames(HzdAssess)=="AvalancheHazard")] <- "comment"
    
    ## Create new columns
    HzdAssess$source <- "Snowbase"
    HzdAssess$date_assess <- substr(HzdAssess$datetime_assess, 1, 10)
    HzdAssess$date_assess_esri <- HzdAssess$date_assess
    HzdAssess$uuid <- sapply(HzdAssess$external_id, function(x) createUuid())
    
    Hour <- as.numeric(substr(HzdAssess$datetime_assess, 12, 13))
    HzdAssess$type <- NA
    HzdAssess$type[Hour < 10] <- 'forecast'
    HzdAssess$type[Hour > 14] <- 'nowcast'
    
    ## Remove unnecessary columns as needed
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "SSG_REPORT")
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "MARK_DELETED")
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "DtCreated")
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "Creator")
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "Modifier")
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "RowId")
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "RowGuid")
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "Terrain")
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "SnowSafetyGuide")
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "SNOW_STRUCT")
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "Area_Code")
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "Mindset")
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "MindsetComments")
    
    ## Split stability and hazard rating
    HzdAssess$alp_hzd <- HzdAssess$alp_stb
    HzdAssess$tl_hzd  <- HzdAssess$tl_stb
    HzdAssess$btl_hzd <- HzdAssess$btl_stb
    
    HzdRatings <- c("1", "2", "3", "4", "5")
    
    HzdAssess$alp_stb[HzdAssess$alp_stb %in% HzdRatings] <- NA
    HzdAssess$tl_stb[HzdAssess$tl_stb %in% HzdRatings] <- NA
    HzdAssess$btl_stb[HzdAssess$btl_stb %in% HzdRatings] <- NA
    
    HzdAssess$alp_hzd[!(HzdAssess$alp_hzd %in% HzdRatings)] <- NA
    HzdAssess$tl_hzd[!(HzdAssess$tl_hzd %in% HzdRatings)] <- NA
    HzdAssess$btl_hzd[!(HzdAssess$btl_hzd %in% HzdRatings)] <- NA
    
    ## Clean up comment columns
    HzdAssess$alp_comment[HzdAssess$alp_comment==""] <-NA
    HzdAssess$tl_comment[HzdAssess$tl_comment==""] <-NA
    HzdAssess$btl_comment[HzdAssess$btl_comment==""] <-NA
    HzdAssess$comment[HzdAssess$comment==""] <-NA
    
    # Output_HzdAssess <<- HzdAssess

      
    ## LOCATION REFERENCES
    ## *******************

    ## Create location_x table
    Loc <- data.frame(uuid = sapply(HzdAssess$uuid, function(x) createUuid()),
                      infoexhzdassess_uuid = HzdAssess$uuid,
                      loccat_uuid = NA,
                      infoexlocation_uuid = NA,
                      externallocation_id = HzdAssess$GEO_ID)
    
    ## Remove GEO_ID from HzdAssess
    HzdAssess <- deleteDFColumnsByName(HzdAssess, "GEO_ID")
    
    
    ## Getting uuids from all Infoex locations in loccat
    AllInfoExLoc <- getRecordsFromQuery(Operation, "Select * from loccat.all_infoex_locations", DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

    ## Fill location table
    Unique_ExternalLocation_id <- unique(Loc$externallocation_id)
    for (Index_UniqueLoc in 1:length(Unique_ExternalLocation_id)) {
      loccat_uuid <- AllInfoExLoc$uuid[AllInfoExLoc$external_id==Unique_ExternalLocation_id[Index_UniqueLoc]]
      loccat_uuid <- loccat_uuid[!is.na(loccat_uuid)]
      if (length(loccat_uuid)==1) {
        Loc$loccat_uuid[Loc$externallocation_id==Unique_ExternalLocation_id[Index_UniqueLoc]] <- loccat_uuid  
      }
    }
    
    ## Check for missing locations
    MissingLoc <- Loc[is.na(Loc$loccat_uuid),]
    
    if (nrow(MissingLoc)>0) {
      if(SkipMissingLoc) {
        Loc <- Loc[!is.na(Loc$loccat_uuid),]
        NumObsSkipped <- nrow(HzdAssess[!(HzdAssess$uuid %in% Loc$infoexhzdassess_uuid),])
        HzdAssess <- HzdAssess[(HzdAssess$uuid %in% Loc$infoexhzdassess_uuid),]
        print(paste("Skipping", NumObsSkipped, "records because of missing location information in target database!"))
      } else {
        Error_MissingLoc <<- MissingLoc
        MissingLocCount <- length(unique(Error_MissingLoc$externallocation_id))
        stop(paste(MissingLocCount, "missing locations for observations!! See data frame Error_MissingLoc"))
      }
    }
    
    # Loc <<- Loc
    
    ## AVALANCHE PROBLEM TABLE
    ## ***********************
    

    ## WRITING TABLE TO GPSDB
    ## **********************
    
    if (Overwrite) {
      print("Deleting existing records ...")
      sendQueryToGPSDB(Operation, "DELETE FROM infoex.hzdassess_loc_x", DBType=DBType)
      sendQueryToGPSDB(Operation, "DELETE FROM infoex.hzdassess_avprob", DBType=DBType)
      sendQueryToGPSDB(Operation, "DELETE FROM infoex.hzdassess", DBType=DBType)
    }
    
    ## Writing to database
    if (nrow(HzdAssess)>0) {
      writeDFToGPSDB(DF=HzdAssess, Operation, "infoex", "hzdassess", DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "hzd assessment")
      writeDFToGPSDB(DF=Loc, Operation, "infoex", "hzdassess_loc_x", DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "hzd assessment locations")
      # writeDFToGPSDB(DF=HzdAssess$AvProblems, Operation, "infoex", "hzdassess_avprob", DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "avalanche problem")
    } else {
      warning("No hazard assessment to import!", immediate. = T)
    }
    
  } else {
    
    warning("No hazard assessment to import!", immediate. = T)
  }
  
}  