#' Retrieve run coding from Snowbase and write into PostgreSQL/PostGIS GPS database.
#'
#' Retrieve run codinge from Snowbase and write into PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS
#' @param DateStart Start date in YYYY-MM-DD format. Default value is NA. 
#' @param DateEnd End date in YYYY-MM-DD format. Default value is DateStart.
#' @param NewRecordsOnly Boolean flag for importing new records only. Default is T.
#' @param SkipMissingLoc Switch for whether records of missing locations should be skipped or produce an error. Default is F.
#' @param Overwrite Boolean flag for whether the entire avobs table should be overwritten (i.e., all records deleted first). Default is F.

#' @export

importSnowbaseRunCoding <- function(Operation, DBType="Main", DateStart=NA, DateEnd=DateStart, NewRecordsOnly=T, SkipMissingLoc=F, Overwrite=F, Verbose=F, SuppressPostgreSQLWarnings=T) {
  
  ## Retrieve information from InfoEx
  print("Retrieving records from Snowbase")
  RunCoding <- SarpCMHSnowbase::getSnowbaseRunCoding(Operation, DateStart=DateStart, DateEnd=DateEnd, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
  
  ## Eliminate records that already exist in DB
  if(NewRecordsOnly) {
    RunCoding <- returnNewExternalRecordsOnly(RunCoding, ImportRelevantId="RUN_CODING_ID", ExistRelevantId="external_id", Operation=Operation, Schema="runlist", DBTbl="codes_skiruns_skilines", DBType=DBType)
  } else {
    print(paste0("Number of run coding retrieved from InfoEx: ", nrow(RunCoding)))
  }
  
  ## Processing
  if(nrow(RunCoding)> 0) {
    
    print(paste0("Number of run coding records retrieved from Snowbase: ", nrow(RunCoding)))
    print("Processing records for importing into GPS database ...")
    
    ## Change column names
    colnames(RunCoding)[which(colnames(RunCoding)=="RUN_CODING_ID")] <- "external_id"
    colnames(RunCoding)[which(colnames(RunCoding)=="RUN_CMNT")] <- "comment"
    colnames(RunCoding)[which(colnames(RunCoding)=="RUN_GEO_ID")] <- "externallocation_id"

    ## Create new columns
    RunCoding$source <- "Snowbase"
    RunCoding$date <- substr(RunCoding$RUNLIST_DATETIME, 1, 10)
    RunCoding$date_esri <- RunCoding$date
    RunCoding$uuid <- sapply(RunCoding$external_id, function(x) createUuid())
    
    RunCoding$code_general <- NA
    RunCoding$code_general[RunCoding$COLOR=="Blk"] <- "black"
    RunCoding$code_general[RunCoding$COLOR=="Grn"] <- "green"
    RunCoding$code_general[RunCoding$COLOR=="Red"] <- "red"
    RunCoding$code_general[RunCoding$COLOR=="Yel"] <- "yellow"
    RunCoding$code_general[is.na(RunCoding$COLOR)] <- "nd"
    RunCoding$code_general <- factor(RunCoding$code_general)
    
    
    ## Remove unnecessary columns as needed
    RunCoding <- deleteDFColumnsByName(RunCoding, "RUNLIST_DATETIME")
    RunCoding <- deleteDFColumnsByName(RunCoding, "RUN_CMNT_ID")
    RunCoding <- deleteDFColumnsByName(RunCoding, "RUNLIST_ID")
    RunCoding <- deleteDFColumnsByName(RunCoding, "MARK_DELETED")
    RunCoding <- deleteDFColumnsByName(RunCoding, "DtCreated")
    RunCoding <- deleteDFColumnsByName(RunCoding, "DTMOD")
    RunCoding <- deleteDFColumnsByName(RunCoding, "Creator")
    RunCoding <- deleteDFColumnsByName(RunCoding, "Modifier")
    RunCoding <- deleteDFColumnsByName(RunCoding, "RowId")
    RunCoding <- deleteDFColumnsByName(RunCoding, "RowGuid")
    RunCoding <- deleteDFColumnsByName(RunCoding, "Area_Code")
    
    RunCoding <- deleteDFColumnsByName(RunCoding, "COLOR")
    
      
    ## Adding location uuids
    ## *********************
    
    ## Getting uuids from all Infoex locations in loccat
    Loccat_Zones    <- getRecordsFromQuery(Operation, "Select external_id from loccat.zones", DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
    Loccat_SkiRuns  <- getRecordsFromQuery(Operation, "Select * from loccat.skiruns", DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
    Loccat_SkiLines <- getRecordsFromQuery(Operation, "Select * from loccat.skilines", DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

    ## Eliminate ratings of zones
    NumCodesWithZones <- nrow(RunCoding[(RunCoding$externallocation_id %in% Loccat_Zones$external_id),])
    if(NumCodesWithZones>0) {
      print(paste0(NumCodesWithZones, " records skipped becasue associated with zones."))
    }
    RunCoding <- RunCoding[!(RunCoding$externallocation_id %in% Loccat_Zones$external_id),]
    
    if(nrow(RunCoding)) {
    
      ## Fill location table - ski runs
      RunCoding$loccatskiruns_uuid <- NA
      Unique_ExternalLocation_id <- unique(RunCoding$externallocation_id)
      for (Index_UniqueLoc in 1:length(Unique_ExternalLocation_id)) {
        loccat_uuid <- Loccat_SkiRuns$uuid[Loccat_SkiRuns$external_id==Unique_ExternalLocation_id[Index_UniqueLoc]]
        loccat_uuid <- loccat_uuid[!is.na(loccat_uuid)]
        if (length(loccat_uuid)==1) {
          RunCoding$loccatskiruns_uuid[RunCoding$externallocation_id==Unique_ExternalLocation_id[Index_UniqueLoc]] <- loccat_uuid  
        }
      }
  
      ## Fill location table - ski lines
      RunCoding$loccatskilines_uuid <- NA
      Unique_ExternalLocation_id <- unique(RunCoding$externallocation_id)
      for (Index_UniqueLoc in 1:length(Unique_ExternalLocation_id)) {
        loccat_uuid <- Loccat_SkiLines$uuid[Loccat_SkiLines$external_id==Unique_ExternalLocation_id[Index_UniqueLoc]]
        if (length(loccat_uuid)>0) {
          RunCoding$loccatskilines_uuid[RunCoding$externallocation_id==Unique_ExternalLocation_id[Index_UniqueLoc]] <- loccat_uuid  
        }
      }
      
      ## Check for missing locations
      MissingLoc <- RunCoding[is.na(RunCoding$loccatskiruns_uuid) & is.na(RunCoding$loccatskilines_uuid),]
      
      if (nrow(MissingLoc)>0) {
        if (SkipMissingLoc) { 
          RunCoding <- RunCoding[!is.na(RunCoding$loccatskiruns_uuid) | !is.na(RunCoding$loccatskilines_uuid),]
          print(paste("Skipping", nrow(MissingLoc), "records because of missing location information in target database."))
        } else {
          Error_MissingLoc <<- MissingLoc
          MissingLocCount <- length(unique(Error_MissingLoc$externallocation_id))
          stop(paste(MissingLocCount, "missing locations for observations!! See data frame Error_MissingLoc"))
        }
      }
    }    
    # Output_RunCoding <<- RunCoding
    
    
    ## CREATE RUNLIST TABLE AND ADD UUID TO RECORDS
    ## ********************************************
    
    if (nrow(RunCoding)>0) {
    
      UniqueDates <- unique(RunCoding$date)
      
      Tbl_Runlists <- data.frame(uuid=NA,
                                 date=UniqueDates,
                                 date_esri=UniqueDates,
                                 source="internal from codes")
      
      for (index in 1:nrow(Tbl_Runlists)) {
        Tbl_Runlists$uuid[index] <- createUuid()
      }
      
      RunCoding$runlistrunlists_uuid <- NA
      for (index in 1:nrow(Tbl_Runlists)) {
        RunCoding$runlistrunlists_uuid[RunCoding$date==Tbl_Runlists$date[index]] <- Tbl_Runlists$uuid[index]
      }
    
    } else {
      
      Tbl_Runlists <- data.frame(uuid=character(0),
                                 date=character(0),
                                 date_esri=character(0),
                                 source=character(0))
      
    }
    
    ## WRITING TABLES TO GPSDB
    ## ***********************
    
    if (Overwrite) {
      print("Deleting existing records ...")
      sendQueryToGPSDB(Operation, "DELETE FROM runlist.codes_skiruns_skilines", DBType=DBType)
      sendQueryToGPSDB(Operation, "DELETE FROM runlist.runlists", DBType=DBType)
    }
    
    ## Writing to database
    if (nrow(RunCoding)>0) {
      writeDFToGPSDB(DF=Tbl_Runlists, Operation, "runlist", "runlists", DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "run list")
      writeDFToGPSDB(DF=RunCoding, Operation, "runlist", "codes_skiruns_skilines", DBType=DBType, Overwrite=Overwrite, ShowCounter=500, ObjLabel = "run coding record")
    } else {
      warning("No run coding records to import!", immediate. = T)
    }

  } else {
    
    warning("No run coding records to import!", immediate. = T)
    
  }
  
}  