#' Retrieve run use from Snowbase and write into PostgreSQL/PostGIS GPS database.
#'
#' Retrieve run use from Snowbase and write into PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS
#' @param DateStart Start date in YYYY-MM-DD format. Default value is NA. 
#' @param DateEnd End date in YYYY-MM-DD format. Default value is DateStart.
#' @param NewRecordsOnly Boolean flag for importing new records only. Default is T.
#' @param SkipMissingLoc Switch for whether records of missing locations should be skipped or produce an error. Default is F.
#' @param Overwrite Boolean flag for whether the entire avobs table should be overwritten (i.e., all records deleted first). Default is F.

#' @export

importSnowbaseRunUsage <- function(Operation, DBType="Main", DateStart=NA, DateEnd=DateStart, NewRecordsOnly=T, SkipMissingLoc=F, Overwrite=F, Verbose=F, SuppressPostgreSQLWarnings=T) {
  
  ## Retrieve information from InfoEx
  print("Retrieving records from Snowbase")
  RunUsage <- SarpCMHSnowbase::getSnowbaseRunUsage(Operation, DateStart=DateStart, DateEnd=DateEnd, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
  # Input <<- RunUsage
  
  ## Eliminate records that already exist in DB
  if(NewRecordsOnly) {
    RunUsage <- returnNewExternalRecordsOnly(RunUsage, ImportRelevantId="RUN_USE_ID", ExistRelevantId="external_id", Operation=Operation, Schema="obs", DBTbl="run_usage", DBType=DBType)
  } else {
    print(paste0("Number of run usage retrieved from InfoEx: ", nrow(RunUsage)))
  }
  
  ## Processing
  if(nrow(RunUsage)> 0) {
    
    print(paste0("Number of run usage records retrieved from Snowbase: ", nrow(RunUsage)))
    print("Processing records for importing into GPS database ...")
    
    ## Change column names
    colnames(RunUsage)[which(colnames(RunUsage)=="RUN_USE_ID")] <- "external_id"
    colnames(RunUsage)[which(colnames(RunUsage)=="GEO_ID")] <- "externallocation_id"
    colnames(RunUsage)[which(colnames(RunUsage)=="NUM_SKIERS")] <- "run_passes"

    ## Create new columns
    RunUsage$source <- "Snowbase"
    RunUsage$date <- substr(RunUsage$RUNUSE_DATE, 1, 10)
    RunUsage$date_esri <- RunUsage$date
    RunUsage$uuid <- sapply(RunUsage$external_id, function(x) createUuid())
    
    ## Remove unnecessary columns as needed
    RunUsage <- deleteDFColumnsByName(RunUsage, "RUNUSE_DATE")
    RunUsage <- deleteDFColumnsByName(RunUsage, "MARK_DELETED")
    RunUsage <- deleteDFColumnsByName(RunUsage, "DtCreated")
    RunUsage <- deleteDFColumnsByName(RunUsage, "DTMOD")
    RunUsage <- deleteDFColumnsByName(RunUsage, "Creator")
    RunUsage <- deleteDFColumnsByName(RunUsage, "Modifier")
    RunUsage <- deleteDFColumnsByName(RunUsage, "RowId")
    RunUsage <- deleteDFColumnsByName(RunUsage, "RowGuid")
    RunUsage <- deleteDFColumnsByName(RunUsage, "Area_Code")
      
    ## Adding location uuids
    ## *********************
    
    ## Getting uuids from all Infoex locations in loccat
    AllInfoExLoc <- getRecordsFromQuery(Operation, "Select * from loccat.all_infoex_locations", DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

    ## Fill location table
    RunUsage$loccat_uuid <- NA
    Unique_ExternalLocation_id <- unique(RunUsage$externallocation_id)
    for (Index_UniqueLoc in 1:length(Unique_ExternalLocation_id)) {
      loccat_uuid <- AllInfoExLoc$uuid[AllInfoExLoc$external_id==Unique_ExternalLocation_id[Index_UniqueLoc]]
      loccat_uuid <- loccat_uuid[!is.na(loccat_uuid)]
      if (length(loccat_uuid)==1) {
        RunUsage$loccat_uuid[RunUsage$externallocation_id==Unique_ExternalLocation_id[Index_UniqueLoc]] <- loccat_uuid  
      }
    }
    
    ## Check for missing locations
    MissingLoc <- RunUsage[is.na(RunUsage$loccat_uuid),]
    
    if (nrow(MissingLoc)>0) {
      if (SkipMissingLoc) { 
        RunUsage <- RunUsage[!is.na(RunUsage$loccat_uuid),]
        print(paste("Skipping", nrow(MissingLoc), "records becasue of missing location information in target database."))
      } else {
        Error_MissingLoc <<- MissingLoc
        MissingLocCount <- length(unique(Error_MissingLoc$externallocation_id))
        stop(paste(MissingLocCount, "missing locations for observations!! See data frame Error_MissingLoc"))
      }
    }
        
    # Output_RunUsage <<- RunUsage
    
    ## WRITING TABLE TO GPSDB
    ## **********************
    
    if (Overwrite) {
      print("Deleting existing records ...")
      sendQueryToGPSDB(Operation, "DELETE FROM obs.run_usage", DBType=DBType)
    }
    
#     ## Compare with existing records in database
#     NumOrigRecords <- nrow(RunUsage)
#     ExistingRecords <- getRecordsFromQuery(Operation, "Select external_id from obs.run_usage", DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
#     RunUsage <- RunUsage[!(RunUsage$external_id %in% ExistingRecords$external_id),]
#     NumFinalRecords <- nrow(RunUsage)
#     
#     if (NumFinalRecords < NumOrigRecords) {
#       print(paste0("Skipping ", NumOrigRecords-NumFinalRecords, " records because they already exist in database!"))
#     }
    
    ## Writing to database
    if (nrow(RunUsage)>0) {
      writeDFToGPSDB(DF=RunUsage, Operation, "obs", "run_usage", DBType=DBType, Overwrite=Overwrite, ShowCounter=50, ObjLabel = "run usage record")
    } else {
      warning("No run usage records to import!", immediate. = T)
    }

  } else {
    
    warning("No run usage records to import!", immediate. = T)
  }
  
}  