#' Lists all tables or views of a PostgrSQL/PostGIS GPS database.
#'
#' Lists all tables or views of a PostgrSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param TblType Type of tables to be listed. Must either be 'Base Table' or 'View'.
#' @param Schema Optional schema name. Default value is NA, which shows the tables of all schemas.
#' @param DBType Optional database type. Value can be 'Main' (default) or 'NodeJS'.
#'
#' @export

listDBTblsViews <- function(Operation, TblType="Base Table", Schema=NA, DBType="Main") {

  ## Input parameter validation
  if ((tolower(TblType)!="base table") & (tolower(TblType)!="view")) {
    stop("The value for the TblType parameter must either be 'Base Table' or 'View'!")
  }

  ## Retrieving list of schemas
  Query <- "SELECT schema_name as name FROM information_schema.schemata ORDER BY schema_name ASC"
  ListSchema <- getRecordsFromQuery(Operation, Query, DBType=DBType)$name

  if (!is.na(Schema)) {
    ListSchema <- ListSchema[ListSchema==Schema]
  }

  if (length(ListSchema)==0) {

    print(paste0("There is no schema with the name '", Schema, "' in the database!"))

  } else {

    ## Getting table information for each schema
    for (IndexSchema in 1:length(ListSchema)) {

      OutputString <- paste0("List of ", tolower(TblType), "s for schema '", ListSchema[IndexSchema], "'")
      print(OutputString)
      print(substr("******************************************************************", 1, nchar(OutputString)))

      Query <- paste0("SELECT pg_class.relname as name, obj_description(pg_class.oid) as comment ",
                      "FROM pg_class INNER JOIN information_schema.tables ON pg_class.relname=information_schema.tables.table_name ",
                      "WHERE information_schema.tables.table_schema = '", ListSchema[IndexSchema], "' and information_schema.tables.table_type='", toupper(TblType), "' ",
                      "ORDER BY information_schema.tables.table_name")

      TblTable <- getRecordsFromQuery(Operation, Query, DBType)

      if(nrow(TblTable)>0) {
        print(TblTable)
      } else {
        print(paste0("No ", tolower(TblType), "s available in this schema."))
      }
      print("")

    }

  }

}
