#' Marks GPS runs as deleted with specified reason.
#'
#' Marks GPS runs as deleted with specified reason. Runs marked as deleted and associated pause points are hidden in quality controlled views (i.e., qc_runs and qc_pausepoints).
#' @param Operation Name of operation for accessing correct database
#' @param User Name of user.
#' @param RunUUID Single value or array of UUIDs of runs to be marked as deleted.
#' @param Reason Single value or array of value with reason(s) for deletion. If an array is provided, it needs to have the same size as RunUUID array.
#' @param DBType Type of database where the run should be marked as deleted. Values can be 'Main' (default) or 'NodeJS'.
#' @export

markGPSRunAsDeleted <- function(Operation, User, RunUUID, Reason, DBType="Main") {

  ## Check parameter validity: Lenth of RunUUID and Reason arrays
  ## ************************************************************
  if (length(RunUUID)>1) {
    if (length(Reason)==1) {
      ## ok
    } else if (length(RunUUID)==length(Reason)) {
      ## ok
    } else {
      stop("The parameter Reason needs to be either a single value or an array of the same length as RunUUID!")
    }
  }

  ## Get current records of run
  ## **************************
  RunRecords <- getRecordsFromUUID(Operation, "gps", "runs", UUID=RunUUID, SuppressOrderWarning = T, DBType=DBType)

  ## Check that Runs have been found
  if (nrow(RunRecords)==0) {

    stop("No run records have been found for the provided UUID(s)!")

  } else if (nrow(RunRecords)!=length(RunUUID)) {

    stop("Records were not found for all of the provided UUID(s)!")

  } else {

    ## Loop for individual runs
    ## ************************

    for (Index_Runs in 1:nrow(RunRecords)) {

      if (RunRecords$flag_delete[Index_Runs]=="Yes") {

        print(paste0("Run ", Index_Runs, " of ", nrow(RunRecords), " (", RunRecords$uuid[Index_Runs], "): Already marked as deleted!" ))

      } else {

        ReasonInd <- ifelse(length(Reason)==1, Reason, Reason[Index_Runs])

        CommentAddon <- paste0(Sys.Date(), " - Delete reason (", User, "): ", ReasonInd)
        ExistingComment <- RunRecords$comment[Index_Runs]
        NewComment <- ifelse((is.na(ExistingComment)|nchar(ExistingComment)==0), CommentAddon, paste0(ExistingComment, "; ", CommentAddon))

        Query <- paste0("UPDATE gps.runs SET flag_delete='Yes', comment=", convertStringToSQL(NewComment), " WHERE uuid=", convertStringToSQL(RunRecords$uuid[Index_Runs]), ";")
        sendQueryToGPSDB(Operation, Query, DBType=DBType)

        print(paste0("Run ", Index_Runs, " of ", nrow(RunRecords), " (", RunRecords$uuid[Index_Runs], "): Deleted because ", ReasonInd, "!" ))

      }

    }

  }

}
