#' Plot hazard chart from avalanche problem component of hazard assessment object.
#'
#' Plot hazard chart from avalanche problem component of hazard assessment object.
#' @param AvProblems DF with avalanche problems. 
#' @param AvProblemsFaded DF with avalanche problems to be plotted faded.
#' @param Title Title of chart. Default value is 'Hazard Chart'.
#' @param WithPoints Switch whether chart should be plotted with points. Default is TRUE.
#' @param RatingCol Colors for ratings bull's eye (alp, tl, btl)
#' @param RatingLoc Location of rating bull's eye. Array of x and y. Default value c(7,7). 
#' @param RatingSize Size of rating bull's eye. Default value 0.8.
#' @param Transparency The transparency of the blob background. Value can be between 00 and 99. Default value is 50.
#' @param Shape For specifying the shape of the blob. Values can be 'ellipse' (default) or 'square'. 
#' @param NumPoints Number of point per quater section of ellipse. Default value is 50.
#' @param SuppressWarnings Switch for suppressing warning message about application of is.na to S4 object. Default is TRUE. 
#' @param cex.axis cex.axis (default=0.5)
#' @param cex.lab cex.lab (default=0.5)
#' 
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "NEH"
#' 
#' DateStart <- "2016-02-01"
#' DateEnd   <- "2016-02-10"
#' DateSeq   <- as.character(seq(from=as.Date(DateStart), to=as.Date(DateEnd), by=1))
#' 
#' ## Daily plots with ellipses for date sequence including hazard ratings
#' for (Index in 1:length(DateSeq)) {
#'   HzdAssess <- getInfoExHzdAssessObject(Operation, DateStart=DateSeq[Index], Time = "PM")
#'   HzdRating <- c(HzdAssess$HzdAssess$alp_hzd[1], HzdAssess$HzdAssess$tl_hzd[1], HzdAssess$HzdAssess$btl_hzd[1])
#'   HzdCol <- getHzdColor(HzdRating)
#'   plotHzdChart(HzdAssess$AvProb, Title=paste("Hzd Chart -", DateSeq[Index], " (PM)"), Shape="square", RatingCol = HzdCol)
#' }
#'  
#' ## All problems for date sequence as squares
#' HzdAssess <- getInfoExHzdAssessObject(Operation, DateStart=DateStart, DateEnd=DateEnd, Time = "PM")
#' plotHzdChart(HzdAssess$AvProb, Title=paste("Hzd Chart -", DateStart, " to ", DateEnd, " (PM)"), Shape = "square")
#' 
#' ## All problems for entire season as squares
#' DateStart <- "2016-01-01"
#' DateEnd   <- "2016-04-01"
#' HzdAssessSeason <- getInfoExHzdAssessObject(Operation, DateStart=DateStart, DateEnd=DateEnd, Time = "PM")
#' plotHzdChart(HzdAssessSeason$AvProb, Title=paste("All Av Problems - Season 2016 (PM)"), Shape = "square")
#' 
#' ## Specific avalanche problems for entire season as squares 
#' plotHzdChart(HzdAssessSeason$AvProb[HzdAssessSeason$AvProb$character=="WIND_SLAB",], Title=paste("Wind Slabs - Season 2016 (PM)"), Shape = "square")
#' plotHzdChart(HzdAssessSeason$AvProb[HzdAssessSeason$AvProb$character=="STORM_SLAB",], Title=paste("Strom Slabs - Season 2016 (PM)"), Shape = "square")
#' plotHzdChart(HzdAssessSeason$AvProb[HzdAssessSeason$AvProb$character=="PERSISTENT_SLAB",], Title=paste("Pers Slabs - Season 2016 (PM)"), Shape = "square")
#' plotHzdChart(HzdAssessSeason$AvProb[HzdAssessSeason$AvProb$character=="PERSISTENT_SLAB"|HzdAssessSeason$AvProb$character=="DEEP_PERSISTENT_SLAB",], Title=paste("Pers and Deep Pers Slabs - Season 2016 (PM)"), Shape = "square")
#' 
#' @export

plotHzdChart <- function(AvProblems, AvProblemsFaded=NULL, Title="Hazard Chart", WithPoints=T, RatingCol=NA, RatingLoc=c(7,7), RatingSize=0.8, Transparency='50', Shape='square', NumPoints=50, SuppressWarnings=T, cex.axis=0.5, cex.lab=0.5) {
  
  ## Convert avalanche problems to SPDF
  AvProbSPDF <- convertAvProbToSPDF(AvProblems=AvProblems, Shape=Shape, NumPoints=NumPoints)
  AvProbFadedSPDF <- convertAvProbToSPDF(AvProblems=AvProblemsFaded, Shape=Shape, NumPoints=NumPoints)

  if (SuppressWarnings) {options(warn=-1)}
  
  ## Calculate colors
  if(!is.na(AvProbSPDF)) {
    AvProbSPDF@data$ColBord <- sapply(AvProbSPDF@data$character, function(x) getAvCharColor(x))
    AvProbSPDF@data$ColPoly <- paste0(AvProbSPDF@data$ColBord, Transparency)
  }
  
  if(!is.na(AvProbFadedSPDF)) {
    FadedBaseCol <- sapply(AvProbFadedSPDF@data$character, function(x) getAvCharColor(x))
    AvProbFadedSPDF@data$ColBord <- paste0(FadedBaseCol, '50')
    AvProbFadedSPDF@data$ColPoly <- paste0(FadedBaseCol, '25')
    rm(FadedBaseCol)
  }
  
  ## Dummy polygon outside plotting area for creating empty chart
  DummySpObjWKT <- "POLYGON ((-1 -1, -1 -2, -2 -2, -2 -1, -1 -1))"
  DummySpObj <- rgeos::readWKT(DummySpObjWKT, id="Dummy")
  
  ## Base chart without any blobs
  sp::plot(DummySpObj, xlim=c(0,8), ylim=c(0,8), yaxt="n", ylab="Likelihood of triggering", xaxt="n", xlab="Destructive size", cex.lab=cex.lab)
  axis(1, at=c(0,2,4,6,8), labels=c("Sz 1", "Sz 2", "Sz 3", "Sz 4", "Sz 5"), cex.axis=cex.axis)
  axis(2, at=c(0,2,4,6,8), labels=c("Unl", "Poss", "Lik", "VLik", "ACer"), cex.axis=cex.axis)
  title(Title)
  abline(h = c(0:8), lty=3, lwd=1, col="dark grey")
  abline(v = c(0:8), lty=3, lwd=1, col="dark grey")
  box()
  
  ## Plotting of faded blobs
  if(!is.na(AvProbFadedSPDF)) {
    sp::plot(AvProbFadedSPDF, add=T, col=AvProbFadedSPDF@data$ColPoly, border=AvProbFadedSPDF@data$ColBord, lwd=2)
  }
  
  ## Plotting of blobs
  if(!is.na(AvProbSPDF)) {
    sp::plot(AvProbSPDF, add=T, col=AvProbSPDF@data$ColPoly, border=AvProbSPDF@data$ColBord, lwd=3)
  } else {
    text(4,4,"No avalanche problems specified!", adj=0.5, cex=cex.lab)
  }
  
  ## Plotting of points
  if (WithPoints & !is.na(AvProbSPDF)) {
    
    for(Index_AvProb in 1:nrow(AvProbSPDF)) {
      
      Points <- data.frame(x=c(AvProbSPDF@data$size_min[Index_AvProb], AvProbSPDF@data$size_typ[Index_AvProb], AvProbSPDF@data$size_max[Index_AvProb], AvProbSPDF@data$size_typ[Index_AvProb], AvProbSPDF@data$size_typ[Index_AvProb]),
                           y=c(AvProbSPDF@data$like_typ[Index_AvProb], AvProbSPDF@data$like_max[Index_AvProb], AvProbSPDF@data$like_typ[Index_AvProb], AvProbSPDF@data$like_min[Index_AvProb], AvProbSPDF@data$like_typ[Index_AvProb]))
      
      
      points(Points, pch=19, cex=1.25, col=AvProbSPDF@data$ColBord[Index_AvProb])
      
    }
    
  }
  
  ## Plotting of ratings
  if (!is.na(RatingCol)) {
    drawAspElevRoseDiagramInsidePlot(x=RatingLoc[1], y=RatingLoc[2], rx=RatingSize, col=RatingCol, lwd=1)
  }
  
  if (SuppressWarnings) {options(warn=0)}
  
}