#' Plot raster in hazard chart with counts or percentages of avalanche problems.
#'
#' Plot raster in hazard chart with counts or percentages of avalanche problems.
#' @param AvProblems DF with avalanche problems 
#' @param Title Title of chart. Default value is 'Hazard Chart'.
#' @param WithContours Switch whether chart should be plotted with contour lines. Default is TRUE.
#' @param Percent Switch whether plotted values should be percentages instead of counts. Default is F.
#' @param PercentInterval If percentage values are plotted with contour lines, this parameter specifies the contour intervals.
#' @param Shape For specifying the shape of the blob. Values can be 'ellipse' (default) or 'square'. 
#' @param cex.AdditionalText Scalling factor for additional text. Default value is 0.66.
#' @param NumPoints Number of point per quater section of ellipse. Default value is 50.
#' @param SuppressWarnings Switch for suppressing warning message about application of is.na to S4 object. Default is TRUE. 
#' 
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "NEH"
#' 
#' ## All problems for entire season as squares
#' DateStart <- "2016-01-01"
#' DateEnd   <- "2016-04-01"
#' HzdAssessSeason <- getInfoExHzdAssessObject(Operation, DateStart=DateStart, DateEnd=DateEnd, Time = "PM")
#' plotHzdChart(HzdAssessSeason$AvProb, Title=paste("All Av Problems - Season 2016 (PM)"), Shape = "square")
#' plotHzdChartCount(HzdAssessSeason$AvProb, Title=paste("All Av Problems - Season 2016 (PM)"), Shape = "square", WithContours=F)
#' plotHzdChartCount(HzdAssessSeason$AvProb, Title=paste("All Av Problems - Season 2016 (PM)"), Shape = "square", Percent=T)
#' plotHzdChartCount(HzdAssessSeason$AvProb, Title=paste("All Av Problems - Season 2016 (PM)"), Shape = "ellipse", Percent=T)
#' 
## Specific avalanche problems for entire season as squares
#' plotHzdChartCount(HzdAssessSeason$AvProb[HzdAssessSeason$AvProb$character=="WIND_SLAB",], Title=paste("Wind Slabs - Season 2016 (PM)"), Shape = "square", Percent=T)
#' plotHzdChartCount(HzdAssessSeason$AvProb[HzdAssessSeason$AvProb$character=="STORM_SLAB",], Title=paste("Wind Slabs - Season 2016 (PM)"), Shape = "square", Percent=T)
#' plotHzdChartCount(HzdAssessSeason$AvProb[HzdAssessSeason$AvProb$character=="PERSISTENT_SLAB",], Title=paste("Pers Slabs - Season 2016 (PM)"), Shape = "square", Percent=T)
#' plotHzdChartCount(HzdAssessSeason$AvProb[HzdAssessSeason$AvProb$character=="DEEP_PERSISTENT_SLAB",], Title=paste("Pers Slabs - Season 2016 (PM)"), Shape = "square", Percent=T)
#' 
#' @export

plotHzdChartCount <- function(AvProblems, Title="Hazard Chart", WithContours=T, Percent=F, PercentInterval=5, Shape='ellipse', NumPoints=50, cex.AdditionalText=0.66, SuppressWarnings=T) {
  
  Shape <- toupper(Shape)
  
  ## Convert avalanche problems to SPDF
  AvProbSPDF <- convertAvProbToSPDF(AvProblems=AvProblems, Shape=Shape, NumPoints=NumPoints)
  
  if (SuppressWarnings) {options(warn=-1)}

  ## Create raster
  if (Shape=="SQUARE") {
    Raster <- raster(ncol=8, nrow=8, xmn=0, xmx=8, ymn=0, ymx=8)
    RasterCoord <- xyFromCell(Raster, cell=c(1:64))
  } else if (Shape=="ELLIPSE") {
    Raster <- raster(ncol=40, nrow=40, xmn=0, xmx=8, ymn=0, ymx=8)
    RasterCoord <- xyFromCell(Raster, cell=c(1:1600))
  } 
  
  RasterPoints <- data.frame(x=RasterCoord[,1],
                             y=RasterCoord[,2],
                             count=0,
                             percent=0)    
  
  ## Counting of blobs in cell
  if(!is.na(AvProbSPDF)) {

    for (Index_AvProb in 1:nrow(AvProbSPDF)) {
      for (Index_Points in 1:nrow(RasterPoints)) {
        RasterPoints$count[Index_Points] <- ifelse(sp::point.in.polygon(RasterPoints$x[Index_Points],
                                                                        RasterPoints$y[Index_Points],
                                                                        AvProbSPDF@polygons[[Index_AvProb]]@Polygons[[1]]@coords[,1],
                                                                        AvProbSPDF@polygons[[Index_AvProb]]@Polygons[[1]]@coords[,2])==1,
                                                   RasterPoints$count[Index_Points]+1, RasterPoints$count[Index_Points])
      }
    }
    
    RasterPoints$percent <- RasterPoints$count/nrow(AvProbSPDF)*100
    
    if (Percent==F) {
      values(Raster) <- RasterPoints$count
    } else {
      RasterPoints$percent <- RasterPoints$count/nrow(AvProbSPDF)*100
      values(Raster) <- RasterPoints$percent
    }
        
  }
  
  ## Base chart without any blobs
  if(Percent) {
    zLabels <- seq(0, 100, by=20)
    sp::plot(Raster, xlim=c(0,8), ylim=c(0,8), breaks=seq(0, 100, by=1), col=rev(heat.colors(100)), yaxt="n", ylab="Likelihood of triggering", xaxt="n", xlab="Destructive size", axis.args=list(at=zLabels, labels=zLabels))
  } else {
    sp::plot(Raster, xlim=c(0,8), ylim=c(0,8), yaxt="n", ylab="Likelihood of triggering", xaxt="n", xlab="Destructive size")
  }
  title(Title)
  abline(h = c(0:8), lty=2, lwd=0.5, col="dark grey")
  abline(v = c(0:8), lty=2, lwd=0.5, col="dark grey")
  box()
  axis(1, at=c(0,2,4,6,8), labels=c("Sz 1", "Sz 2", "Sz 3", "Sz 4", "Sz 5"))
  axis(2, at=c(0,2,4,6,8), labels=c("Unl", "Poss", "Lik", "VLik", "ACer"))
  
  mtext(paste("Total number of avalanche problems:", nrow(AvProbSPDF)), side=3, adj=0, cex=cex.AdditionalText)
  if(Percent) {
    mtext(("Percent"), side=3, adj=1, cex=cex.AdditionalText)
  } else {
    mtext(("Counts"), side=3, adj=1, cex=cex.AdditionalText)
  }
  
  ## Contour
  if(WithContours) {
    if(Percent) {
      contour(Raster, add=T, zlim=c(0,100), levels=seq(0, 100, by=PercentInterval))
    } else {
      contour(Raster, add=T)
    }
  }
  
  ## Error message
  if(is.na(AvProbSPDF)) {
    text(5,5,"No avalanche problems specified!", adj=0.5)
  }
  
  if (SuppressWarnings) {options(warn=0)}
  
}