#' Calculate frequencies of answers.
#'
#' Calculate frequencies of answers.
#' @param FreqTbl Frequency table with run characteristics (per SOM group or all).
#' @param FreqTbl_Ref Reference frequency table with run characteristics of all runs  (default=NULL).
#' @param plotQuestCodes Question code to be plotted.  
#' @param plotElev Elevations to be plotted (default=c(NA, "all", "btl", "tl", "alp")). 
#' @param nRuns Number of runs in each group (used for labeling on y-axis, default=NULL).
#' @param color Color for bars (default="red").
#' @param alpha Alpha value for transparency of bars (default=0.4).
#' @param barwidth Width of bars (default=0.9).
#' @param plotTitle Title to be plotted (default=NULL).
#' @param plotBox Switch for plotting a box around the plot area (default=F).
#' @examples 
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' QuestCode <- "familiarity"
#' RunChar <- getRecordsFromUUID(Operation, "runchar", "assessments")
#' QuestDef <- getRecordsFromQuery(Operation, "SELECT * FROM runchar.questions")
#' LineLevels <- convertArrayFromSQL(QuestDef[QuestDef$code==QuestCode,]$linelevels,Numeric=F)
#' ElevLevels <- convertArrayFromSQL(QuestDef[QuestDef$code==QuestCode,]$elevlevels,Numeric=F)
#' QuestEnum <- getRecordsFromQuery(Operation, paste0("SELECT * FROM runchar.enum_", QuestCode))
#' RunCharMinRelevantSkiLine <- getRunCharMinRelevantSkiLine(Operation=Operation)
#' QuestColNames <- getRunCharQuestColNames(QuestCode=QuestCode, Operation=Operation, AssessTblFormatted=F, ElevLevels=ElevLevels)
#' RunCharCompact <- RunChar[,c("uuid", "loccatskiruns_uuid","loccatskilines_uuid", QuestColNames)]
#' RunCharCompact <- formatRunCharAssessmentDF(Operation, RunCharCompact)
#' 
#' FreqTbl <- getRunCharAnswerFreqTbl(RunCharCompact, transformToMinimalLine=T, Operation=Operation, QuestCode=QuestCode, QuestDef=QuestDef, RunCharMinRelevantSkiLine=RunCharMinRelevantSkiLine, QuestEnum=QuestEnum)
#' 
#' plotRunCharFreq(FreqTbl=FreqTbl, plotQuestCodes=QuestCode, color="darkgreen", alpha=0.9)
#' 
#' @export

plotRunCharFreq <- function(FreqTbl, 
                            FreqTbl_Ref=NULL, 
                            plotQuestCodes, 
                            plotElev=c(NA, "all", "btl", "tl", "alp"), 
                            nRuns=NULL, 
                            color="red", 
                            alpha=0.4, 
                            barwidth=0.9, 
                            plotTitle=NULL, 
                            plotBox=F){
  
  if(!(plotQuestCodes %in% FreqTbl$Quest)){
    stop("Specified 'plotQuestCodes' not found in data frame provided")
  }
  
  ## SUBSET PLOT DF TO PLOT QUESTION
  ## *******************************
  plotDF <- FreqTbl[(FreqTbl$Quest %in% plotQuestCodes & FreqTbl$Elev %in% plotElev),]
  if(!is.null(FreqTbl_Ref)){
    plotDF_Ref <- FreqTbl_Ref[(FreqTbl_Ref$Quest %in% plotQuestCodes & FreqTbl_Ref$Elev %in% plotElev),]
  } else {
    plotDF_Ref <- NULL
  }
  
  # if (plotQuestCodes=="usab_oprole"){
  # 
  #   plotSeq <- data.frame("Answ"=c("safe", "bread", "jump", "lunch", "time", "destination",  "signature", "openseason", "rarely", "notpref"),
  #                         "order"=1:10)
  # 
  #   plotDF$Answ <- as.character(plotDF$Answ)
  #   plotDF <- merge(plotSeq, plotDF,  by="Answ", all.x=T)
  #   plotDF <- plotDF[order(plotDF$order),]
  #   plotDF <- plotDF[,!(names(plotDF) %in% c("order"))]
  # 
  #   if(!is.null(plotDF_Ref)){
  #     plotDF_Ref$Answ <- as.character(plotDF_Ref$Answ)
  #     plotDF_Ref <- merge(plotSeq, plotDF_Ref,by="Answ", all.x=T)
  #     plotDF_Ref <- plotDF_Ref[order(plotDF_Ref$order),]
  #     plotDF_Ref <- plotDF_Ref[,!(names(plotDF_Ref) %in% c("order"))]
  #   }
  # }
  
  ## DETERMINE NUMBER OF CLUSTERS IN FREQ TBL PROVIDED
  ## **************************************************
  skipColNames <- c("Answ", "Quest", "Elev")
  nClust <- ncol(plotDF[,!(names(plotDF) %in% skipColNames),drop=F])
  
  ## SET UP EMPTY PLOT
  ## *****************
  par(mar=c(6,4,6,4))
  bw <- barwidth
  plot(NULL, xlim=c(bw/2,nrow(plotDF)+bw/2), ylim=c(0.2+nClust+bw/2, bw/2), axes=F, xaxs="i", yaxs="i", xlab="", ylab="")
  abline(v=seq(1,nrow(plotDF),by=1), lty=3, col="grey")
  
  
  
  # if(!is.null(plotDF_Ref)){
  #   plot(NULL, xlim=c(bw/2,nrow(plotDF)+bw/2), ylim=c(0.2+nClust+bw/2+1, bw/2), axes=F, xaxs="i", yaxs="i", xlab="", ylab="")
  #   abline(v=seq(1,nrow(plotDF)+1,by=1), lty=3, col="grey")
  #   
  #   for(row in 1:nrow(plotDF_Ref)){
  #     rect(row-bw/2, 1+bw/2, row+bw/2, bw/2-0.5-plotDF_Ref[row,c("Freq")]/100, col="grey", border=NA)
  #   }
  #   for(col in 1:nClust){
  #     for(row in 1:nrow(plotDF)){
  #       rect(row-bw/2, 1+col+0.5, row+bw/2,1+col+0.5-plotDF[row,col+length(skipColNames)]/100, col=addAlphaToColor(color, alpha), border=NA)
  #     }
  #   }
  #   axis(1, plotDF$Answ, at=1:nrow(plotDF), las=2, tick = F)
  #   axis(2, at=2:(nClust+1), 1:nClust, las=1, tick = F)
  #   
  #   ## ADD NUMBER OF RUNS IN GROUP
  #   if(!is.null(nRuns)){
  #     axis(4, paste0("n=",nRuns), at=2:(1+length(nRuns)), las=1, cex.axis=1, tick = F)
  #   }
  #   
  # }
  # abline(h=seq(0, nClust+1,bw*0.5), lty=3, col="grey")
  
  ## ADD DATA TO PLOT
  ## ****************
  
  ## ADD ALL
  ## *******
  
  if ("All" %in% names(plotDF)){
    for(row in 1:nrow(plotDF)){
      xleft <- row-bw/2
      xright <- row+bw/2
      ybottom <- 1.5
      # ytop <- 1.5 - (plotDF[row, 1+length(skipColNames)] / sum(plotDF[,1+length(skipColNames)]))
      ytop <- 1.5 - plotDF[row, 1+length(skipColNames)]
      rect(xleft=xleft, xright=xright, ybottom=ybottom, ytop=ytop, col=addAlphaToColor("grey", alpha), border=NA)
    }
  }
  
  startCol <- 0
  if ("All" %in% names(plotDF)) {startCol <- startCol + 1}
  plotDFCompact <- plotDF[,!(names(plotDF) %in% c("Answ", "Quest", "Elev","All"))]
  
  for(col in 1:ncol(plotDFCompact)){
    for(row in 1:nrow(plotDF)){
      if(!is.null(plotDF_Ref)){
        rect(row-bw/2, col+0.5, row+bw/2, col+0.5-plotDF_Ref[row,c("Freq")]/100, col="grey", border=NA)
      }
      
      xleft   <- row-bw/2
      xright  <- row+bw/2
      ybottom <- startCol+col+0.5
      # ytop    <- startCol+col+0.5 - (plotDFCompact[row,col] / sum(plotDFCompact[,col]))
      ytop    <- startCol+col+0.5 - plotDFCompact[row,col]
      
      rect(xleft=xleft, xright=xright, ybottom=ybottom, ytop=ytop, col=addAlphaToColor(color, alpha), border=NA)
      
    }
  }
  
  ## ADD AXES
  ## ********
  axis(1, plotDF$Answ, at=1:nrow(plotDF), las=2, tick = F)
  if ("All" %in% names(plotDF)){
    axis(2, at=1:nClust, c("All", 1:ncol(plotDFCompact)), las=1, tick = F) 
  } else if (nClust > 1){
    axis(2, at=1:nClust, 1:nClust, las=1, tick = F) 
  } else {
    axis(2, c(0.5,1.5), c(100,0), las=1)
    mtext(2,text = "Percent", line=1)
  }
  
  
  ## ADD HORIZONTAL LINES BETWEEN CLUSTERS
  ## *************************************
  ElevLevels <- unique(plotDF$Elev)
  nAnsw_ElevLevels <- table(plotDF$Elev)
  if (!is.na(ElevLevels[1])){
    for (elev in 1:length(ElevLevels)){
      if (elev==1){
        x0 <- rep(bw/2, nClust)
        x1 <- x0 + nAnsw_ElevLevels[elev]
        y0 <- (1:nClust)+0.5
        y1 <- y0
        segments(x0=x0, y0=y0, x1=x1, y1=y1)
      } else {
        x0 <- rep(sum(nAnsw_ElevLevels[1:(elev-1)]), nClust)
        x1 <- x0 + nAnsw_ElevLevels[elev]
        x0 <- x0 + 1 - bw/2
        x1 <- x1 + bw/2
        y0 <- (1:nClust)+0.5
        y1 <- y0
        segments(x0=x0, y0=y0, x1=x1, y1=y1)
      }
    }
  } else {
    abline(h=seq(1+0.5,nClust+0.5,by=1))
  }
  
  ## ADD ELEVATION BAND BARS OUTSIDE PLOT
  ## ************************************
  ElevLevels <- unique(plotDF$Elev)
  nAnsw_ElevLevels <- table(plotDF$Elev)
  if (!is.na(ElevLevels[1])){
    for (elev in 1:length(ElevLevels)){
      nAnsw <- nrow(plotDF[plotDF$Elev==ElevLevels[elev],])
      if (elev==1){
        x0 <- 1 #bw/2
        x1 <- x0 + nAnsw_ElevLevels[elev] - 1
        axis(3,at=c(x0,x1),col="black",line=0.5,tick=T,labels=rep("",2),lwd=2,lwd.ticks=0)
        mtext(ElevLevels[elev], adj=0.5*nAnsw_ElevLevels[elev] / sum(nAnsw_ElevLevels), line=1.2)
      } else {
        x0 <- sum(nAnsw_ElevLevels[1:(elev-1)])
        x1 <- x0 + nAnsw_ElevLevels[elev]
        x0 <- x0 + 1 #- bw/2
        x1 <- x1 #+ bw/2
        axis(3,at=c(x0,x1),col="black",line=0.5,tick=T,labels=rep("",2),lwd=2,lwd.ticks=0)
        mtext(ElevLevels[elev], adj=(sum(nAnsw_ElevLevels[1:(elev-1)]) + 0.5*nAnsw_ElevLevels[elev]) / sum(nAnsw_ElevLevels), line=1.2)
      }
    }
  }
  
  ## ADD NUMBER OF RUNS IN GROUP
  ## *****************************
  if(!is.null(nRuns)){
    axis(4, paste0("n=",nRuns), at=1:length(nRuns), las=1, cex.axis=1, tick = F)
  }
  
  
  
  ## ADD TITLE
  ## *********
  if(!is.null(plotTitle)){
    mtext(plotTitle, side=3, line=3, font=2, cex=1.2)
  }
  
  ## ADD BOX
  ## *******
  if (plotBox){box()}
}