#' Plots subflower plot with mean and range for ordinal run characterization variables.
#'
#' Plots subflower plot with mean and range for ordinal run characterization variables.
#' @param Tbl Intput data frame.
#' @param sort Type of sorting. Options include alphabetical (NA, default), assessment mean ("mean") or assessment range ("range"), guide assessment ("guide")
#' @param groupLines Switch for specifying whether line classes should be grouped. Default is FALSE.
#' @param UserID UserId of survey participant whose assessments are plotted on top of chart
#' @param withSunflowers Switch for whether sunflowers shoud be added (default is T)
#' @param vertical Switch for turning chart vertical (Default is FALSE).
#' @param col.brewer.name Name of the RColorBrewer color scale. Default is 'YlOrRd'
#' @param main Title of plot
#' @param xlab.show Switch for showing xlab. default is True.
#' @param xlab Label of x axis
#' @param ylab Label of y axis
#' @param xlab.line Distance of x axis label to axis in number of lines as used by mtext. Default is 9
#' @param xlab.line Distance of y axis label to axis in number of lines as used by mtext. Default is 3.5
#' 
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Survey <- getRunCharAssessmentsSurvey("NEH")
#' Tbls <- convertRunCharRaw2Long(Survey)
#' 
#' par(mar = c(13.1,7.1,4.1,2.1))
#' plotRunCharSunflower(Tbl=Tbls$TblLine, ColChar = "hzd_steep")
#' plotRunCharSunflower(Tbl=Tbls$TblLine, ColChar = "hzd_steep", sort = "mean", col.brewer.name = "Greens")
#' plotRunCharSunflower(Tbl=Tbls$TblLine, ColChar = "hzd_steep", sort = "range", col.brewer.name = "Blues")
#' 
#' ## Plots assessments of user 2 on top
#' plotRunCharSunflower(Tbl=Tbls$TblLine, ColChar = "hzd_steep", sort = "mean", col.brewer.name = "Blues", UserID = 2)
#' plotRunCharSunflower(Tbl=Tbls$TblLine, ColChar = "hzd_steep", sort = "guide", col.brewer.name = "Blues", UserID = 2)
#' 
#' @export

plotRunCharSunflower <- function(Tbl, ColChar, ColLabel = "Label",
                                 sort = NA,
                                 groupLines = FALSE,
                                 UserID = NA,
                                 withSunflowers = TRUE,
                                 vertical = FALSE,
                                 col.brewer.name = "YlOrRd",
                                 col.above = "#e66101",
                                 col.below = "#5e3c99",
                                 main = "Sunflower plot of run characteristic",
                                 xlab.show = TRUE,
                                 xlab = "Run/Line",
                                 ylab = paste0("Characteristic (", ColChar, ")"),
                                 xlab.line = 9,
                                 ylab.line = 3.5) {
  
  ## Check if Line column exists
  if ("Line" %in% names(Tbl)) {
    groupLines <- groupLines
  } else {
    groupLines <- FALSE
  }
  
  ## Pass label and char columns
  Tbl$X <- Tbl[,ColLabel]
  Tbl$Y <- Tbl[,ColChar]
  
  Col <- RColorBrewer::brewer.pal(4, col.brewer.name)
  
  ## Calculating mean statistics
  Mean <- aggregate(as.numeric(Tbl$Y),
                    by = list(Tbl$X), FUN = mean, na.rm = T)
  names(Mean) <- c("Label", "mean")
  
  ## Check whether any runs/lines with no values
  if (sum(is.nan(Mean$mean)) > 0) {
    warning(paste(sum(is.nan(Mean$mean)), "items removed because no assessment values:", paste0(Mean$Label[is.nan(Mean$mean)], collapse = ", ")))
  }
  
  LabelWithValues <- Mean$Label[!is.nan(Mean$mean)]
  Tbl <- Tbl[Tbl$X %in% LabelWithValues,]
  
  ## Calculating max statistics
  Max <- aggregate(as.numeric(Tbl$Y),
                   by = list(Tbl$X), FUN = max, na.rm = T)
  names(Max) <- c("Label", "max")
  
  ## Calculating min statistics
  Min <- aggregate(as.numeric(Tbl$Y),
                   by = list(Tbl$X), FUN = min, na.rm = T)
  names(Min) <- c("Label", "min")
  
  ## Calculate frequencies
  Freq <- as.data.frame.matrix(table(Tbl$X,
                                     as.numeric(Tbl$Y)))
  Freq$Label <- rownames(Freq)
  
  Stats <- merge(Mean, Max)
  Stats <- merge(Stats, Min)
  Stats <- merge(Stats, Freq)
  Stats$range <- Stats$max - Stats$min
  rm(Mean, Max, Min)
  
  ## Add user ratings
  if (!is.na(UserID)) {
    TblUser <- Tbl[Tbl$user_id == UserID, c("Label", "user_id", "Y")]
    names(TblUser) <- c("Label", "user_id", "guide")
    Stats <- merge(Stats, TblUser)
    Stats$guidediff <- as.numeric(Stats$guide) - Stats$mean
  }
  
  ## Extract line information
  if (groupLines) {
    Stats$Line <- sapply(Stats$Label, function(x) strsplit(as.character(x), " - ")[[1]][2])
  }
  
  ## Sort run names (might need to be changed)
  if (is.na(sort)) {
    ## Do nothing
  } else if (sort == "mean") {
    if (length(levels(Tbl$Y)) == 2) {
      Stats <- Stats[order(Stats$mean),]
    } else if (length(levels(Tbl$Y)) == 3) {
      Stats <- Stats[order(Stats$mean, Stats$`3`, Stats$`2`),]
    } else {
      Stats <- Stats[order(Stats$mean, Stats$`4`, Stats$`3`, Stats$`2`),]
    }
  } else if (sort == "range") {
    Stats <- Stats[order(Stats$range, -Stats$mean),]
  } else if (sort == "guide") {
    Stats <- Stats[order(Stats$guide, Stats$mean),]
  } else if (sort == "diff") {
    Stats <- Stats[order(Stats$guidediff),]
  } else if (sort == "diffabs") {
    Stats <- Stats[order(abs(Stats$guidediff)),]
  }
  
  ## group by line
  if (groupLines) {
    Stats <- Stats[order(Stats$Line),]
  }
  
  ## Turn run names into factors to accomodate sorting
  Tbl$X <- factor(Tbl$X, levels = Stats$Label)
  Stats$Label <- factor(Stats$Label, levels = Stats$Label)
  

  ## HORIZONTAL CHART
  ## ****************
  if (!vertical) {
    
    ## Base plot
    plot(NULL, xlim = c(1,length(levels(Tbl$X))), ylim = c(1,length(levels(Tbl$Y))), xaxt = "n", yaxt = "n", xlab = "", ylab = "")
    
    if (xlab.show) {
      axis(side = 1, at = c(1:length(levels(Tbl$X))), labels = levels(Tbl$X), las = 2)
      
      if ("Line" %in% names(Tbl)) {
        axis(side = 1, at = which(grepl("Line 1", levels(Tbl$X))), labels = levels(Tbl$X)[which(grepl("Line 1", levels(Tbl$X)))], col.axis = "#238b45", las = 2)
        axis(side = 1, at = which(grepl("Line 2", levels(Tbl$X))), labels = levels(Tbl$X)[which(grepl("Line 2", levels(Tbl$X)))], col.axis = "#2171b5", las = 2)
      }
      
      mtext(xlab, side = 1, line = xlab.line)
    } else {
      axis(side = 1, at = c(1:length(levels(Tbl$X))), labels = FALSE)
    }
    
    axis(side = 2, at = c(1:length(levels(Tbl$Y))), labels = levels(Tbl$Y), las = 1) 
    mtext(ylab, side = 2, line = ylab.line)
    
    title(main)
    
    ## Range
    for (i in 1:nrow(Stats)) {
      rect(xleft = i - 0.4, ybottom = Stats$min[i], xright = i + 0.4, ytop = Stats$max[i], col = Col[1], border = NA)
    }
    
    ## Grid
    grid(nx = NA, ny = NULL)
    abline(v = c(1:length(levels(Tbl$X))), lwd = 0.5, lty = 3,
           col = "lightgrey")
    
    ## Adding guide difference
    if (!is.na(UserID)) {
      
      Pos <- Stats[Stats$guidediff > 0.5,]
      for (i in 1:nrow(Pos)) {
        lines(x = rep(as.numeric(Pos$Label[i]), 2), y = c(Pos$mean[i], Pos$guide[i]), col = col.above, lwd = 3)
      }
      
      Neu <- Stats[(Stats$guidediff <= 0.5) & (Stats$guidediff >= -0.5),]
      for (i in 1:nrow(Neu)) {
        lines(x = rep(as.numeric(Neu$Label[i]), 2), y = c(Neu$mean[i], Neu$guide[i]), col = " dark grey", lwd = 3)
      }
      
      Neg <- Stats[Stats$guidediff < -0.5,]
      for (i in 1:nrow(Neg)) {
        lines(x = rep(as.numeric(Neg$Label[i]), 2), y = c(Neg$mean[i], Neg$guide[i]), col = col.below , lwd = 3)
      }
      
      rm(Pos, Neg, Neu)
    }
    
    ## Sunflower plot
    if (withSunflowers) {
      sunflowerplot(Y ~ X, data = Tbl, 
                    col = Col[3], seg.lwd = 2, seg.col = Col[3], size = 1/16,
                    cex.fact = 1, add = T)
    }
    
    ## Mean
    for (i in 1:nrow(Stats)) {
      lines(x = c(i - 0.4, i + 0.4), y = rep(Stats$mean[i], 2), col = Col[4], lwd = 2)
    }
    
    ## Guide rating
    if (!is.na(UserID)) {
      points(x = Stats$Label, y = Stats$guide, pch = 16)
      mtext(text = paste0("User ID: ", UserID), adj = 0)
    }
    
  ## VERTICAL CHART
  ## **************
  } else {
    
    ## Base plot
    plot(NULL, ylim = c(1, length(levels(Tbl$X))), xlim = c(1, length(levels(Tbl$Y))), yaxt = "n", xaxt = "n", ylab = "", xlab = "")
    
    if (xlab.show) {
      axis(side = 2, at = c(1:length(levels(Tbl$X))), labels = levels(Tbl$X), las = 2)
      
      if ("Line" %in% names(Tbl)) {
        axis(side = 2, at = which(grepl("Line 1", levels(Tbl$X))), labels = levels(Tbl$X)[which(grepl("Line 1", levels(Tbl$X)))], col.axis = "#238b45", las = 2)
        axis(side = 2, at = which(grepl("Line 2", levels(Tbl$X))), labels = levels(Tbl$X)[which(grepl("Line 2", levels(Tbl$X)))], col.axis = "#2171b5", las = 2)
      }
      
      mtext(xlab, side = 2, line = xlab.line)
    } else {
      axis(side = 2, at = c(1:length(levels(Tbl$X))), labels = FALSE)
    }
    
    axis(side = 3, at = c(1:length(levels(Tbl$Y))), labels = levels(Tbl$Y), las = 1)
    axis(side = 1, at = c(1:length(levels(Tbl$Y))), labels = levels(Tbl$Y), las = 1)
    mtext(ylab, side = 1, line = ylab.line)
    
    title(main)
    
    ## Range
    for (i in 1:nrow(Stats)) {
      rect(ybottom = i - 0.4, xleft = Stats$min[i], ytop = i + 0.4, xright = Stats$max[i], col = Col[1], border = NA)
    }
    
    ## Grid
    grid(nx = NULL, ny = NA)
    abline(h = c(1:length(levels(Tbl$X))), lwd = 0.5, lty = 3,
           col = "lightgrey")
    
    ## Adding guide difference
    if (!is.na(UserID)) {
      
      Pos <- Stats[Stats$guidediff > 0.5,]
      for (i in 1:nrow(Pos)) {
        lines(y = rep(as.numeric(Pos$Label[i]), 2), x = c(Pos$mean[i], Pos$guide[i]), col = col.above, lwd = 3)
      }
      
      Neu <- Stats[(Stats$guidediff <= 0.5) & (Stats$guidediff >= -0.5),]
      for (i in 1:nrow(Neu)) {
        lines(y = rep(as.numeric(Neu$Label[i]), 2), x = c(Neu$mean[i], Neu$guide[i]), col = " dark grey", lwd = 3)
      }
      
      Neg <- Stats[Stats$guidediff < -0.5,]
      for (i in 1:nrow(Neg)) {
        lines(y = rep(as.numeric(Neg$Label[i]), 2), x = c(Neg$mean[i], Neg$guide[i]), col = col.below , lwd = 3)
      }
      
      rm(Pos, Neg, Neu)
    }
    
    ## Sunflower plot
    if (withSunflowers) {
      sunflowerplot(X ~ Y, data = Tbl, 
                    col = Col[3], seg.lwd = 2, seg.col = Col[3], size = 1/16,
                    cex.fact = 1, add = T)
    }
    
    ## Mean
    for (i in 1:nrow(Stats)) {
      lines(y = c(i - 0.4, i + 0.4), x = rep(Stats$mean[i], 2), col = Col[4], lwd = 2)
    }
    
    ## Guide rating
    if (!is.na(UserID)) {
      points(y = Stats$Label, x = Stats$guide, pch = 16)
      mtext(text = paste0("User ID: ", UserID), adj = 0)
    }
    
  }
  
  ## Return table
  if (!is.na(UserID)) {
    Output <- Stats[,c("Label", "mean", "max", "min", "range", "user_id", "guide")]
    names(Output)[2:7] <- paste0(ColChar, "_", names(Output)[2:7])
  } else {
    Output <- Stats[,c("Label", "mean", "max", "min", "range")]
    names(Output)[2:5] <- paste0(ColChar, "_", names(Output)[2:5])
  }
  invisible(Output)
  
}
