#' Plots subflower plot with mean and range for ordinal run characterization variables.
#'
#' Plots subflower plot with mean and range for ordinal run characterization variables.
#' @param Tbl Intput data frame.
#' @param sort Type of sorting. Options include alphabetical (NA, default), assessment mean ("mean") or assessment range ("range"), guide assessment ("guide")
#' @param YesPercForConcensus Percentage of yes answers for general question that is considered a concensus. Default is 50.
#' @param UserID UserId of survey participant whose assessments are plotted on top of chart (NOT IMPLEMENTED!).
#' @param withSunflowers Switch for whether sunflowers shoud be added. Default is T.
#' @param col.brewer.name Name of the RColorBrewer color scale. Default is 'YlOrRd'
#' @param main Title of plot
#' @param xlab.show Switch for showing xlab. default is True.
#' @param xlab Label of x axis
#' @param ylab Label of y axis
#' @param xlab.line Distance of x axis label to axis in number of lines as used by mtext. Default is 9.
#' @param xlab.line Distance of y axis label to axis in number of lines as used by mtext. Default is 3.5.
#' @param DotSizeMin Mimimum dot size. Default is 0.5.
#' @param DotSizeMax Maximum dot size. Default is 2.5.
#' 
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Survey <- getRunCharAssessmentsSurvey("NEH")
#' Tbls <- convertRunCharRaw2Long(Survey)
#' 
#' par(mar = c(13.1,7.1,4.1,2.1))
#' ## Terrain hazard
#' plotRunCharSunflowerMultiple(Tbl = Tbls$TblLine,
#'                              ColChar = "hzd_terrainhzd",
#'                              ylab.line = 5,
#'                              col.brewer.name = "Greens")
#' 
#' plotRunCharSunflowerMultiple(Tbl = Tbls$TblLine,
#'                              ColChar = "hzd_terrainhzd",
#'                              ylab.line = 5,
#'                              col.brewer.name = "Greens",
#'                              sort = "mean")
#' 
#' plotRunCharSunflowerMultiple(Tbl = Tbls$TblLine,
#'                              ColChar = "hzd_terrainhzd",
#'                              ylab.line = 5,
#'                              col.brewer.name = "Greens",
#'                              sort = "largeoh")
#' 
#' plotRunCharSunflowerMultiple(Tbl = Tbls$TblLine,
#'                              ColChar = "hzd_terrainhzd",
#'                              ylab.line = 5,
#'                              col.brewer.name = "Greens",
#'                              sort = "largeoh", 
#'                              YesPercForConcensus = 75)
#' 
#' ## Operational role
#' plotRunCharSunflowerMultiple(Tbl = Tbls$TblLine,
#'                              ColChar = "use_oprole",
#'                              ylab.line = 5,
#'                              col.brewer.name = "Blues",
#'                              sort = "mean")
#' 
#' plotRunCharSunflowerMultiple(Tbl = Tbls$TblLine,
#'                              ColChar = "use_oprole",
#'                              ylab.line = 5,
#'                              col.brewer.name = "Blues",
#'                              sort = "bread")
#' 
#' @export

plotRunCharSunflowerMultiple <- function(Tbl, ColChar, ColLabel = "Label",
                                 sort = NA,
                                 YesPercForConcensus = 50,
                                 # UserID = NA,
                                 withSunflowers = TRUE,
                                 col.brewer.name = "YlOrRd",
                                 main = "Sunflower plot of run characteristic",
                                 xlab.show = TRUE,
                                 xlab = "Run/Line",
                                 ylab = paste0("Characteristic (", ColChar, ")"),
                                 xlab.line = 9,
                                 ylab.line = 3.5,
                                 DotSizeMin = 0.5,
                                 DotSizeMax = 2.5) {
  
  ## Reduce Tbl to relevant columns
  Wide <- Tbl[, c(1, 2, grep(ColLabel, names(Tbl)), grep(ColChar,names(Tbl)))]
  
  
  ## Extract information for Y/N master question
  Long_Main <- Wide[, c(1:4)]
  Long_Main$X <- Long_Main[,ColLabel]
  Long_Main$Y <- Long_Main[,ColChar]
  
  ## Extract information for sub questions
  Wide_Add  <- Wide[Wide[,ColChar] == "yes", c(1:3, 5:(length(names(Wide)) - 1))]
  names(Wide_Add) <- gsub(pattern = paste0(ColChar, "_"), "", names(Wide_Add))
  
  Long_Add <- reshape2::melt(Wide_Add, id.vars = c("user_id", "id", "Label"))
  Long_Add <- na.omit(Long_Add[Long_Add$value == "yes",])
  
  Levels <- c(rev(levels(Long_Add$variable)), "yes", "no")
  
  Long_Add$variable <- factor(Long_Add$variable, levels = Levels)
  Long_Main$Y <- factor(Long_Main$Y, levels = Levels)
  
  ## Color scale
  Col <- RColorBrewer::brewer.pal(4, col.brewer.name)
  
  ## Calculating mean statistics
  Mean <- aggregate(as.numeric(Long_Main$Y),
                    by = list(Long_Main$X), FUN = mean, na.rm = T)
  names(Mean) <- c("Label", "mean")
  
  ## Check whether any runs/lines with no values
  if (sum(is.nan(Mean$mean)) > 0) {
    warning(paste(sum(is.nan(Mean$mean)), "items removed because no assessment values:", paste0(Mean$Label[is.nan(Mean$mean)], collapse = ", ")))
  }
  
  LabelWithValues <- Mean$Label[!is.nan(Mean$mean)]
  Long_Main <- Long_Main[Long_Main$X %in% LabelWithValues,]
  
  ## Calculating max statistics
  Max <- aggregate(as.numeric(Long_Main$Y),
                   by = list(Long_Main$X), FUN = max, na.rm = T)
  names(Max) <- c("Label", "max")
  
  ## Calculating min statistics
  Min <- aggregate(as.numeric(Long_Main$Y),
                   by = list(Long_Main$X), FUN = min, na.rm = T)
  names(Min) <- c("Label", "min")
  
  ## Merge
  Stats <- merge(Mean, Max)
  Stats <- merge(Stats, Min)
  Stats$range <- Stats$max - Stats$min
  rm(Mean, Max, Min)
  
  
  ## Add user ratings
  # if (!is.na(UserID)) {
  #   TblUser <- Tbl[Tbl$user_id == UserID, c("Label", "user_id", "Y")]
  #   names(TblUser) <- c("Label", "user_id", "guide")
  #   Stats <- merge(Stats, TblUser)
  #   Stats$guidediff <- as.numeric(Stats$guide) - Stats$mean
  # }
  
  
  ## Calculate cross tables for sub questions
  xtab_Main <- as.data.frame.matrix(table(Long_Main$X, factor(Long_Main$Y, levels = c("yes", "no"))))
  xtab_Main$Label <- rownames(xtab_Main)
  xtab_Add <- as.data.frame.matrix(table(Long_Add$Label, Long_Add$variable))
  xtab_Add$Label <- rownames(xtab_Add)
  xtab_Add <- deleteDFColumnsByName(xtab_Add, c("yes", "no"))
  
  xtab <- merge(xtab_Add, xtab_Main, by.x = "Label", by.y = "Label")
  xtab[,-1] <- xtab[,-1]/xtab$yes
  xtab <- deleteDFColumnsByName(xtab, c("yes", "no"))
  xtab <- merge(xtab, xtab_Main, by.x = "Label", by.y = "Label")
  
  ## Merges sub question percentages to Stats dataframe
  Stats <- merge(Stats, xtab, by.x = "Label", by.y = "Label")
  
  ## Percentage threshold filter
  Stats$PercentYes <- Stats$yes / (Stats$yes + Stats$no) * 100
  Stats$AboveThreshold <- ifelse(Stats$PercentYes >= YesPercForConcensus, 1, 0)
  
  ## Sort run names (might need to be changed)
  if (is.na(sort)) {
    ## Do nothing
  } else if (sort == "mean") {
    Stats <- Stats[order(Stats$mean),]
  } else if (sort == "range") {
    Stats <- Stats[order(Stats$range, -Stats$mean),]
  } else if (sort == "guide") {
    Stats <- Stats[order(Stats$guide, Stats$mean),]
  } else if (sort == "diff") {
    Stats <- Stats[order(Stats$guidediff),]
  } else if (sort == "diffabs") {
    Stats <- Stats[order(abs(Stats$guidediff)),]
  } else {
    Stats <- Stats[order(-Stats$AboveThreshold, -Stats[,sort], Stats$mean),]
  }
  
  ## Turn run names into factors to accomodate sorting
  Long_Main$X <- factor(Long_Main$X, levels = Stats$Label)
  Stats$Label <- factor(Stats$Label, levels = Stats$Label)
  xtab$Label <- factor(xtab$Label, levels = Stats$Label)
  
  ## Base plot
  plot(NULL, xlim = c(1,length(levels(Long_Main$X))), ylim = c(1,length(levels(Long_Main$Y))), xaxt = "n", yaxt = "n", xlab = "", ylab = "")
  
  if (xlab.show) {
    axis(side = 1, at = c(1:length(levels(Long_Main$X))), labels = levels(Long_Main$X), las = 2)
    mtext(xlab, side = 1, line = xlab.line)
  } else {
    axis(side = 1, at = c(1:length(levels(Long_Main$X))), labels = FALSE)
  }
  
  axis(side = 2, at = c(1:length(levels(Long_Main$Y))), labels = levels(Long_Main$Y), las = 1) 
  mtext(ylab, side = 2, line = ylab.line)
  
  title(main)
  
  ## Range
  for (i in 1:nrow(Stats)) {
    rect(xleft = i - 0.4, ybottom = Stats$min[i], xright = i + 0.4, ytop = Stats$max[i], col = Col[1], border = NA)
  }
  
  ## Grid
  abline(h = c(1:length(levels(Long_Main$Y))), lwd = 0.5, lty = 3, col = "lightgrey")
  abline(v = c(1:length(levels(Long_Main$X))), lwd = 0.5, lty = 3, col = "lightgrey")
  
  ## Adding guide difference
  # if (!is.na(UserID)) {
  #   
  #   Pos <- Stats[Stats$guidediff > 0,]
  #   for (i in 1:nrow(Pos)) {
  #     lines(x = rep(as.numeric(Pos$Label[i]), 2), y = c(Pos$mean[i], Pos$guide[i]), col = "#e66101", lwd = 3)
  #   }
  #   
  #   Neg <- Stats[Stats$guidediff < 0,]
  #   for (i in 1:nrow(Neg)) {
  #     lines(x = rep(as.numeric(Neg$Label[i]), 2), y = c(Neg$mean[i], Neg$guide[i]), col = "#5e3c99", lwd = 3)
  #   }
  #   
  #   rm(Pos, Neg)
  # }
  
  ## Sunflower plot
  if (withSunflowers) {
    sunflowerplot(Y ~ X, data = Long_Main, 
                  col = Col[3], seg.lwd = 2, seg.col = Col[3], size = 1/16,
                  cex.fact = 1, add = T)
  }
  
  ## Mean
  for (i in 1:nrow(Stats)) {
    lines(x = c(i - 0.4, i + 0.4), y = rep(Stats$mean[i], 2), col = Col[4], lwd = 2)
  }
  
  ## Guide rating
  # if (!is.na(UserID)) {
  #   
  #   points(x = Stats$Label, y = Stats$guide, pch = 16)
  #   mtext(text = paste0("User ID: ", UserID), adj = 0)
  #   
  # }
  
  ## Plot agreement of additional questions
  colRamp <- c(RColorBrewer::brewer.pal(4, col.brewer.name), "#000000")
  
  NumYesMax <- max(xtab$yes, na.rm = T)
  xtab$DotSize <- DotSizeMin + (xtab$yes/NumYesMax)*(DotSizeMax - DotSizeMin)
  
  for (i in 1:nrow(xtab)) {
    for (k in 2:ncol(xtab)) {
      if (names(xtab)[k] != "yes" & names(xtab)[k] != "no") {
        if (!is.nan(xtab[i,k]) & xtab[i,k] > 0 ) {
          if (xtab$yes[i] == 1) {
            points(x = as.numeric(xtab$Label[i]), y = k - 1, pch = 21, bg = "white", col = "grey", cex = xtab$DotSize[i])
          } else {
            points(x = as.numeric(xtab$Label[i]), y = k - 1, pch = 21, bg = colRamp[round(5*xtab[i,k], 0)], col = "black", cex = xtab$DotSize[i], lwd = 1)
          }
        }
      }
    }
  }
  
  ## Threshold lines
  if (!is.na(sort) & sort != "mean" & sort != "range" & sort != "guide" & sort != "diff" & sort != "diffabs") {
    LabelIndexThreshold <- max(which(Stats$AboveThreshold == 1))
    abline(v = LabelIndexThreshold + 0.5, lty = 2)
    mtext(text = paste0("Yes percentage for concensus: ", YesPercForConcensus, "%"), side = 3, adj = 0, cex = 0.75)
  }
  
}