#' Plots all of the GPS tracks associated with a ski run or run line from the locacation catalog.
#'
#' Plots all of the GPS tracks associated with a ski run or run line from the locacation catalog
#' @param Oper Abbreviation of operation
#' @param RunName Name of run
#' @param DEM DEM raster for background
#' @param BackgroundType Type of background image. Default is 'Terrain'. Other option is 'Slope'.
#' @param GPSTracks Optional parameter for passing GPS tracks. This is to improve efficiency if you produce multiple charts for the same GPS tracks.
#' @param LocCat Optional parameter for passing location catalog. Default is NULL, which means that the location catalog is retrieved within the function.
#' @param LocCatCentroids Optional parameter for passing location catalog with centroids. Default is NULL, which means that the location catalog is retrieved within the function.
#' 
#' @return RunsObject with GPS tracks.
#' 
#' @examples
#' require(SarpGPSToolsPrivate)
#' require(SarpGPSTools)
#' 
#' Oper <- "CMHGL"
#' 
#' DEM <- getDEM(Oper, highres=T, localpath = "C:/Temp/Test")
#' LocCat <- getLocCatlog(Oper, LocalProjection = F)
#' LocCatC <- getLocCatlog(Oper, LocalProjection = F, AsCentroidsSkiRuns = T)
#' 
#' RunName1 <- c("Bunker Hill")
#' RunName2 <- c("TNT")
#' 
#' par(mfrow=c(2,2))
#' par(mar=c(0.5,0.5,4.1,0.5))
#' GPSTracks1 <- plot.AllGPSRunsOnSkiRunSkiLine(Oper = Oper, RunName = RunName1, 
#'                                              DEM = DEM, BackgroundType = "Terrain", 
#'                                              LocCat = LocCat, LocCatCentroids = LocCatC)
#' 
#' GPSTracks2 <- plot.AllGPSRunsOnSkiRunSkiLine(Oper = Oper, RunName = RunName2, 
#'                                              DEM = DEM, BackgroundType = "Terrain", 
#'                                              LocCat = LocCat, LocCatCentroids = LocCatC)
#' 
#' plot.AllGPSRunsOnSkiRunSkiLine(Oper = Oper, RunName = RunName1, 
#'                                DEM = DEM, BackgroundType = "Slope", 
#'                                GPSTracks = GPSTracks1,
#'                                LocCat = LocCat, LocCatCentroids = LocCatC)
#' 
#' plot.AllGPSRunsOnSkiRunSkiLine(Oper = Oper, RunName = RunName2, 
#'                                DEM = DEM, BackgroundType = "Slope", 
#'                                GPSTracks = GPSTracks2,
#'                                LocCat = LocCat, LocCatCentroids = LocCatC)
#' par(mfrow=c(1,1))
#' par(mar=c(5.1,4.1,4.1,2.1))
#' 
#' @export
#' 

plot.AllGPSRunsOnSkiRunSkiLine <- function(Oper, RunName, DEM, BackgroundType="Terrain", GPSTracks=NULL, LocCat=NULL, LocCatCentroids=NULL) {
  
  ## Check
  if (!(RunName %in% LocCat$SkiRuns$name) & !(RunName %in% LocCat$SkiLines$name)) {
    stop (paste0("Run '", RunName, "' does not exist!"))
  }
  BackgroundType <- substr(tolower(BackgroundType), 1, 4)
  
  ## Get loccation catalog if necessary
  if (is.null(LocCat)) LocCat <- getLocCatlog(Oper, LocalProjection = F)
  if (is.null(LocCatCentroids)) LocCatCentroids <- getLocCatlog(Oper, LocalProjection = F, AsCentroidsSkiRuns = T)
  
  ## Get GPS tracks
  if (is.null(GPSTracks)) {
    GPSUUID <- getGPSRunsUUIDsFromLoccatSkiRunLineUUIDsNames(Oper, Names = RunName)
    GPSTracks <- getRunsObjectFromUUIDs(Oper, GPSUUID, WithObsPoints = F, LocalProjection = F)
  }
    
  ## Plot background
  if (BackgroundType=="terr") {
    plot.LocCatalog(LocCat, ZoomArea = RunName, DEM=DEM, Elev.Cont=T, main=paste0("All GPS tracks on ", RunName, " (Elev)"))
  } else {
    plot.LocCatalog(LocCat, ZoomArea = RunName, DEM=DEM, Elev.Cont=T, Type="slope", Slp.Cont=T, main=paste0("All GPS tracks on ", RunName, " (Slope)"))  
  }
  
  ## Plot tracks
  plot(GPSTracks$Runs, add=T, col="#2b8cbe", lwd=1)
  plot(GPSTracks$PausePoints[GPSTracks$PausePoints$activity=="Waiting",], pch=8, col="#fc9272", cex=0.5, add=T)
  plot(GPSTracks$PausePoints[GPSTracks$PausePoints$activity=="Waiting" & GPSTracks$PausePoints$timediff_sec > 30,], pch=8, col="#de2d26", cex=0.5, add=T)
  plot(GPSTracks$PausePoints[GPSTracks$PausePoints$activity=="GettingReady",], pch=21, bg="orange", add=T)
  plot(GPSTracks$PausePoints[GPSTracks$PausePoints$activity=="PackingUp",], pch=21, bg="purple", add=T)
  
  # if (!is.null(LocCatCentroids$SkiRuns) > 0) text(LocCatCentroids$SkiRuns, LocCatCentroids$SkiRuns$name)
  # if (!is.null(LocCatCentroids$SkiLines) > 0) text(LocCatCentroids$SkiLines, LocCatCentroids$SkiLines$name)
  
  ## return
  invisible(GPSTracks)
  
}
