#' Plots LocCatalog object.
#'
#' Plots LocCatalog object.
#' @param LocCatalogObject LocCatalog object
#' @param main Title for plot. Default is "GPS Tracks of Ski Runs".
#' @param ZoomArea Name or names of an area (e.g., zones, ski run) that the map should focus on. Default is NA , which zooms into the Operating area
#' @param DEM Raster DEM with elevation values for plotting terrain background
#' @param ShowZones Switch for showing zones. Default is TRUE.
#' @param ShowCatRoads Switch for showing cat roads. Default is TRUE.
#' @param ShowSkiRuns Switch for showing ski runs. Default is TRUE.
#' @param ShowSkiRunSegments Switch for showing ski run segements. This only works for NEH. Default is FALSE.
#' @param ShowSkiLines Switch for showing ski lines. Default is TRUE.
#' @param ShowWxSite Switch for showing weather sites. Default is TRUE.
#' @param ShowLandings Switch for showing helicopter landings and pickups. Default is TRUE.
#' @param ShowCruxes Switch for showing cruxes. Default is TRUE.
#' @param ShowClusters Switch for showing run clusters. Default is FALSE.
#' 
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "CMHGL"
#' LocCat <- getLocCatlog(Operation)
#' plot(LocCat, main='CMH Galena')
#' 
#' ## Include terrain
#' DEM <- getDEM(Operation)
#' plot.LocCatalog(LocCat, DEM=DEM, WithTerrainCol = F) ## Grey scale in 500m intervals
#' plot.LocCatalog(LocCat, DEM=DEM)  ## Terrain coloring
#' 
#' Operation <- "MPS"
#' LocCat <- getLocCatlog(Operation, AsCentroidsSkiRuns = T, AsCentroidsSkiLines = T)
#' plot(LocCat, main='Monashee Powder')
#' 
#' Operation <- "NEH"
#' LocCat <- getLocCatlog(Operation)
#' plot(LocCat, ZoomArea=c("Promised Land", "Wild West"))
#' 
#' @export

plot.LocCatalog <- function(LocCatalog, main="Location Catalog", ZoomArea=NA, DEM=NULL, WithTerrainCol=T, WithContours=F, ShowZones=T, ColZones="grey", ShowCatRoads=T, ColCatRoads="brown", ShowSkiRuns=T, ColSkiRuns="#e41a1c", ShowSkiRunSegments=FALSE, ColSkiRunSegments=c("#a1d99b", "#9ecae1", "#bdbdbd"), TransSkiRunSegments=85, ShowSkiLines=T, ColSkiLines="#e41a1c", ShowWxSites=T, ColWxSites="skyblue", ShowLandings=T, ColLandings=c("red", "blue"), ShowCruxes=T, ColCruxes="red", ShowClusters=F, ColClusters="grey", ...) {

  ## Initialize flags
  ZoomLevel = "Area"
  
  
  if (is.na(ZoomArea[1])) {
  
    ## Base plot  
    plot(LocCatalog$OpArea, main=main, col="white", border="white")
    
  } else {
    
    Check <- sum(LocCatalog$Zones$name %in% ZoomArea)
    
    if (Check>0) {
      
      ZoomLevel = "Zone"
      
      Which <- which(LocCatalog$Zones$name %in% ZoomArea)
      ZoomAreaSP <- LocCatalog$Zones[Which,]
      
      plot(ZoomAreaSP, col="white", border="white", main=main)
      
    } else {
      
      Check <- sum(LocCatalog$Clusters$name %in% ZoomArea)
      
       if (Check>0) {
         
         ZoomLevel = "Cluster"
        
         Which <- which(LocCatalog$Clusters$name %in% ZoomArea)
         ZoomAreaSP <- LocCatalog$Clusters[Which,]
         
         plot(ZoomAreaSP, col="white", border="white", main=main)
         
      } else {
      
        Check <- sum(LocCatalog$SkiRuns$name %in% ZoomArea)
        
        if (Check>0) {
          
          ZoomLevel = "Run"
          
          Which <- which(LocCatalog$SkiRuns$name %in% ZoomArea)
          ZoomAreaSP <- LocCatalog$SkiRuns[Which,]
          
          plot(ZoomAreaSP, col="white", border="white", main=main)
          
        } else {
          
          Check <- sum(LocCatalog$SkiLines$name %in% ZoomArea)
          
          
          if (Check>0) {
            
            ZoomLevel = "Run"
            
            Which <- which(LocCatalog$SkiLines$name %in% ZoomArea)
            ZoomAreaSP <- LocCatalog$SkiLines[Which,]
            
            plot(ZoomAreaSP, col="white", border="white", main=main)
            
          } else {
            
            warning(paste0("No locations found with name '", paste(ZoomArea, collapse = "', '"), "'!"), immediate.=T)
            
          }
          
        }
        
      }
      
    }
    
  }
  
  ## Adding terrain
  if(!is.null(DEM)) {
    
    if (ZoomLevel=="Area") {
    
      ## Plotting
      warning("If Terrain is added, the LocCat needs to be in LatLon! Only default setting are supported! Use plotTerrain function directly to customize terrain representation.", immediate. = T)
      plotTerrain(DEM, add=T, ...)
      
    } else if(ZoomLevel=="Zone" | ZoomLevel=="Cluster") {
      
      ## Cropping
      ZoomExtent <- extent(ZoomAreaSP)
      CropExtent <- c(floor(ZoomExtent[1]), ceiling(ZoomExtent[2]), floor(ZoomExtent[3]), ceiling(ZoomExtent[4]))
      DEMCrop <- crop(DEM, CropExtent)
      
      Elev.Min <- floor(minValue(DEMCrop)/100)*100
      Elev.Max <- ceiling(maxValue(DEMCrop)/100)*100
      
      ## Plotting
      plotTerrain(DEMCrop, add = T, ...)
      
    } else if (ZoomLevel=="Run") {
      
      ## Cropping
      ZoomExtent <- extent(ZoomAreaSP)
      CropExtent <- c(floor(10*ZoomExtent[1])/10, ceiling(10*ZoomExtent[2])/10, floor(10*ZoomExtent[3])/10, ceiling(10*ZoomExtent[4])/10)
      DEMCrop <- crop(DEM, CropExtent)
      
      Elev.Min <- floor(minValue(DEMCrop)/100)*100
      Elev.Max <- ceiling(maxValue(DEMCrop)/100)*100
      
      ## Plotting
      plotTerrain(DEMCrop, add = T, ...)
      
    }
  }
  
  ## Operating zones
  if(ShowZones & !is.null(LocCatalog$Zones)) {
    plot(LocCatalog$Zones, lty=3, border=ColZones, add=T)
  }
  
  
  ## Run clusters
  if(ShowClusters & !is.null(LocCatalog$Clusters)) {
    plot(LocCatalog$Clusters, lty=3, border=ColClusters, add=T)
  }
  
  ## Cat roads
  if(ShowCatRoads & !is.null(LocCatalog$CatRoads)) {
    plot(LocCatalog$CatRoad, lty=3, col=ColCatRoads, add=T)
  }
  
  ## Operating ski runs and ski run segements - points & polygons
  if(ShowSkiRuns & !is.null(LocCatalog$SkiRuns)) {
    if(class(LocCatalog$SkiRuns)=="SpatialPointsDataFrame") {
      plot(LocCatalog$SkiRuns, pch=21, bg=ColSkiRuns, cex=0.5, add=T)
    } else {
      ## Ski run segements
      if(ShowSkiRunSegments & !is.null(LocCatalog$SkiRunSegments)) {
        plot(LocCatalog$SkiRunSegments[!is.na(LocCatalog$SkiRunSegments@data$terrain_rating) & LocCatalog$SkiRunSegments@data$terrain_rating=="green",], col=paste0(ColSkiRunSegments[1], TransSkiRunSegments), border=NA, add=T)
        plot(LocCatalog$SkiRunSegments[!is.na(LocCatalog$SkiRunSegments@data$terrain_rating) & LocCatalog$SkiRunSegments@data$terrain_rating=="blue",], col=paste0(ColSkiRunSegments[2], TransSkiRunSegments), border=NA, add=T)
        plot(LocCatalog$SkiRunSegments[!is.na(LocCatalog$SkiRunSegments@data$terrain_rating) & LocCatalog$SkiRunSegments@data$terrain_rating=="black",], col=paste0(ColSkiRunSegments[3], TransSkiRunSegments), border=NA, add=T)
      }
      ## Ski runs
      plot(LocCatalog$SkiRuns, border=ColSkiRuns, add=T)
    }
  }
  
  ## Operating ski lines - points & polygons
  if(ShowSkiLines & !is.null(LocCatalog$SkiLines)) {
    if(class(LocCatalog$SkiLines)=="SpatialPointsDataFrame") {
      plot(LocCatalog$SkiLines, pch=21, bg=ColSkiLines, cex=0.5, add=T)
    } else {
      plot(LocCatalog$SkiLines, col=ColSkiLines, add=T)
    }
  }
  
  ## Weather sites
  if(ShowWxSites & !is.null(LocCatalog$WxSites)) {
    plot(LocCatalog$WxSites, pch=21, bg=ColWxSites, cex=0.5, add=T)
  }
  
  ## Landings
  if(ShowLandings & !is.null(LocCatalog$Landings)) {
    plot(LocCatalog$Landings[LocCatalog$Landings@data$type=="Landing"|LocCatalog$Landings@data$type=="Both",], pch=24, bg=ColLandings[1], cex=0.3, add=T)
    plot(LocCatalog$Landings[LocCatalog$Landings@data$type=="Pickup"|LocCatalog$Landings@data$type=="Both",], pch=25, bg=ColLandings[2], cex=0.3, add=T)
  }

  ## Cruxes
  if(ShowCruxes & !is.null(LocCatalog$Cruxes)) {
    plot(LocCatalog$Cruxes, density=20, col=ColCruxes, add=T)
  }

  ## Overplot operating area
  plot(LocCatalog$OpArea, add=T)
  box()
  
}
  