#' Get run character assessments
#'
#' Get run character assessments
#' @param RunChar Run character assessment data frame
#' @param ShowCons Switch for showing assessment of most conervative line(s). Default is TRUE.
#' @param ShowTyp Switch for showing assessment of typical line(s). Default is TRUE.
#' @param ShowAggr Switch for showing assessment of more aggressive line(s). Default is TRUE.
#' @param ShowFam Switch for showing familiarity section. Default is TRUE.
#' @param ShowAccess Switch for showing access section. Default is TRUE.
#' @param ShowSkiExp Switch for showing skiing experience section. Default is TRUE.
#' @param ShowUsab Switch for showing usability section. Default is TRUE.
#' @param ShowHzd Switch for showing hazard potential section. Default is TRUE.
#' @param ShowMitig Switch for showing hazard mitigation section. Default is TRUE.
#' @export
#' 
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "CMHGL"
#' 
#' ## Single run
#' SkirunNames <- c("Alexis")
#' RunChar <- getRunCharAssessments(Operation, SkirunNames=SkirunNames)
#' printRunCharAssessments(RunChar)
#' 
#' ## Most conservative lines on runs of Wagner Blows cluster
#' ClusterNames <- "Wagner Bowls"
#' RunChar <- getRunCharAssessments(Operation, ClusterNames=ClusterNames)
#' printRunCharAssessments(RunChar, ShowTyp=F, ShowAggr=F, TitleAddition = "Cluster: Wanger Bowls")
#' 
#' ## All assessments
#' RunChar <- getRunCharAssessments(Operation)
#' printRunCharAssessments(RunChar, ShowTyp=F, ShowAggr=F, TitleAddition = "All runs on Nov. 26, 2017")

printRunCharAssessments <- function(RunChar, TitleAddition=NA,
                                    ShowCons=T, ShowTyp=T, ShowAggr=T,
                                    ShowFam=T, ShowConflict=T, ShowAccess=T, ShowSkiExp=T, ShowUsab=T, ShowHzd=T, ShowMitig=T){

  ## FUNCTIONS FOR SINGLE SUMMARIES
  ## ******************************
  
  ## Hazard potential (elevation specific)
  printHazardElevSingle <- function(SumStr, RunCar, Line, Elev) {
    
    if (Elev=="alp") {
      SumStr <- paste0(SumStr, "ALPINE:\n")
    } else if (Elev=="tl") {
      SumStr <- paste0(SumStr, "TREELINE:\n")
    } else if (Elev=="btl") {
      SumStr <- paste0(SumStr, "BELOW TREELINE:\n")
    }
    SumStr <- paste0(SumStr, "1) Type of terrain: ", RunChar[1, paste0("ski_terrain_", Line, "_", Elev)], "\n")
    SumStr <- paste0(SumStr, "2) Steepness: ", RunChar[1, paste0("hzd_steep_", Line, "_", Elev)])
    if (!is.na(RunChar[1, paste0("hzd_steep_", Line, "_", Elev)]) & RunChar[1, paste0("hzd_steep_", Line, "_", Elev)]!="gent") {SumStr <- paste0(SumStr, " with ", RunChar[1, paste0("hzd_expo_", Line, "_", Elev)])}
    SumStr <- paste0(SumStr, "\n")
    SumStr <- paste0(SumStr, "3) Av related terrain hazards: ", RunChar[1, paste0("hzd_terrainhzd_", Line, "_", Elev)], "\n")
    SumStr <- paste0(SumStr, "4) Av hzd promoting snowpack cond: ", RunChar[1, paste0("hzd_snowcond_", Line, "_", Elev)], "\n")
    SumStr <- paste0(SumStr, "5) Other hazards: ", RunChar[1, paste0("hzd_otherhzd_", Line, "_", Elev)], "\n")
    SumStr <- paste0(SumStr, "6) Overall friendliness: ", RunChar[1, paste0("hzd_friendly_", Line, "_", Elev)], "\n")
    SumStr <- paste0(SumStr, "\n")
    
    return(SumStr)
  }
  
  ## Skiing terrain (elevation specific)
  printSkiTerrainElevSingle <- function(SumStr, RunChar, Line, Elev) {
    
    if (Elev=="alp") {
      SumStr <- paste0(SumStr, "   - ALP: ")
    } else if (Elev=="tl") {
      SumStr <- paste0(SumStr, "   - TL:  ")
    } else if (Elev=="btl") {
      SumStr <- paste0(SumStr, "   - BTL: ")
    } else if (Elev=="all") {
      SumStr <- paste0(SumStr, "   - ALL: ")
    }
    
    SumStr <- paste0(SumStr, RunChar[1, paste0("ski_terrain_", Line, "_", Elev)])
    if (!is.na(RunChar[1, paste0("ski_spec_", Line, "_", Elev)])) {SumStr <- paste0(SumStr, " with ", RunChar[1, paste0("ski_spec_", Line, "_", Elev)])}
    SumStr <- paste0(SumStr, "\n")
    
    return(SumStr)
  }
  
  ## Printing of run line for single 
  printRunLineSingle <- function (SumStr, RunChar, Line, ShowAccess=T, ShowSkiExp=T, ShowUsab=T, ShowHzd=T, ShowMitig=T) {
    
    ## Header
    if(Line=="c") {
      SumStr <- paste0(SumStr, "MOST CONSERVATIVE LINE(S)\n")
      SumStr <- paste0(SumStr, "*************************\n")
    } else if (Line=="t") {
      SumStr <- paste0(SumStr, "TYPICAL LINE(S)\n")
      SumStr <- paste0(SumStr, "***************\n")
    } else if (Line=="a") {
      SumStr <- paste0(SumStr, "MORE AGGRESSIVE LINE(S)\n")
      SumStr <- paste0(SumStr, "***********************\n")
    }
    
    ## Access
    if(ShowAccess) {
      SumStr <- paste0(SumStr, "ACCESS\n")
      SumStr <- paste0(SumStr, "1) Accessibility of run: ", RunChar[1,paste0("acc_gen_feel_", Line)], "\n")
      SumStr <- paste0(SumStr, "2) Flight approach of best landing(s): ", RunChar[1,paste0("acc_best_land_", Line)], "\n")
      SumStr <- paste0(SumStr, "3) Standout features of landing(s): ", RunChar[1,paste0("acc_land_feat_", Line)], "\n")
      SumStr <- paste0(SumStr, "4) Standout features of pickup(s): ", RunChar[1,paste0("acc_pickup_feat_", Line)], "\n")
      SumStr <- paste0(SumStr, "5) Aware of any flying incidents/near misses: ", RunChar[1,paste0("acc_incident_", Line)], "\n")
      SumStr <- paste0(SumStr, "\n")
    }
    
    ## Skiing experience
    if(ShowSkiExp) {
      SumStr <- paste0(SumStr, "SKIING EXPERIENCE\n")
      SumStr <- paste0(SumStr, "1) Type of skiing terrain & special experiences:\n")
      if(!is.na(RunChar[1,paste0("ski_terrain_", Line, "_all")]) & RunChar[1,paste0("ski_terrain_", Line, "_all")]!="dna") {
        SumStr <- printSkiTerrainElevSingle(SumStr, RunChar, Line=Line, Elev="all")
      } else {
        if(!is.na(RunChar[1,paste0("ski_terrain_", Line, "_alp")]) & RunChar[1,paste0("ski_terrain_", Line, "_alp")]!="dna") {
          SumStr <- printSkiTerrainElevSingle(SumStr, RunChar, Line=Line, Elev="alp")
        }
        if(!is.na(RunChar[1,paste0("ski_terrain_", Line, "_tl")]) & RunChar[1,paste0("ski_terrain_", Line, "_tl")]!="dna") {
          SumStr <- printSkiTerrainElevSingle(SumStr, RunChar, Line=Line, Elev="tl")
        }
        if(!is.na(RunChar[1,paste0("ski_terrain_", Line, "_btl")]) & RunChar[1,paste0("ski_terrain_", Line, "_btl")]!="dna") {
          SumStr <- printSkiTerrainElevSingle(SumStr, RunChar, Line=Line, Elev="btl")
        }
      }
      SumStr <- paste0(SumStr, "2) Skiing difficulty: ", RunChar[1,paste0("ski_diff_", Line)], "\n")
      SumStr <- paste0(SumStr, "3) Skiing challenges: ", RunChar[1,paste0("ski_chall_", Line)], "\n")
      SumStr <- paste0(SumStr, "4) Overall experience: ", RunChar[1,paste0("ski_overall_", Line)], "\n")
      SumStr <- paste0(SumStr, "\n")
    }
    
    ## Usability
    if(ShowUsab) {
      SumStr <- paste0(SumStr, "USABILITY\n")
      SumStr <- paste0(SumStr, "1) Operational role: ", RunChar[1,paste0("usab_oprole_", Line)], "\n")
      SumStr <- paste0(SumStr, "2) Number of fresh tracks: ", RunChar[1,paste0("usab_num_tracks_", Line)], "\n")
      SumStr <- paste0(SumStr, "3) Reset factor: approx. ", RunChar[1,"usab_reset"], " cm HN24\n")
      SumStr <- paste0(SumStr, "4) Suncrust susceptibility: ", RunChar[1,"usab_suncrust"], " (on April 1)\n")
      SumStr <- paste0(SumStr, "5) High pickup options: ", RunChar[1,paste0("acc_pickup_opt_", Line)], "\n")
      SumStr <- paste0(SumStr, "\n")
    }
    
    ## Hazard potential
    if(ShowHzd) {
      SumStr <- paste0(SumStr, "HAZARD POTENTIAL\n")
      if(!is.na(RunChar[1,paste0("ski_terrain_", Line, "_all")]) & RunChar[1,paste0("ski_terrain_", Line, "_all")]!="dna") {
        SumStr <- printHazardElevSingle(SumStr, RunCar, Line=Line, Elev="all")
      } else if (!is.na(RunChar[1,paste0("ski_terrain_", Line, "_alp")]) |
                 !is.na(RunChar[1,paste0("ski_terrain_", Line, "_tl")]) |
                 !is.na(RunChar[1,paste0("ski_terrain_", Line, "_btl")])) {
        if(!is.na(RunChar[1,paste0("ski_terrain_", Line, "_alp")]) & RunChar[1,paste0("ski_terrain_", Line, "_alp")]!="dna") {
          SumStr <- printHazardElevSingle(SumStr, RunCar, Line=Line, Elev="alp")
        }
        if(!is.na(RunChar[1,paste0("ski_terrain_", Line, "_tl")]) & RunChar[1,paste0("ski_terrain_", Line, "_tl")]!="dna") {
          SumStr <- printHazardElevSingle(SumStr, RunCar, Line=Line, Elev="tl")
        }
        if(!is.na(RunChar[1,paste0("ski_terrain_", Line, "_btl")]) & RunChar[1,paste0("ski_terrain_", Line, "_btl")]!="dna") {
          SumStr <- printHazardElevSingle(SumStr, RunCar, Line=Line, Elev="btl")
        }
      } else {
        SumStr <- paste0(SumStr, "No data for any elevation bands.\n")
        SumStr <- paste0(SumStr, "\n")
      }
    }
    
    ## Mitigation options
    if(ShowMitig) {
      SumStr <- paste0(SumStr, "MITIGATION\n")
      SumStr <- paste0(SumStr, "1) Guidability: ", RunChar[1,paste0("mitig_guideabil_", Line)], "\n")
      SumStr <- paste0(SumStr, "2) Use of explosives:\n")
      SumStr <- paste0(SumStr, "   - On ski line: ", RunChar[1,paste0("mitig_explo_run_", Line)], "\n")
      SumStr <- paste0(SumStr, "   - For overhead hazard: ", RunChar[1,paste0("mitig_explo_oh_", Line)], "\n")
      SumStr <- paste0(SumStr, "   - For pickup(s): ", RunChar[1,paste0("mitig_explo_pickup_", Line)], "\n")
      SumStr <- paste0(SumStr, "3) Suitable for maintenance with skier traffic: ", RunChar[1,paste0("mitig_skiertraf_", Line)], "\n")
      SumStr <- paste0(SumStr, "4) Aware of any guiding incidents: ", RunChar[1,paste0("mitig_incident_", Line)], "\n")
      SumStr <- paste0(SumStr, "\n")
      
    }
    
  }
  
  ## FUNCTIONS FOR SUMMARIZING MULTIPLE ASSESSMENTS
  ## **********************************************
  
  ## Distribution for single choice question
  DistributionSC <- function(ValueArray) {
    
    ## Count
    Count <- sum(!is.na(ValueArray))
    Output <- paste0(Count, "/", length(ValueArray))
    
    ## Distribution
    if (Count>0) {
      Output <- paste0(Output, " [")
      
      Levels <- levels(ValueArray)
      Freq <- table(ValueArray)
      
      DistrArray <- character(0)
      for (i in 1:length(Levels)) {
        DistrArray <- c(DistrArray, paste0(Levels[i], ":"))
        if (i==length(Levels)) {
          DistrArray <- c(DistrArray, Freq[i], "")
        } else {
          DistrArray <- c(DistrArray, Freq[i], " ")
        }
      }
      Output <- paste0(Output, paste0(DistrArray, collapse = ""), "]")
    }
    
    ## Return
    return(Output)
    
  }
  
  ## Distribution for multiple choice question
  DistributionMC <- function(Operation, RunChar, VarName, Line=NA, Elev=NA) {
    
    ColName <- VarName
    if(!is.na(Line)){ColName <- paste0(ColName, "_", Line)}
    if(!is.na(Elev)){ColName <- paste0(ColName, "_", Elev)}
    
    ## Count
    Count <- sum(!is.na(RunChar[,ColName]))
    Output <- paste0(Count, "/", length(RunChar[,ColName]))
    
    ## Distribution
    if (Count>0) {
      Output <- paste0(Output, " [")
      
      ## Get enumerations
      Query <- paste0("Select code from runchar.enum_", VarName, " order by \"order\"")
      Levels <- getRecordsFromQuery(Operation, Query)$code
      
      Freq <- numeric(0)
      for (i in 1:length(Levels)) {
        Freq <- c(Freq, sum(RunChar[,paste0(ColName, "_", Levels[i])]))
      }
      
      DistrArray <- character(0)
      for (i in 1:length(Levels)) {
        DistrArray <- c(DistrArray, paste0(Levels[i], ":"))
        if (i==length(Levels)) {
          DistrArray <- c(DistrArray, Freq[i], "")
        } else {
          DistrArray <- c(DistrArray, Freq[i], " ")
        }
      }
      Output <- paste0(Output, paste0(DistrArray, collapse = ""), "]")
    }
    
    ## Return
    return(Output)
    
  }
  
  ## Skiing terrain (elevation specific)
  printSkiTerrainElevMultiple <- function(Operation, SumStr, RunChar, Line, Elev) {
    
    if (Elev=="alp") {
      SumStr <- paste0(SumStr, "   - ALP: ")
      Indent <- "          "
    } else if (Elev=="tl") {
      SumStr <- paste0(SumStr, "   - TL:  ")
      Indent <- "          "
    } else if (Elev=="btl") {
      SumStr <- paste0(SumStr, "   - BTL: ")
      Indent <- "          "
    } else if (Elev=="all") {
      SumStr <- paste0(SumStr, "   - ALL: ")
      Indent <- "          "
    }
    
    
    SumStr <- paste0(SumStr, "Skiing terrain: ", DistributionMC(Operation, RunChar, VarName="ski_terrain", Line=Line, Elev=Elev), "\n")
    SumStr <- paste0(SumStr, Indent, "Special exp.:   ", DistributionMC(Operation, RunChar, VarName="ski_spec", Line=Line, Elev=Elev), "\n")

    return(SumStr)
  }
  
  ## Hazard potential (elevation specific)
  printHazardElevMultiple <- function(Operation, SumStr, RunCar, Line, Elev) {
    
    if (Elev=="alp") {
      SumStr <- paste0(SumStr, "ALPINE:\n")
    } else if (Elev=="tl") {
      SumStr <- paste0(SumStr, "TREELINE:\n")
    } else if (Elev=="btl") {
      SumStr <- paste0(SumStr, "BELOW TREELINE:\n")
    } else if (Elev=="all") {
      SumStr <- paste0(SumStr, "GENERAL:\n")
    }
    SumStr <- paste0(SumStr, "1) Type of terrain: ", DistributionMC(Operation, RunChar, VarName="ski_terrain", Line=Line, Elev=Elev), "\n")
    SumStr <- paste0(SumStr, "2a) Steepness: ", DistributionSC(RunChar[, paste0("hzd_steep_", Line, "_", Elev)]), "\n")
    SumStr <- paste0(SumStr, "2b) Slope size: ", DistributionSC(RunChar[, paste0("hzd_expo_", Line, "_", Elev)]), "\n")
    SumStr <- paste0(SumStr, "3) Av related terrain hazards: ", DistributionMC(Operation, RunChar, VarName="hzd_terrainhzd", Line=Line, Elev=Elev), "\n")
    SumStr <- paste0(SumStr, "4) Av hzd promoting snowpack cond: ", DistributionMC(Operation, RunChar, VarName="hzd_snowcond", Line=Line, Elev=Elev), "\n")
    SumStr <- paste0(SumStr, "5) Other hazards: ", DistributionMC(Operation, RunChar, VarName="hzd_otherhzd", Line=Line, Elev=Elev), "\n")
    SumStr <- paste0(SumStr, "6) Overall friendliness: ", DistributionSC(RunChar[, paste0("hzd_friendly_", Line, "_", Elev)]), "\n")
    SumStr <- paste0(SumStr, "\n")
    
    return(SumStr)
  }
  
  ## Printing of run line for multiple 
  printRunLineMultiple <- function (Operation, SumStr, RunChar, Line, ShowAccess=T, ShowSkiExp=T, ShowUsab=T, ShowHzd=T, ShowMitig=T) {
    
    ## Header
    if(Line=="c") {
      SumStr <- paste0(SumStr, "MOST CONSERVATIVE LINE(S)\n")
      SumStr <- paste0(SumStr, "*************************\n")
    } else if (Line=="t") {
      SumStr <- paste0(SumStr, "TYPICAL LINE(S)\n")
      SumStr <- paste0(SumStr, "***************\n")
    } else if (Line=="a") {
      SumStr <- paste0(SumStr, "MORE AGGRESSIVE LINE(S)\n")
      SumStr <- paste0(SumStr, "***********************\n")
    }
    
    ## Access
    if(ShowAccess) {
      SumStr <- paste0(SumStr, "ACCESS\n")
      SumStr <- paste0(SumStr, "1) Accessibility of run: ", DistributionSC(RunChar[,paste0("acc_gen_feel_", Line)]), "\n")
      SumStr <- paste0(SumStr, "2) Flight approach of best landing(s): ", DistributionSC(RunChar[,paste0("acc_best_land_", Line)]), "\n")
      SumStr <- paste0(SumStr, "3) Standout features of landing(s): ", DistributionMC(Operation, RunChar, "acc_land_feat", Line), "\n")
      SumStr <- paste0(SumStr, "4) Standout features of pickup(s): ", DistributionMC(Operation, RunChar, "acc_pickup_feat", Line), "\n")
      SumStr <- paste0(SumStr, "5) Aware of any flying incidents/mear misses: ", DistributionSC(RunChar[,paste0("acc_incident_", Line)]), "\n")
      SumStr <- paste0(SumStr, "\n")
    }
    
    ## Skiing experience
    if(ShowSkiExp) {
      SumStr <- paste0(SumStr, "SKIING EXPERIENCE\n")
      SumStr <- paste0(SumStr, "1) Type of skiing terrain & special experiences:\n")
      SumStr <- printSkiTerrainElevMultiple(Operation, SumStr, RunChar, Line=Line, Elev="all")
      SumStr <- printSkiTerrainElevMultiple(Operation, SumStr, RunChar, Line=Line, Elev="alp")
      SumStr <- printSkiTerrainElevMultiple(Operation, SumStr, RunChar, Line=Line, Elev="tl")
      SumStr <- printSkiTerrainElevMultiple(Operation, SumStr, RunChar, Line=Line, Elev="btl")
      
      SumStr <- paste0(SumStr, "2) Skiing difficulty: ", DistributionSC(RunChar[,paste0("ski_diff_", Line)]), "\n")
      SumStr <- paste0(SumStr, "3) Skiing challenges: ", DistributionSC(RunChar[,paste0("ski_chall_", Line)]), "\n")
      SumStr <- paste0(SumStr, "4) Overall experience: ", DistributionSC(RunChar[,paste0("ski_overall_", Line)]), "\n")
      SumStr <- paste0(SumStr, "\n")
    }
    
    ## Usability
    if(ShowUsab) {
      SumStr <- paste0(SumStr, "USABILITY\n")
      SumStr <- paste0(SumStr, "1) Operational role: ", DistributionMC(Operation, RunChar, "usab_oprole", Line), "\n")
      SumStr <- paste0(SumStr, "2) Number of fresh tracks (sum): ", sum(RunChar[,paste0("usab_num_tracks_", Line)]), "\n")
      SumStr <- paste0(SumStr, "3) Reset factor (approx. cm HN24): ", DistributionSC(RunChar[,"usab_reset"]), "\n")
      SumStr <- paste0(SumStr, "4) Suncrust susceptibility (on April 1): ", DistributionSC(RunChar[,"usab_suncrust"]), "\n")
      SumStr <- paste0(SumStr, "5) High pickup options: ", DistributionSC(RunChar[,paste0("acc_pickup_opt_", Line)]), "\n")
      SumStr <- paste0(SumStr, "\n")
    }
    
    ## Hazard potential
    if(ShowHzd) {
      SumStr <- paste0(SumStr, "HAZARD POTENTIAL\n")
      SumStr <- printHazardElevMultiple(Operation, SumStr, RunCar, Line=Line, Elev="all")
      SumStr <- printHazardElevMultiple(Operation, SumStr, RunCar, Line=Line, Elev="alp")
      SumStr <- printHazardElevMultiple(Operation, SumStr, RunCar, Line=Line, Elev="tl")
      SumStr <- printHazardElevMultiple(Operation, SumStr, RunCar, Line=Line, Elev="btl")
    }
    
    
    ## Mitigation options
    if(ShowMitig) {
      SumStr <- paste0(SumStr, "MITIGATION\n")
      SumStr <- paste0(SumStr, "1) Guidability: ", DistributionSC(RunChar[,paste0("mitig_guideabil_", Line)]), "\n")
      SumStr <- paste0(SumStr, "2) Use of explosives:\n")
      SumStr <- paste0(SumStr, "   - On ski line: ", DistributionSC(RunChar[,paste0("mitig_explo_run_", Line)]), "\n")
      SumStr <- paste0(SumStr, "   - For overhead hazard: ", DistributionSC(RunChar[,paste0("mitig_explo_oh_", Line)]), "\n")
      SumStr <- paste0(SumStr, "   - For pickup(s): ", DistributionSC(RunChar[,paste0("mitig_explo_pickup_", Line)]), "\n")
      SumStr <- paste0(SumStr, "3) Suitable for maintenance with skier traffic: ", DistributionSC(RunChar[,paste0("mitig_skiertraf_", Line)]), "\n")
      SumStr <- paste0(SumStr, "4) Aware of any guiding incidents: ", DistributionSC(RunChar[,paste0("mitig_incident_", Line)]), "\n")
      SumStr <- paste0(SumStr, "\n")
    }
      
  }
    
  
  ## SINGLE OVERVIEW
  ## ***************
  
  if (nrow(RunChar)==1) {
  
    ## Header 
    SumStr <- paste0("\n",
                     "Run Character Assessment for ", RunChar$loccatskirun_name[1], "\n", 
                     "*****************************", paste(rep("*", nchar(as.character(RunChar$loccatskirun_name[1]))), collapse = ""), "\n",
                     "\n")
  
    ## Location
    SumStr <- paste0(SumStr, "Zone: ", RunChar$loccatzone_name, "\n")
    SumStr <- paste0(SumStr, "Cluster: ", RunChar$loccatclusters_name, "\n")
    SumStr <- paste0(SumStr, "\n")
    
    ## Familiarity
    if(ShowFam) {
      SumStr <- paste0(SumStr, "Guide familiarity: ", RunChar$familiarity, "\n")
      SumStr <- paste0(SumStr, "\n")
    }
  
    ## Conflict
    if(ShowConflict) {
      SumStr <- paste0(SumStr, "Conflicts\n")
      SumStr <- paste0(SumStr, "1) Other users: ", RunChar[1,"usab_conflict"])
      if(!is.na(RunChar[1,"usab_conflict"]) & RunChar[1,"usab_conflict"]!="none") {SumStr <- paste0(SumStr, " (", RunChar[1,"usab_conflict_type"], ")")}
      SumStr <- paste0(SumStr, "\n")
      SumStr <- paste0(SumStr, "2) Wildlife: ", RunChar[1,"usab_wildlife"])
      if(!is.na(RunChar[1,"usab_wildlife"]) & RunChar[1,"usab_wildlife"]=="yes") {SumStr <- paste0(SumStr, " (", RunChar[1,"usab_wildlife_type"], ")")}
      SumStr <- paste0(SumStr, "\n")
      SumStr <- paste0(SumStr, "\n")
    }
    
    ## Conservative lines
    if(ShowCons) {
      SumStr <- printRunLineSingle(SumStr, RunChar, Line = "c", ShowAccess=ShowAccess, ShowSkiExp=ShowSkiExp, ShowUsab=ShowUsab, ShowHzd=ShowHzd, ShowMitig=ShowMitig)
    }

    ## Typical lines
    if(ShowTyp) {
      SumStr <- printRunLineSingle(SumStr, RunChar, Line = "t", ShowAccess=ShowAccess, ShowSkiExp=ShowSkiExp, ShowUsab=ShowUsab, ShowHzd=ShowHzd, ShowMitig=ShowMitig)
    }
    
    ## More aggressive lines
    if(ShowAggr) {
      SumStr <- printRunLineSingle(SumStr, RunChar, Line = "a", ShowAccess=ShowAccess, ShowSkiExp=ShowSkiExp, ShowUsab=ShowUsab, ShowHzd=ShowHzd, ShowMitig=ShowMitig)
    }

  ## MULTIPE ASSESSMENTS  
  } else {
    
    ## Header
    Title <- ifelse(is.na(TitleAddition),
                    paste0("Summary of ", nrow(RunChar) , " Run Character Assessments"),
                    paste0("Summary of ", nrow(RunChar) , " Run Character Assessments (", TitleAddition, ")"))
    
    SumStr <- paste0("\n", Title, "\n")
    SumStr <- paste0(SumStr, paste(rep("*", nchar(Title)), collapse = ""))
    SumStr <- paste0(SumStr, "\n")                 

    ## Familiarity
    if(ShowFam) {
      SumStr <- paste0(SumStr, "Guide familiarity: ", DistributionSC(RunChar$familiarity), "\n")
      SumStr <- paste0(SumStr, "\n")
    }
    
    ## Conflict
    if(ShowConflict) {
      SumStr <- paste0(SumStr, "Conflicts\n")
      SumStr <- paste0(SumStr, "1) Other users: ", DistributionSC(RunChar[,"usab_conflict"]))
      SumStr <- paste0(SumStr, " - Type: ", DistributionMC(Operation, RunChar, "usab_conflict_type"), "\n")
      SumStr <- paste0(SumStr, "2) Wildlife: ", DistributionSC(RunChar[,"usab_wildlife"]))
      SumStr <- paste0(SumStr, " - Type: ", DistributionMC(Operation, RunChar, "usab_wildlife_type"), "\n")
      SumStr <- paste0(SumStr, "\n")
    }
    
    ## Conservative lines
    if(ShowCons) {
      SumStr <- printRunLineMultiple(Operation, SumStr, RunChar, Line = "c", ShowAccess=ShowAccess, ShowSkiExp=ShowSkiExp, ShowUsab=ShowUsab, ShowHzd=ShowHzd, ShowMitig=ShowMitig)
    }
    
    ## Typical lines
    if(ShowTyp) {
      SumStr <- printRunLineMultiple(Operation, SumStr, RunChar, Line = "t", ShowAccess=ShowAccess, ShowSkiExp=ShowSkiExp, ShowUsab=ShowUsab, ShowHzd=ShowHzd, ShowMitig=ShowMitig)
    }
    
    ## More aggressive lines
    if(ShowAggr) {
      SumStr <- printRunLineMultiple(Operation, SumStr, RunChar, Line = "a", ShowAccess=ShowAccess, ShowSkiExp=ShowSkiExp, ShowUsab=ShowUsab, ShowHzd=ShowHzd, ShowMitig=ShowMitig)
    }
    
    
    
  }
  
  ## Printing summary
  cat(SumStr)
  
}