#' Processes GPS tracks
#'
#' Processes GPS tracks and imports results into PostgreSQL/PostGIS GPS database or writes them out as CSV files. Produces various additional output files (cvs, txt and png).
#' @param Operation Name of Operation. Used for accessing PostgreSQL/PostGIS GPS database.
#' @param Folder_GPSTracks Complete path of folder with GPS to be processed. You can use function paste0(getLocalRootFolderPath(Operation, User), "WhateverSubFolderPathYouWant") to make script transferable between users.
#' @param Folder_OutputRoot Complete path of local root folder for processing output.
#' @param DBType Type of database the records should be written to. Can be 'Main' (default) or 'NodeJS' for database that feeds online viewer.
#' @param UseElevRaster Flag whether elevation values from raster are used to determine uphill and downhill sections of GPS tracks. Typically produces better results than the altitude values from the GPS units.
#' @param ElevRaster Elevation raster of class RasterLayer. Required if UseElevRaster==TRUE. You can use getRaster(Operation, User, "Elevation") function for retrieving elevation raster.
#' @param FileNameThirdPart Specifies what the thirs part of the file name of the GPS tracks means. Possible values include 'guide' (default), 'program' or NA.
#' @param DefaultGuideType Specifies default entry for guide type field in tracks_day table (e.g., 'Lead guide' (default), Snowsafety, ...).
#' @param WriteToDB Flag for specifying whether the results of the processing should be written to the PostgreSQL/PostGIS GPS database (TRUE, default) or to CSV file (FALSE). To avoid the processing of the same GPS data multiple times, only trackdays are processed that do not yet have any records in the database. Partially processed trackdays need to be deleted before they can be reprocessed again. Use deleteTrackdayRecordsPostgreSQL() to delete trackday from database.
#' @param WriteChartsToFile Flag for specifying whether charts should be written to png file (default) or presented in a window in R.
#' @param OnlyProcessTrackdaysNotInDB Flag for specifying whether only trackdays that have not been written to database previously should be processed. Only applies when output is written to CSV file (i.e., WriteToDB==F), since it is impossible to write a trackday multiple time to the database. Default value is FALSE.
#' @param Process_TrackFileNames Single file name or list of file names of GPS tracks in Folder_GPSTracks to be processed. By default, all CSV files in Folder_GPSTracks are processed.
#' @param Process_FirstTrackIndex Index of first GPS track file in Folder_GPSTracks to be processed. This parameter allows the efficient skipping of files that might have been processed previously. Default values is 1, which means that all files are processed. If parameter Process_TrackFileNames is used, Process_FirstTrackIndex must be smaller or equal to the number of file names provided in Process_TrackFileNames.
#' @param Process_TrackdayDates Single date or list of dates for which GPS tracks should be processed. By default all of the dates included in the GPS track files are processed.
#' @param NeedToDownSampleGPSTrack Flag specifying whether the recording frequency of the GPS track needs to be checked and potentially downsampled to the trget interval specified with the parameter TargetRecordingInterval. Only works if the TargetRecordingInterval is a multiple of the existing recording interval (e.g., GPS track has records very 2 sec, but the target interval is 4 sec).
#' @param TargetRecordingInterval Target recording interval for GPS tracks. Default value is 4 sec, which has been used in most GPS tracks.
#' @param InclRunUseList Flag for specifying whether run use lists of processed trackdays should be included. Default is TRUE.
#' @param InclRunUseListFromPrevProcTrackdays Flag for specifying whether run use lists of previously processed trackdays should be included in the output file. Default value is FALSE to reduce processing time and the size of the output file.
#' @param TimeDiffGMT Constant for time difference from GMT (time recorded with GPS observations) to local time. Default is 8 hrs (GMT to PST)
#' @param MinNumObsInsideOpAreaForProcessing Minimum number of observations inside of operating area required for a day to be processed. Default value is 10.
#' @param DilutionThreshold Constant for minimun dilution value for GPS observations to be excluded from analysis. Default value is 6 (see Sigrist et al. (1999))
#' @param GPSSpatialAccuracy Constant for known spatial accuracy of GPS observations in meters. Default value is 4 m (Rupf et al., 2010). This parameter is currently not used for any computations.
#' @param OpArea_Buffer Constant for buffer distance around operating area in meters. GPS observations are only processed if they are within the buffer distance of the operating area. The buffer ensures that runs that are at the boundary of the operating area are processed even if the accuracy of the operation area boundary polygon is limited. Default value is 500 m.
#' @param Flight_MinSpeed Constant for minimum speed that is considered to be associated with flying in a helicopter in km/h.  Default value is 60 km/h. This value is used to identify helicopter flight in GPS tracks from heliskiing.
#' @param Flight_MinDuration Constant for minimum duration of a helicopter flight in seconds. Default value is 20 sec. Shorter periods with speeds above Flight_MinSpeed are not marked as flights.
#' @param Flight_MinAltGain Constant for minimum elevation gain for a potential flight segment to be considered a flight in m. Default value is 100 m. The required elevation gain is between the beginning and end of the segment (start to end location). This requirements is not applied to the last flight of the day, which might be going straight into the valley.
#' @param CatRoad_MaxDist Constant for maximum distance of GPS observation to cat road stored in PostgreSQL/PostGIS GPS database to be considered on road in m. Defaults value is 20 m.
#' @param CatRoad_MinDist Constant for minimum distance of continuous car road ride. Default value is 200m.
#' @param CatRoad_MinDuration Constant for minimum duration on cat road for potentially being considered riding a snowcat in sec. Default value is 60 sec.
#' @param CatRoad_MaxPercentageOfRun Maximum percentage of observations that can be on a cat road for ski run not considered a cat ride. Default of 0.8.
#' @param Pause_MinTime Constant for minimum duration of shorter pause periods in sec. Value needs to be smaller than Break_MinTime. Default value is 10 sec.
#' @param Pause_MoveMinTime Constant for minimum duration of moving period between shorter pause periods. Default value is 10 sec. This value needs to be equal or smaller than the value of Pause_MinTime.
#' @param Pause_DistRadius Constant for maximum travel distance during shorter pause periods in m. Value needs to be smaller than Break_DistRadius. Default value is 5 m.
#' @param Break_MinTime Constant for minimum duration of a longer break period (e.g., getting ready at beginning of run, packing up at end of run, longer breaks duriong run...) in sec. Value needs to be larger than Pause_MinTime. Default value is 90 sec.
#' @param Break_MoveMinTime Constant for minimum duration of moving period between break periods. Default value is 10 sec. This value needs to be equal or smaller than the value of Break_MinTime.
#' @param Break_DistRadius Constant for maximum travel distance during a break in m. Value needs to be larger than Pause_DistRadius. Default value is 15 m.
#' @param UpDown_AltDiffThreshold Constant for altitude difference threshold in m that will trigger the switch from an uphill to a downhill section or vice versa. Default value is 50 m.
#' @param UpDown_MinDuration Constant for minimum duration of up-or downhill section in sec. Default value is 20 sec.
#' @param UpDown_StartEndAltBuffer Constant for elevation metre buffer for GPS observation still to be considered part of a downhill section in m. Default value is 25 m. This means that GPS observations prior to a true downhill section that are less than 25 m lower than the absolute high point of the downhill section are still included in the extraction of a potential ski run.
#' @param NewRun_DistToPrevObsWithInclReq_Dist Constant for minimum distance between GPS observations that would trigger the marking of the start of a potential new ski run with the additional altitude change requirement in m. Default value is 500 m.
#' @param NewRun_DistToPrevObsWithInclReq_Incl Maximum average incline requirement for distances between NewRun_DistToPrevObsWithInclReq_Dist and NewRun_DistToPrevObsWithoutInclReq to trigger a new start/end of a run in degrees.Default is 15 deg.
#' @param NewRun_AltDiffToPrevObs Constant for minimum altitude difference between GPS observations that would trigger the marking of the start of a potential new ski run in m. Default value is 100 m.
#' @param NewRun_DistToPrevObsWithoutInclReq Constant for minimum distance between GPS observations that would trigger the marking of the start of a potential new ski run without the additional altitude change requirement in m. Default value is 1500 m.
#' @param PotRun_MinDuration Constant for minimum duration of potential ski run in sec. Default value is 120 sec.
#' @param PotRun_MinDurationSkiing Constant for minimum duration of skiing during potential ski run in sec. Default value is 40 sec.
#' @param PotRun_ShowProcessingDetail Switch for showing or hiding detailed output during processing of potential runs. Default is F.
#' @param Run_MinLength Constant for minimum length of ski run in m. Default value is 200 m.
#' @param Run_MinMaxSkiingSpeed Constant for minimum value of maximum skiing speed during ski run in km/h. Default value is 10 km/h.
#' @param Simplify_RemoveBightMaxDist Constant for maximum distance in m within a bight in a ski run line for the bight to be removed. Default value is 10 m. Set parameter to NA for bights not to be removed.
#' @param Simplify_ForcePausePointsReplaceDist Constant for minimum distance for forcing the ski line through a pause or break point. Default value is 10 m.  Set parameter to NA for not forcing line through pause points.
#' @param Simplify_SimplifyThreshold Constant for distance value in m used to simplify run line geometry using rgeos:gsimplfy. The value is the numerical tolerance to be used by the Douglas-Peuker algorithm. See description of gsimplify for more detail. Default is NA, which means that the ski run lines are not being simplfied.
#' @param Col_GettingReady Standard color for 'getting ready' break points. Default value is brown.
#' @param Col_PackingUp Standard color for 'packing up' break points. Default value is dark green.
#' @param Col_Pause Standard color for pause points. Default value is steelblue1.
#' @param Col_Break Standard color for break points. Default value is royalblue3.
#' @param Col_Skiing Standard color for skiing points. Default value is sienna1.
#' @param Col_CatRiding Standard color for snowcat riding points. Default value is beige.
#' @param Col_Flying Standard color for flying points. Default value is palegreen2.
#' @param Col_Outside Standard color got points outside of the operating area or inside an exclusion polygon. Default value is grey.
#' @param Chart_Width Costant for width of png output charts.
#' @param Chart_Height Constant for height of png output charts.
#' @return Dataframe with basic processing summary
#'
#' @export

processGPSTracks <- function(Operation,
                             Folder_GPSTracks,
                             Folder_OutputRoot,
                             DBType = "Main",
                             UseElevRaster=T,
                             ElevRaster=NA,
                             FileNameThirdPart='guide',
                             DefaultGuideType='Lead guide',
                             WriteToDB=T,
                             WriteChartsToFile=T,
                             OnlyProcessTrackdaysNotInDB=F,
                             Process_TrackFileNames=NA,
                             Process_FirstTrackIndex=1,
                             Process_TrackdayDates=NA,
                             NeedToDownSampleGPSTrack=T,
                             TargetRecordingInterval=4,
                             InclRunUseList=T,
                             InclRunUseListFromPrevProcTrackdays=F,
                             TimeDiffGMT=8,
                             MinNumObsInsideOpAreaForProcessing=10,
                             DilutionThreshold=6,
                             GPSSpatialAccuracy=4,
                             OpArea_Buffer=500,
                             Flight_MinSpeed=60,
                             Flight_MinDuration=20,
                             Flight_MinAltGain=100,
                             CatRoad_MaxDist=20,
                             CatRoad_MinDist=200,
                             CatRoad_MinDuration=60,
                             CatRoad_MaxPercentageOfRun=0.85,
                             Pause_MinTime=10,
                             Pause_MoveMinTime=10,
                             Pause_DistRadius=5,
                             Break_MinTime=90,
                             Break_MoveMinTime=10,
                             Break_DistRadius=15,
                             UpDown_AltDiffThreshold=50,
                             UpDown_MinDuration=20,
                             UpDown_StartEndAltBuffer=25,
                             # NewRun_DistToPrevObsWithAltDiffReq=500,
                             NewRun_DistToPrevObsWithInclReq_Dist=500,
                             NewRun_DistToPrevObsWithInclReq_Incl=15,
                             NewRun_AltDiffToPrevObs=100,
                             NewRun_DistToPrevObsWithoutInclReq=1500,
                             PotRun_MinDuration=120,
                             PotRun_MinDurationSkiing=40,
                             PotRun_ShowProcessingDetail=F,
                             Run_MinLength=200,
                             Run_MinMaxSkiingSpeed=10,
                             Simplify_RemoveBightMaxDist=10,
                             Simplify_ForcePausePointsReplaceDist=10,
                             Simplify_SimplifyThreshold=NA,
                             Col_GettingReady="brown",
                             Col_PackingUp="dark green",
                             Col_Pause="steelblue1",
                             Col_Break="royalblue3",
                             Col_Skiing="sienna1",
                             Col_CatRiding="beige",
                             Col_Flying="palegreen2",
                             Col_Outside="grey",
                             Chart_Width=900,
                             Chart_Height=600) {

## vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
## 1. VALIDATION CHECK OF INPUT PARAMETERS ====
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

  if (Pause_MinTime < Pause_MoveMinTime) {
    stop ("Value of Pause_MoveMinTime needs to be smaller or equal to value of Pause_MinTime!")
  }

  if (Break_MinTime < Break_MoveMinTime) {
    stop ("Value of Break_MoveMinTime needs to be smaller or equal to value of Break_MinTime!")
  }

  if (Break_MinTime <= Pause_MinTime) {
    stop ("Value of Break_MinTime needs to be larger than value of Pause_MinTime!")
  }

  if (Break_DistRadius <= Pause_DistRadius) {
    stop ("Value of Break_DistRadius needs to be larger than value of Pause_DistRadius!")
  }

## vvvvvvvvvvvvvvvvvvvvvvvvv
## 2. REQUIRED PACKAGES ====
## ^^^^^^^^^^^^^^^^^^^^^^^^^

#   require(SarpGPSTools)
#   require(SarpGPSToolsPrivate)
#
#   require(RPostgreSQL) # for database interaction
#   require(chron)       # for dates, times, chron
#   require(sp)          # for geospatial objects
#   require(rgdal)       # for readORG
#   require(geosphere)   # for distHaversine, bearing
#   require(maptools)    # for spRbind
#   require(rgeos)       # for simplification of geometries
#   require(raster)      # for ElevRaster


## vvvvvvvvvvvvvvvvvvvvvvvv
## 3. CUSTOM FUNCTIONS ====
## ^^^^^^^^^^^^^^^^^^^^^^^^

  ## Adds entry to progress report dataframe
  ## ***************************************
  addToProgressDF <- function(ProgressDF=Output_Progress, Comment=NA, Num=NA, DTCre=Sys.time()) {

    ProgressDF <- rbind(ProgressDF, data.frame(Comment=Comment,
                                               Num=Num,
                                               DTCre=DTCre))

    return(ProgressDF)

  }

  ## Create parameter string for progress report dataframe entry
  ## ***********************************************************
  createParameterString <- function(Parameter, Unit=NA) {
    if (is.na(Unit)) {
      String <- paste0(deparse(substitute(Parameter)), ": ", Parameter)
    } else {
      String <- paste0(deparse(substitute(Parameter)), ": ", Parameter, " ", Unit)
    }
    return(String)
  }

  ## Deleting every nth record in a dataframe
  ## ****************************************
  deleteNth <- function(dataframe, n) {
    dataframe[-(seq(n,to=nrow(dataframe), by=n)),]
  }



  ## Creates data frame that summarizes periods in time series
  ## *********************************************************
  summarizePeriodsInTimeSeries <- function(DF, ColName, ColNameDist="dist", ColNameDuration="timediff_sec") {

    #   DF <- DF_Day
    #   ColName <- "marker_downhill"
    #   ColNameDist="dist"
    #   ColNameDuration="timediff_sec"

    ## Create output df
    DF_Output <- data.frame(StartDateTimeLocal=numeric(0), EndDateTimeLocal=numeric(0), Value=numeric(0), Duration=numeric(0), NumObs=numeric(0), CumDist=numeric(0))

    ## Eliminate any NA in column to be summarized
    DF <- DF[!is.na(DF[,ColName]),]

    ## Summarize DF
    for (Index in 1:nrow(DF)) {

      # print(paste("Index", Index, "-", DF$datetime_local[Index], "-", DF[Index, ColName]))

      ## Very first observation -> Initialisation
      if (Index==1) {
        StartDateTimeLocal <- DF$datetime_local[Index]
        Value <- DF[Index, ColName]
        NumObs <- 1
        CumDist <- DF[Index, ColNameDist]
        Duration <- DF[Index, ColNameDuration]

        # print(paste("1 -", NumObs, CumDist, Duration))

        ## Any subsequent observations
      } else {

        ## If current value is the same as the previous value
        if (DF[Index, ColName]==Value) {
          NumObs <- NumObs+1
          CumDist <- CumDist+DF[Index, ColNameDist]
          Duration <- Duration+DF[Index, ColNameDuration]

          # print(paste("2 -", NumObs, CumDist, Duration))

          ## In case last observation has been reached
          if (Index==nrow(DF)) {
            DF_Output <- rbind(DF_Output, data.frame(StartDateTimeLocal=StartDateTimeLocal,
                                                     EndDateTimeLocal=DF$datetime_local[Index],
                                                     Value=Value,
                                                     Duration=Duration,
                                                     NumObs=NumObs,
                                                     CumDist=CumDist))

          }

          ## If current value is different from previous value
        } else {

          DF_Output <- rbind(DF_Output, data.frame(StartDateTimeLocal=StartDateTimeLocal,
                                                   EndDateTimeLocal=DF$datetime_local[Index-1],
                                                   Value=Value,
                                                   Duration=Duration,
                                                   NumObs=NumObs,
                                                   CumDist=CumDist))

          StartDateTimeLocal <- DF$datetime_local[Index]
          Value <- DF[Index, ColName]
          NumObs <- 1
          CumDist <- DF[Index, ColNameDist]
          Duration <- DF[Index, ColNameDuration]

          # print(paste("3 -", NumObs, CumDist, Duration))

          ## In case last observation has been reached
          if (Index==nrow(DF)) {
            DF_Output <- rbind(DF_Output, data.frame(StartDateTimeLocal=StartDateTimeLocal,
                                                     EndDateTimeLocal=DF$datetime_local[Index],
                                                     Value=Value,
                                                     Duration=Duration,
                                                     NumObs=NumObs,
                                                     CumDist=CumDist))

          }

        }

      }

    }

    ## Calculate duration of periods
    # DF_Output$Duration <- (DF_Output$EndDateTimeLocal-DF_Output$StartDateTimeLocal)*24*60*60

    ## Return output df
    return(DF_Output)

  }


  ## Identify stationary period longer than a time threshold within a distance
  ## *************************************************************************

  identifyStationaryPeriod <- function(DF, DistanceThreshold, TimeThreshold) {

    ## Prepare output
    DF_Output <- data.frame(DateTimeLocalStart=numeric(0), DateTimeLocalEnd=numeric(0), Duration=numeric(0), MaxDist=numeric(0), NumObs=numeric(0))

    ## Initialize parameter for outer loop
    Index_Obs <- 1

    ## Outer loop: Looping through all observations
    while (Index_Obs < nrow(DF)) {

      ## Extract information for first point
      Coord_P1 <- c(DF$gps_longitude[Index_Obs], DF$gps_latitude[Index_Obs])
      Time_P1  <- DF$datetime_local[Index_Obs]

      ## Initialize parameters for nested loop
      PointDist <- 0
      SubIndex_Obs <- 1
      DistArray <- numeric(0)

      ## Nested loop: Looping through points starting from first point
      while ((PointDist<=DistanceThreshold) & (Index_Obs+SubIndex_Obs<=nrow(DF))) {

        ## Extract information for second point
        Coord_P2 <- c(DF_Day$gps_longitude[Index_Obs+SubIndex_Obs],   DF_Day$gps_latitude[Index_Obs+SubIndex_Obs])
        Time_P2 <- DF_Day$datetime_local[Index_Obs+SubIndex_Obs]

        ## Calculate distance
        PointDist <- round(distHaversine(Coord_P1, Coord_P2), 1)
        DistArray <- c(DistArray, PointDist)

        ## Update index of nested loop
        SubIndex_Obs <- SubIndex_Obs+1

      }

      ## Reverse last update of index of nested loop
      SubIndex_Obs <- SubIndex_Obs-1

      ## Output: only if SubIndex is larger than 1 (i.e., there was actually a period)
      if (SubIndex_Obs> 1) {
        DF_Output <- rbind(DF_Output, data.frame(DateTimeLocalStart=Time_P1,
                                                 DateTimeLocalEnd=DF_Day$datetime_local[Index_Obs+SubIndex_Obs-1],
                                                 Duration=NA,
                                                 MaxDist=max(DistArray[1:SubIndex_Obs-1]), # Does not include last dist calculation, which exited the loop
                                                 NumObs=SubIndex_Obs)) # Includes first observation
      }

      ## Update Index of outer loop
      Index_Obs <- Index_Obs+SubIndex_Obs

    }

    ## Calculate duration
    DF_Output$Duration <- (DF_Output$DateTimeLocalEnd - DF_Output$DateTimeLocalStart)*24*60*60

    ## Calculating max distance

    ## Filter periods with respect to time threshold
    DF_Output <- DF_Output[DF_Output$Duration>TimeThreshold,]

    ## Output
    return(DF_Output)

  }

  ## Eliminate short periods
  ## ***********************
  eliminateShortPeriods <- function(ObsDF, ColName, SummaryDF, DurationThreshold, Verbose=F) {

    #    ObsDF <- DF_Day
    #    ColName <- "marker_oncatroad"
    #    SummaryDF <- PeriodSummary_CatRoads1
    #    DurationThreshold <- 60
    #    Verbose <- T

    ## Condition for iterative reduction of short periods
    while (nrow(SummaryDF[SummaryDF$Duration<DurationThreshold,])>0) {
      
      ## Loop through summary DF
      for (Index_Period in 1:nrow(SummaryDF)) {

        # Index_Period <- 3

        ## Recognize periods that are shorter than threshold
        if (SummaryDF$Duration[Index_Period] < DurationThreshold) {

          ## Assigning of new value to both ObsDF and SummaryDF (ensures subsequent adjacent short period gets correct value)
          if (Index_Period==1) {
            ObsDF[ObsDF$datetime_local>=SummaryDF$StartDateTimeLocal[Index_Period] & ObsDF$datetime_local<=SummaryDF$EndDateTimeLocal[Index_Period], ColName] <- SummaryDF$Value[Index_Period+1]
            SummaryDF$Value[Index_Period] <- SummaryDF$Value[Index_Period+1]
            if(Verbose) {print(paste0("Index ", Index_Period, ": Replaced ", nrow(ObsDF[ObsDF$datetime_local>=SummaryDF$StartDateTimeLocal[Index_Period] & ObsDF$datetime_local<=SummaryDF$EndDateTimeLocal[Index_Period],]), " obs with next value (", SummaryDF$Value[Index_Period+1], ")."))}
          } else {
            ObsDF[ObsDF$datetime_local>=SummaryDF$StartDateTimeLocal[Index_Period] & ObsDF$datetime_local<=SummaryDF$EndDateTimeLocal[Index_Period], ColName] <- SummaryDF$Value[Index_Period-1]
            SummaryDF$Value[Index_Period] <- SummaryDF$Value[Index_Period-1]
            if(Verbose) {print(paste0("Index ", Index_Period, ": Replaced ", nrow(ObsDF[ObsDF$datetime_local>=SummaryDF$StartDateTimeLocal[Index_Period] & ObsDF$datetime_local<=SummaryDF$EndDateTimeLocal[Index_Period],]), " obs with previous value (", SummaryDF$Value[Index_Period-1], ")."))}
          }

        }

      }

      ## Summarize periods
      SummaryDF <- summarizePeriodsInTimeSeries(ObsDF, ColName)
      
    }

    ## Return
    return(ObsDF)

  }
  
  
  ## Eliminate short distances
  ## *************************
  eliminateShortDistances <- function(ObsDF, ColName, SummaryDF, DistThreshold, Verbose=F) {
    
    #    ObsDF <- DF_Day
    #    ColName <- "marker_oncatroad"
    #    SummaryDF <- PeriodSummary_CatRoads1
    #    DurationThreshold <- 60
    #    Verbose <- T
    
    ## Condition for iterative reduction of short periods
    while (nrow(SummaryDF[SummaryDF$CumDist<DistThreshold,])>0) {
      
      ## Loop through summary DF
      for (Index_Period in 1:nrow(SummaryDF)) {
        
        # Index_Period <- 3
        
        ## Recognize periods that are shorter than threshold
        if (SummaryDF$CumDist[Index_Period] < DistThreshold) {
          
          ## Assigning of new value to both ObsDF and SummaryDF (ensures subsequent adjacent short period gets correct value)
          if (Index_Period==1) {
            ObsDF[ObsDF$datetime_local>=SummaryDF$StartDateTimeLocal[Index_Period] & ObsDF$datetime_local<=SummaryDF$EndDateTimeLocal[Index_Period], ColName] <- SummaryDF$Value[Index_Period+1]
            SummaryDF$Value[Index_Period] <- SummaryDF$Value[Index_Period+1]
            if(Verbose) {print(paste0("Index ", Index_Period, ": Replaced ", nrow(ObsDF[ObsDF$datetime_local>=SummaryDF$StartDateTimeLocal[Index_Period] & ObsDF$datetime_local<=SummaryDF$EndDateTimeLocal[Index_Period],]), " obs with next value (", SummaryDF$Value[Index_Period+1], ")."))}
          } else {
            ObsDF[ObsDF$datetime_local>=SummaryDF$StartDateTimeLocal[Index_Period] & ObsDF$datetime_local<=SummaryDF$EndDateTimeLocal[Index_Period], ColName] <- SummaryDF$Value[Index_Period-1]
            SummaryDF$Value[Index_Period] <- SummaryDF$Value[Index_Period-1]
            if(Verbose) {print(paste0("Index ", Index_Period, ": Replaced ", nrow(ObsDF[ObsDF$datetime_local>=SummaryDF$StartDateTimeLocal[Index_Period] & ObsDF$datetime_local<=SummaryDF$EndDateTimeLocal[Index_Period],]), " obs with previous value (", SummaryDF$Value[Index_Period-1], ")."))}
          }
          
        }
        
      }
      
      ## Summarize periods
      SummaryDF <- summarizePeriodsInTimeSeries(ObsDF, ColName)
      
    }
    
    ## Return
    return(ObsDF)
    
  }
  

  ## Calculate high and low points in a series of elevation values
  ## *************************************************************

  extractHighAndLowPoints <- function(ElevArray, AltDiffThreshold, UpValue="Up", DownValue="Down") {

    #   ElevArray = DF_Day$dem_altitude
    #   AltDiffThreshold = UpDown_AltDiffThreshold
    #   UpValue="Up"
    #   DownValue="Down"

    ## Initialize min and max reference elevations
    AltMax <- AltMin <- ElevArray[min(which(!is.na(ElevArray)))]
    AltMaxIndex <- AltMinIndex <- min(which(!is.na(ElevArray)))
    Direction <- UpValue

    ## Start Output DF
    Output <- data.frame(StartElev=ElevArray[min(which(!is.na(ElevArray)))],
                         StartIndex=min(which(!is.na(ElevArray))),
                         EndElev=NA,
                         EndIndex=NA,
                         Dir=Direction)
    OutputIndex <- 1

    ## Calcuations
    for (PointIndex in (min(which(!is.na(ElevArray)))+1):max(which(!is.na(ElevArray)))) {

      if (!is.na(ElevArray[PointIndex])) {

        if (Direction==UpValue) {
          if (ElevArray[PointIndex]>=AltMax) {
            AltMax <- ElevArray[PointIndex]
            AltMaxIndex <- PointIndex
          } else if (ElevArray[PointIndex]<AltMax-AltDiffThreshold){
            Direction <- DownValue
            Output$EndElev[OutputIndex] <- AltMax
            Output$EndIndex[OutputIndex] <- AltMaxIndex
            Output <- rbind(Output, data.frame(StartElev=AltMax, StartIndex=AltMaxIndex, EndElev=NA, EndIndex=NA, Dir=Direction))
            OutputIndex <- OutputIndex + 1
            AltMin <- ElevArray[PointIndex]
            AltMinIndex <- PointIndex
          }
        } else {
          if (ElevArray[PointIndex]<=AltMin) {
            AltMin <- ElevArray[PointIndex]
            AltMinIndex <- PointIndex
          } else if (ElevArray[PointIndex]>AltMin+AltDiffThreshold){
            Direction <- UpValue
            Output$EndElev[OutputIndex] <- AltMin
            Output$EndIndex[OutputIndex] <- AltMinIndex
            Output <- rbind(Output, data.frame(StartElev=AltMin, StartIndex=AltMinIndex, EndElev=NA, EndIndex=NA, Dir=Direction))
            OutputIndex <- OutputIndex + 1
            AltMax <- ElevArray[PointIndex]
            AltMaxIndex <- PointIndex
          }
        }
      }
    }

    ## Complete end point of last segment
    if (is.na(ElevArray[length(ElevArray)])) {
      Output$EndElev[OutputIndex] <- ElevArray[max(which(!is.na(ElevArray)))]
      Output$EndIndex[OutputIndex] <- max(which(!is.na(ElevArray)))
    } else {
      Output$EndElev[OutputIndex] <- ElevArray[length(ElevArray)]
      Output$EndIndex[OutputIndex] <- length(ElevArray)
    }

    ## Finishing DF
    Output$ElevDiff <- Output$EndElev-Output$StartElev

    ## Checking first row for being correctly identified - if not, combine with second row.
    if(nrow(Output)>1) {
      if (abs(Output$ElevDiff[1]) < AltDiffThreshold) {
        Output$StartElev[2] <- Output$StartElev[1]
        Output$StartIndex[2] <- Output$StartIndex[1]
        Output$ElevDiff[2] <- Output$EndElev[2]-Output$StartElev[2]
        Output <- Output[-1,]
      }
    }

    return(Output)

  }

  ## Assign up and down sections
  ## ***************************
  assignUpAndDownToPoints <- function(ElevArray, AltDiffThreshold, UpValue, DownValue) {

    #   ElevArray = DF_Day$dem_altitude
    #   AltDiffThreshold = UpDown_AltDiffThreshold
    #   UpValue="Up"
    #   DownValue="Down"

    HighAndLowPoints <- extractHighAndLowPoints(ElevArray, AltDiffThreshold, UpValue=UpValue, DownValue=DownValue)

    UpAndDownArray <- rep(NA, length(ElevArray))

    for (i in 1:nrow(HighAndLowPoints)) {
      UpAndDownArray[HighAndLowPoints$StartIndex[i]:HighAndLowPoints$EndIndex[i]] <- as.character(HighAndLowPoints$Dir[i])
    }

    return(UpAndDownArray)

  }


  ## Converts chron datetime to just time
  chronDateTime2Time <- function(chronDate) {

    return(times(paste(hours(chronDate),":",minutes(chronDate), ":", seconds(chronDate))))

  }


  ## Force run line through pasue points
  ## ***********************************

  forceLineThroughPoints <- function(SpLine, SpPoints, DistThreshold) {

    #SpLine <- SPL_Run_SimpleV02
    #SpPoints <- SPP_RunPausePoints
    #DistThreshold <- 10

    ## Transfer
    LineCoordsOut <- SpLine@lines[[1]]@Lines[[1]]@coords

    ## Processing
    for (Index_PausePoints in 1:length(SpPoints)) {

      # Index_PausePoints <- 7
      PausePoint <- SpPoints@coords[Index_PausePoints,c(1:2)]

      Dist <- round(distHaversine(PausePoint, LineCoordsOut),1)
      DistMin <- min(Dist)
      DistMinWhich <- which.min(Dist)

      ## If distance<0.1m, do nothing
      if (DistMin<0.1) {

        ## If distance<=DistThreshold, replace point
      } else if (DistMin<=DistThreshold) {
        LineCoordsOut[DistMinWhich, 1] <- PausePoint[1]
        LineCoordsOut[DistMinWhich, 2] <- PausePoint[2]

        ## If distance>DistThreshold, add point.
      } else {

        ## calculate points that are slightly before and after point with shortest distance
        if (DistMinWhich==1) {
          PointDiff_X <- 0.02 * (LineCoordsOut[DistMinWhich+1, 1]-LineCoordsOut[DistMinWhich, 1])
          PointDiff_Y <- 0.02 * (LineCoordsOut[DistMinWhich+1, 2]-LineCoordsOut[DistMinWhich, 2])
        } else {
          PointDiff_X <- 0.02 * (LineCoordsOut[DistMinWhich, 1]-LineCoordsOut[DistMinWhich-1, 1])
          PointDiff_Y <- 0.02 * (LineCoordsOut[DistMinWhich, 2]-LineCoordsOut[DistMinWhich-1, 2])
        }

        PointBefore <- c(LineCoordsOut[DistMinWhich, 1] - PointDiff_X, LineCoordsOut[DistMinWhich, 2] - PointDiff_Y)
        PointAfter  <- c(LineCoordsOut[DistMinWhich, 1] + PointDiff_X, LineCoordsOut[DistMinWhich, 2] + PointDiff_Y)

        ## Calculate distance for before and after points
        DistBefore <- distHaversine(PausePoint, PointBefore)
        DistAfter  <- distHaversine(PausePoint, PointAfter)

        ## Add point either before or after point with closest distance
        if (DistBefore<DistMin) {
          if(DistMinWhich==1) {
            # If the point is before the start point, do nothing
            # LineCoordsOut <- rbind(PausePoint,
            #                       LineCoordsOut[c(DistMinWhich:nrow(LineCoordsOut)), c(1,2)])
          } else {
            LineCoordsOut <- rbind(LineCoordsOut[c(1:(DistMinWhich-1)), c(1,2)],
                                   PausePoint,
                                   LineCoordsOut[c(DistMinWhich:nrow(LineCoordsOut)), c(1,2)])
          }
        } else {
          if(DistMinWhich==nrow(LineCoordsOut)) {
            # If the point is after the end point, do nothing
            # LineCoordsOut <- rbind(LineCoordsOut[c(1:(DistMinWhich)), c(1,2)],
            #                       PausePoint)
          } else {
            LineCoordsOut <- rbind(LineCoordsOut[c(1:(DistMinWhich)), c(1,2)],
                                   PausePoint,
                                   LineCoordsOut[c((DistMinWhich+1):nrow(LineCoordsOut)), c(1,2)])
          }
        }
      }
    }

    row.names(LineCoordsOut) <- c(1:nrow(LineCoordsOut))
    colnames(LineCoordsOut) <- c("x", "y")


    ## Return updated coordinates
    return (LineCoordsOut)

  }

  ## Calculated index of next start/end of marker period
  ## ***************************************************
  NextStartOrEnd <- function (DF, MarkerColName, StartIndex, MaxIter=1000, BeforeOrAfter="After", StartOrEnd="End", DistThreshold=50, MaxIterMax=1000, NoWarnings=F) {

    #   DF=DF_Day
    #   MarkerColName="marker_break"
    #   StartIndex=1040
    #   MaxIter=1040-538
    #   BeforeOrAfter="Before"
    #   StartOrEnd="End"
    
    if(MaxIter > MaxIterMax) {MaxIter <- MaxIterMax}

    NotFound <- T
    Output_Index <- NA
    Index_Add <- 1
    
    if(tolower(BeforeOrAfter)=="after") {

      while (Index_Add <= MaxIter & ((StartIndex+Index_Add)<=nrow(DF)) & NotFound) {

        if(Index_Add==MaxIter & NoWarnings==F) {warning(paste0("Maximum number of iterations (", MaxIter, ") reached at index ", StartIndex+Index_Add, "!"), immediate.=T)}

        ## Beginning of next period
        if (tolower(StartOrEnd)=="start") {
          
          ## Returns NA if no start transition is found before next NA value shows up
          if(is.na(DF[StartIndex+Index_Add, MarkerColName])) {
            Output_Index <- NA
            NotFound <- F
            ## Transition from 0 to 1 indicating start of next period
          } else if (DF[StartIndex+Index_Add, MarkerColName]==1 & DF[StartIndex+(Index_Add-1), MarkerColName]==0) {
            Output_Index <- StartIndex+Index_Add
            NotFound <- F
            ## Return NA if no start transition is found before next run start marker
          } else if ((!is.na(DF$runmarker_startend[StartIndex+Index_Add])) & (DF$runmarker_startend[StartIndex+Index_Add]=="Start")) {
            Output_Index <- NA
            NotFound <- F
          }

          ## End of current period
        } else if (tolower(StartOrEnd)=="end") {
          
          ## Returns NA if no start transition is found before next NA value shows up
          if(is.na(DF[StartIndex+Index_Add, MarkerColName])) {
            Output_Index <- NA
            NotFound <- F
          ## Transition from 1 to 0
          } else if (DF[StartIndex+Index_Add, MarkerColName]==0 & DF[StartIndex+(Index_Add-1), MarkerColName]==1) {
            Output_Index <- StartIndex+Index_Add-1
            NotFound <- F
          ## Transition when the next run start marker is found
          } else if ((!is.na(DF$runmarker_startend[StartIndex+Index_Add])) & (DF$runmarker_startend[StartIndex+Index_Add]=="Start")) {
            Output_Index <- StartIndex+Index_Add-1
            NotFound <- F
          ## Large distance between two break/pause points  
          } else if (DF[StartIndex+Index_Add, MarkerColName]==1 & DF[StartIndex+(Index_Add-1), MarkerColName]==1 & DF$dist[StartIndex+Index_Add]>DistThreshold) {
            Output_Index <- StartIndex+Index_Add-1
            NotFound <- F
          }
        } else {
          stop("Incorrect value for StartOrEnd parameter")
        }

        Index_Add <- Index_Add + 1

      }

    } else if (tolower(BeforeOrAfter)=="before") {

      while (Index_Add <= MaxIter & ((StartIndex-Index_Add)>=1) & NotFound) {

        if(Index_Add==MaxIter & NoWarnings==F) {warning(paste0("Maximum number of iterations (", MaxIter, ") reached at index ", StartIndex-Index_Add, "!"), immediate.=T)}

        ## Beginning of current period
        if (tolower(StartOrEnd)=="start") {
          
          ## Returns NA if no start transition is found before next NA value shows up
          if(is.na(DF[StartIndex-Index_Add, MarkerColName])) {
            Output_Index <- NA
            NotFound <- F
          ## Transition from 1 to 0
          } else if (DF[StartIndex-Index_Add, MarkerColName]==0 & DF[StartIndex-(Index_Add-1), MarkerColName]==1) {
            Output_Index <- StartIndex-(Index_Add-1)
            NotFound <- F
          ## Transition when the previous run end is found  
          } else if ((!is.na(DF$runmarker_startend[StartIndex-Index_Add])) & (DF$runmarker_startend[StartIndex-Index_Add]=="End")) {
            Output_Index <- StartIndex-(Index_Add-1)
            NotFound <- F
          ## Large distance between two break/pause points  
          } else if (DF[StartIndex-Index_Add, MarkerColName]==0 & DF[StartIndex-(Index_Add-1), MarkerColName]==1 & DF$dist[StartIndex-(Index_Add-1)]>DistThreshold) {
            Output_Index <- StartIndex-(Index_Add-1)
            NotFound <- F
          }
          
        ## End of previous period
        } else if (tolower(StartOrEnd)=="end") {
          
          ## Returns NA if no end transition is found before next NA value shows up
          if(is.na(DF[StartIndex-Index_Add, MarkerColName])) {
            Output_Index <- NA
            NotFound <- F
          ## Transition from 0 to 1 indicating end of previous period
          } else if (DF[StartIndex-Index_Add, MarkerColName]==1 & DF[StartIndex-(Index_Add-1), MarkerColName]==0) {
            Output_Index <- StartIndex-Index_Add
            NotFound <- F
            ## Return NA if no end transition is found before the previous run end marker
          } else if ((!is.na(DF$runmarker_startend[StartIndex-Index_Add])) & (DF$runmarker_startend[StartIndex-Index_Add]=="End")) {
            Output_Index <- NA
            NotFound <- F
          }
        } else {
          stop("Incorrect value for StartOrEnd parameter")
        }

        Index_Add <- Index_Add + 1

      }

    } else {
      stop("Incorrect value for BeforeOrAfter parameter")
    }
    
    ## Return value
    return(Output_Index)

    ## Possibilities for returning NA
    ## before:
    ##  - if no end of a previous period has been found because max iteration, the beginning of DF, or the end marker of the previous run has been reached
    ##  - NA value is encountered
    ## after:
    ##  - if no start of a next period has been found because max iteration, the end of DF, or the start marker of the next run has been reached
    ##  - NA value is encountered

  }
  
  
  ## Find next or previous break/pause and adjust observation records of potential runs
  ## **********************************************************************************
  adjustRunToNextPreviousBreakPause <- function(DF, PotRunNum, IndexStart, IndexEnd, StartOrEnd, PreviousOrNextOrBoth, MaxDistPrevious=1000000, MaxDistNext=1000000, NoWarnings=T, DetailMsg=PotRun_ShowProcessingDetail) {
    
    ## FUNCTION FOR FINDING MIN OR MAX INDEX
    findMinMax <- function(IndexBreak=NA, IndexPause=NA, MinOrMax) {
      if(!is.na(IndexBreak) & !is.na(IndexPause)) {
        if (toupper(MinOrMax) == "MIN") {
          IndexOut <- min(IndexBreak, IndexPause)
        } else if (toupper(MinOrMax) == "MAX") {
          IndexOut <- max(IndexBreak, IndexPause)
        } else {
          stop("incorrect value for MinOrMax parameter in findMinMax")
        }
      } else if (!is.na(IndexBreak)) {
        IndexOut <- IndexBreak
      } else if (!is.na(IndexPause)) {
        IndexOut <- IndexPause
      } else {
        IndexOut <- NA
      }
      return(IndexOut)
    }
    
    ## FUNCTION FOR FINDING CLOSEST INDEX
    findClosest <- function(DF, StartOrEnd, IndexCurrent, IndexNewBefore=NA, IndexNewAfter=NA, MaxDistBefore, MaxDistAfter, DetailMsg=T, OutputStringStart="") {
      
      ## String for messages
      if(toupper(StartOrEnd)=="START") {
        OutputStringPointType <- "Start point - DETAIL: "
      } else {
        OutputStringPointType <- "End point - DETAIL: "
      }
      
      ## Calculation of distance and comparison to thresholds
      if (!is.na(IndexNewBefore) & !is.na(IndexNewAfter)) {
        
        Point_Current <- c(DF$gps_longitude[IndexCurrent],   DF$gps_latitude[IndexCurrent]) 
        Point_Before  <- c(DF$gps_longitude[IndexNewBefore], DF$gps_latitude[IndexNewBefore])
        Point_After   <- c(DF$gps_longitude[IndexNewAfter],  DF$gps_latitude[IndexNewAfter]) 
        
        IndexNewBeforeDist <- round(distHaversine(Point_Current, Point_Before), 0)
        if (IndexNewBeforeDist > MaxDistBefore) {IndexNewBefore <- NA}
        if (is.na(IndexNewBefore)) {
          if(DetailMsg) {print(paste0(OutputStringStart, OutputStringPointType, "Did not find earlier break or pause within distance threshold (", MaxDistBefore, " m)."))}
        } else {
          if(DetailMsg) {print(paste0(OutputStringStart, OutputStringPointType, "IndexNewBeforeDist: ", IndexNewBeforeDist, " m (", (IndexNewBefore-IndexCurrent), " obs earlier)."))}
        }
        
        IndexNewAfterDist  <- round(distHaversine(Point_Current, Point_After), 0)
        if (IndexNewAfterDist  > MaxDistAfter)  {IndexNewAfter <- NA}
        if (is.na(IndexNewAfter)) {
          if(DetailMsg) {print(paste0(OutputStringStart, OutputStringPointType, "Did not find later break or pause within distance threshold (", MaxDistAfter, " m)."))}
        } else {
          if(DetailMsg) {print(paste0(OutputStringStart, OutputStringPointType, "IndexNewAfterDist: ", IndexNewAfterDist," m (", (IndexNewAfter-IndexCurrent), " obs later)."))}
        }
        
      } else if (!is.na(IndexNewBefore)) {
        
        Point_Current <- c(DF$gps_longitude[IndexCurrent],   DF$gps_latitude[IndexCurrent]) 
        Point_Before  <- c(DF$gps_longitude[IndexNewBefore], DF$gps_latitude[IndexNewBefore])
        
        IndexNewBeforeDist <- round(distHaversine(Point_Current, Point_Before), 0)
        if (IndexNewBeforeDist > MaxDistBefore) {IndexNewBefore <- NA}
        if (is.na(IndexNewBefore)) {
          if(DetailMsg) {print(paste0(OutputStringStart, OutputStringPointType, "Did not find earlier break or pause within distance threshold (", MaxDistBefore, " m)."))}
        } else {
          if(DetailMsg) {print(paste0(OutputStringStart, OutputStringPointType, "IndexNewBeforeDist: ", IndexNewBeforeDist, " m (", (IndexNewBefore-IndexCurrent), " obs earlier)."))}
        }
        
      } else if (!is.na(IndexNewAfter)) {
        
        Point_Current <- c(DF$gps_longitude[IndexCurrent],   DF$gps_latitude[IndexCurrent]) 
        Point_After   <- c(DF$gps_longitude[IndexNewAfter],  DF$gps_latitude[IndexNewAfter]) 
        
        IndexNewAfterDist  <- round(distHaversine(Point_Current, Point_After), 0)
        if (IndexNewAfterDist  > MaxDistAfter)  {IndexNewAfter <- NA}
        if (is.na(IndexNewAfter)) {
          if(DetailMsg) {print(paste0(OutputStringStart, OutputStringPointType, "Did not find later break or pause within distance threshold (", MaxDistAfter, " m)."))}
        } else {
          if(DetailMsg) {print(paste0(OutputStringStart, OutputStringPointType, "IndexNewAfterDist: ", IndexNewAfterDist," m (", (IndexNewAfter-IndexCurrent), " obs later)."))}
        }
        
      }
       
      ## Selecting final Index for new start/end and producing output
      
      Output <- list(IndexNew=NA, IndexNewDist=NA)
      
      if (!is.na(IndexNewBefore) & !is.na(IndexNewAfter)) { 
        Output$IndexNew <- ifelse(IndexNewBeforeDist <= IndexNewAfterDist, IndexNewBefore, IndexNewAfter)
        Output$IndexNewDist <- ifelse(IndexNewBeforeDist <= IndexNewAfterDist, IndexNewBeforeDist, IndexNewAfterDist)
      } else if (!is.na(IndexNewBefore)) {
        Output$IndexNew <- IndexNewBefore
        Output$IndexNewDist <- IndexNewBeforeDist
      } else if (!is.na(IndexNewAfter)) {
        Output$IndexNew <- IndexNewAfter
        Output$IndexNewDist <- IndexNewAfterDist
      }
      
      return(Output)
      
    }
    
    ## PREP
    PreviousOrNextOrBoth <- toupper(PreviousOrNextOrBoth)
    StartOrEnd <- toupper(StartOrEnd)
    OutputStringStart <- paste0("Pot Run ", PotRunNum, " - ") 
    
    
    ## FOR START OF RUN
    if (StartOrEnd=="START") {
      
      if (is.na(DF$marker_break[IndexStart])) {
        
        DF$run_numpot[c(IndexStart:IndexEnd)] <- NA
        DF$activity[c(IndexStart:IndexEnd)] <- DF$activity[IndexEnd+1]
        print(paste0(OutputStringStart, "DF_Day$marker_break[", IndexStart, "] is NA --> potential run deleted."))
        IndexNew <- NA
        
      } else {
      
        IndexStartIsBreak <- ifelse(DF$marker_break[IndexStart]==1, T, F)
        IndexStartIsPause <- ifelse(DF$marker_pause[IndexStart]==1, T, F)
        
        ## If initial starting point is already a break -> expand to start
        if (IndexStartIsBreak) {
          
          IndexNewBefore <- NextStartOrEnd(DF, "marker_break", IndexStart, MaxIter=IndexStart-min(which(!is.na(DF$marker_break))), BeforeOrAfter = "Before", StartOrEnd = "Start", NoWarnings=NoWarnings)
          if (is.na(IndexNewBefore)) {
            if (DetailMsg) {print(paste0(OutputStringStart, "Start point - DETAIL: Did not find start of break (iteration limit reached)."))}
            IndexNewBefore <- IndexStart
          }
          IndexNewAfter <- NA
          
        ## If initial starting point is already a pause -> expand to start 
        } else if (IndexStartIsPause) {
          
          IndexNewBefore <- NextStartOrEnd(DF, "marker_pause", IndexStart, MaxIter=IndexStart-min(which(!is.na(DF$marker_pause))), BeforeOrAfter = "Before", StartOrEnd = "Start", NoWarnings=NoWarnings)
          if (is.na(IndexNewBefore)) {
            if(DetailMsg) {print(paste0(OutputStringStart, "Start point - DETAIL: Did not find start of pause (iteration limit reached)."))}
            IndexNewBefore <- IndexStart
          }
          IndexNewAfter <- NA
          
        ## If starting starting point is not a break or pause -> find closest within specified limits
        } else {
        
          ## Find first break or pause before
          if (PreviousOrNextOrBoth=="PREVIOUS" | PreviousOrNextOrBoth=="BOTH") {
            IndexNewBeforeBreak <- NextStartOrEnd(DF, "marker_break", IndexStart, MaxIter=IndexStart-min(which(!is.na(DF$marker_break))), BeforeOrAfter = "Before", StartOrEnd = "Start", NoWarnings=NoWarnings)
            IndexNewBeforePause <- NextStartOrEnd(DF, "marker_pause", IndexStart, MaxIter=IndexStart-min(which(!is.na(DF$marker_pause))), BeforeOrAfter = "Before", StartOrEnd = "Start", NoWarnings=NoWarnings)
            IndexNewBefore <- findMinMax(IndexNewBeforeBreak, IndexNewBeforePause, "Max")
            if (is.na(IndexNewBefore) & DetailMsg) {print(paste0(OutputStringStart, "Start point - DETAIL: Did not find earlier break or pause (iteration limit reached)."))}
          } else {
            IndexNewBefore <- NA
          }  
          
          ## Find first break or pause after
          if (PreviousOrNextOrBoth=="NEXT" | PreviousOrNextOrBoth=="BOTH") {
            IndexNewAfterBreak <- NextStartOrEnd(DF, "marker_break", IndexStart, MaxIter=IndexEnd-IndexStart, BeforeOrAfter = "After", StartOrEnd = "Start", NoWarnings=NoWarnings)
            IndexNewAfterPause <- NextStartOrEnd(DF, "marker_pause", IndexStart, MaxIter=IndexEnd-IndexStart, BeforeOrAfter = "After", StartOrEnd = "Start", NoWarnings=NoWarnings)
            IndexNewAfter <- findMinMax(IndexNewAfterBreak, IndexNewAfterPause, "Min")
            if (is.na(IndexNewAfter) & DetailMsg) {print(paste0(OutputStringStart, "Start point - DETAIL: Did not find later break or pause (iteration limit reached)."))}
          } else {
            IndexNewAfter <- NA
          } 
          
        }
        
        ## Find closest pause or break points
        IndexNewInfo <- findClosest(DF, "Start", IndexStart, IndexNewBefore, IndexNewAfter, MaxDistPrevious, MaxDistNext, DetailMsg=DetailMsg, OutputStringStart)
        IndexNew <- IndexNewInfo$IndexNew
        IndexNewDist <- IndexNewInfo$IndexNewDist
        

        ## Adjust records for potential run
        if (is.na(IndexNew)) {
          
          ## Delete entire run
          DF$run_numpot[c(IndexStart:IndexEnd)] <- NA
          DF$activity[c(IndexStart:IndexEnd)] <- DF$activity[IndexEnd+1]
          
        } else if(IndexNew > IndexStart) {
          
          ## Delete unnecessary run tags at beginning
          DF$run_numpot[c(IndexStart:(IndexNew-1))] <- NA
          DF$activity[c(IndexStart:(IndexNew-1))] <- DF$activity[IndexStart-1]
          
        } else if (IndexNew < IndexStart) {
          
          ## Add new run tags records at beginning
          DF$run_numpot[c(IndexNew:IndexStart)] <- PotRunNum
          
        } else {
          
          ## Nothing is required
          
        }
        
        
        ## Output
        if(IndexStartIsBreak) {
          EndString <- " (Expanded to beginning of existing break)."
        } else if (IndexStartIsPause) {
          EndString <- " (Expanded to beginning of existing pause)."
        } else if (!is.na(IndexNew)) {
          if (IndexNew > IndexStart) {
            EndString <- " (Beginning of next break/pause)."
          } else if (IndexNew < IndexStart) {
          EndString <- " (Beginning of previous break/pause)."
          }
        }
        
        if (is.na(IndexNew)) {
          print(paste0(OutputStringStart, "No meaningful start point found --> potential run deleted."))
        } else if(IndexNew > IndexStart) {
          print(paste0(OutputStringStart, "Adjusted start point (", IndexNew, "): ", IndexNew-IndexStart, " obs later, ", IndexNewDist, " m away", EndString))
        } else if (IndexNew < IndexStart) {
          print(paste0(OutputStringStart, "Adjusted start point (", IndexNew, "): ", IndexNew-IndexStart, " obs earlier, ", IndexNewDist, " m away", EndString))
        } else {
          print(paste0(OutputStringStart, "Original start point not moved (", IndexNew, ")."))
        }
        
      }
      
    ## FOR END OF RUN
    } else if (StartOrEnd=="END") {
      
      if(is.na(DF$marker_break[IndexEnd])) {
        
        DF$run_numpot[c(IndexStart:IndexEnd)] <- NA
        DF$activity[c(IndexStart:IndexEnd)] <- DF$activity[IndexEnd+1]
        print(paste0(OutputStringStart, "DF_Day$marker_break[", IndexEnd, "] is NA --> potential run deleted."))
        IndexNew <- NA
        
      } else {
      
        IndexEndIsBreak <- ifelse(DF$marker_break[IndexEnd]==1, T, F)
        IndexEndIsPause <- ifelse(DF$marker_pause[IndexEnd]==1, T, F)
        
        ## If initial end point is already a break -> expand to end
        if (IndexEndIsBreak) {
          
          IndexNewBefore <- NA
          IndexNewAfter <- NextStartOrEnd(DF, "marker_break", IndexEnd, MaxIter=max(which(!is.na(DF$marker_break)))-IndexEnd, BeforeOrAfter = "After", StartOrEnd = "End", NoWarnings=NoWarnings)
          if (is.na(IndexNewAfter)) {
            if (DetailMsg) {print(paste0(OutputStringStart, "End point - DETAIL: Did not find later end of break (iteration limit reached)."))}
            IndexNewAfter <- IndexEnd
          }
          
        ## If initial end point is already a pause -> expand to end 
        } else if (IndexEndIsPause) {
          
          IndexNewBefore <- NA
          IndexNewAfter <- NextStartOrEnd(DF, "marker_pause", IndexEnd, MaxIter=max(which(!is.na(DF$marker_break)))-IndexEnd, BeforeOrAfter = "After", StartOrEnd = "End", NoWarnings=NoWarnings)
          if (is.na(IndexNewAfter)) {
            if (DetailMsg) {print(paste0(OutputStringStart, "End point - DETAIL: Did not find later end of pause (iteration limit reached)."))}
            IndexNewAfter <- IndexEnd
          }
          
        ## If end point is not a break or pause -> find closest within specified limits
        } else {
          
          ## Find first break or pause before
          if (PreviousOrNextOrBoth=="PREVIOUS" | PreviousOrNextOrBoth=="BOTH") {
            IndexNewBeforeBreak <- NextStartOrEnd(DF, "marker_break", IndexEnd, MaxIter=IndexEnd-IndexStart, BeforeOrAfter = "Before", StartOrEnd = "End", NoWarnings=NoWarnings)
            IndexNewBeforePause <- NextStartOrEnd(DF, "marker_pause", IndexEnd, MaxIter=IndexEnd-IndexStart, BeforeOrAfter = "Before", StartOrEnd = "End", NoWarnings=NoWarnings)
            IndexNewBefore <- findMinMax(IndexNewBeforeBreak, IndexNewBeforePause, "Max")
            if (is.na(IndexNewBefore) & DetailMsg) {print(paste0(OutputStringStart, "End point - DETAIL: Did not find earlier break or pause for end point (iteration limit reached)."))}
          } else {
            IndexNewBefore <- NA
          }  
            
          ## Find first break or pause after
          if (PreviousOrNextOrBoth=="NEXT" | PreviousOrNextOrBoth=="BOTH") {
            IndexNewAfterBreak <- NextStartOrEnd(DF, "marker_break", IndexEnd, MaxIter=max(which(!is.na(DF$marker_break)))-IndexEnd, BeforeOrAfter = "After", StartOrEnd = "End", NoWarnings=NoWarnings)
            IndexNewAfterPause <- NextStartOrEnd(DF, "marker_pause", IndexEnd, MaxIter=max(which(!is.na(DF$marker_pause)))-IndexEnd, BeforeOrAfter = "After", StartOrEnd = "End", NoWarnings=NoWarnings)
            IndexNewAfter <- findMinMax(IndexNewAfterBreak, IndexNewAfterPause, "Min")
            if (is.na(IndexNewAfter) & DetailMsg) {print(paste0(OutputStringStart, "DETAIL: Did not find later break or pause for end point (iteration limit reached)."))}
          } else {
            IndexNewAfter <- NA
          } 
         
        }
        
        ## Find closest pause or break points
        IndexNewInfo <- findClosest(DF, "End", IndexEnd, IndexNewBefore, IndexNewAfter, MaxDistPrevious, MaxDistNext, DetailMsg=DetailMsg, OutputStringStart)
        IndexNew <- IndexNewInfo$IndexNew
        IndexNewDist <- IndexNewInfo$IndexNewDist
      

        ## Adjust records for potential run
        if (is.na(IndexNew)) {
          
          ## Delete entire run
          DF$run_numpot[c(IndexStart:IndexEnd)] <- NA
          DF$activity[c(IndexStart:IndexEnd)] <- DF$activity[IndexEnd+1]
          
        } else if(IndexNew > IndexEnd) {
          
          ## Run longer: Add new run tags at end
          DF$run_numpot[c(IndexEnd:IndexNew)] <- PotRunNum
          
        } else if (IndexNew < IndexEnd) {
          
          ## Run shorter: Delete unnecessary run tags at end
          DF$run_numpot[c((IndexNew+1):IndexEnd)] <- NA
          DF$activity[c((IndexNew+1):IndexEnd)] <- DF$activity[IndexEnd+1]
          
        } else {
          
          ## Nothing is required
          
        }
        
        ## Output
        if(IndexEndIsBreak) {
          EndString <- " (Expanded to end of existing break)."
        } else if (IndexEndIsBreak) {
          EndString <- " (Expanded to end of existing pause)."
        } else if (!is.na(IndexNew)) {
          if (IndexNew > IndexEnd) {
            EndString <- " (End of next break/pause)."
          } else if (IndexNew < IndexEnd) {
            EndString <- " (End of previous break/pause)."
          }  
        }
      
        if (is.na(IndexNew)) {
          print(paste0(OutputStringStart, "No meaningful end point found --> potential run deleted."))
        } else if (IndexNew > IndexEnd) {
          print(paste0(OutputStringStart, "Adjusted end point (", IndexNew, "): ", IndexNew-IndexEnd, " obs later, ", IndexNewDist, " m away", EndString))
        } else if (IndexNew < IndexEnd) {
          print(paste0(OutputStringStart, "Adjusted end point (", IndexNew, "): ", IndexNew-IndexEnd, " obs earlier, ", IndexNewDist, " m away", EndString))
        } else {
          print(paste0(OutputStringStart, "Original end point not moved (", IndexNew, ")."))
        }
        
      }
      
    }
    
    ## Create output
    if (StartOrEnd=="START") {
      AdjustedIndexStart <-  ifelse(is.na(IndexNew), IndexStart, IndexNew)
      AdjustedIndexEnd <- IndexEnd
    }
    if (StartOrEnd=="END") {
      AdjustedIndexStart <- IndexStart
      AdjustedIndexEnd <-  ifelse(is.na(IndexNew), IndexEnd, IndexNew)
    }
    
    Output <- list(DataFrame = DF,
                   AdjustedIndexStart = AdjustedIndexStart,
                   AdjustedIndexEnd = AdjustedIndexEnd)
    
    ## Return output
    return (Output)

  }

  ## Write DF_Day to DB or CVS
  ## *************************

  writeDFDayToDBorCSV <- function(Operation, DF_Day, WriteToDB, CSVFileName, DBType="Main") {

    ## Create spatial points
    SPP_ObsPoints <- createSpatialPointsFromCoord(DF_Day, x="gps_longitude", y="gps_latitude")

    ## Eliminate irrelevant columns from dataframe
    SPP_ObsPoints_Data <- deleteDFColumnsByName(DF_Day, c("gps_longitude", "gps_latitude"))
    SPP_ObsPoints_Data <- deleteDFColumnsByName(SPP_ObsPoints_Data, c("date_gmt", "time_gmt"))

    ## Create SPDF
    SPDF_ObsPoints <- SpatialPointsDataFrame(SPP_ObsPoints, SPP_ObsPoints_Data)

    ## Write to DB or CSV
    if(WriteToDB) {
      writeSPDFToGPSDB(SPDF_ObsPoints, Operation, Schema="gps", DBTbl="obspoints", DBType=DBType, ShowCounter = 200, ObjLabel="obs point")
    } else {
      writeSPDFToCSV(SPDF_ObsPoints, CSVFileName, na = "")
    }

  }


  ## Write Info_TrackDays to DB or CSV
  ## *********************************

  writeInfotrackDaysToDBorCSV <- function(Operation, Info_TrackDays, WriteToDB, CSVFileName, DBType="Main") {

    ## Write to DB or CSV
    if(WriteToDB) {
      writeDFToGPSDB(Info_TrackDays, Operation, "gps", "tracks_days", DBType=DBType, ObjLabel="trackday")
    } else {
      write.csv(Info_TrackDays, file=CSVFileName, na = "")
    }

  }

  
  ## Write DF_PotRunSkipped to DB or CSV
  ## **********************************
  
  writePotRunSkippedToDBorCSV <- function(Operation, DF, WriteToDB, CSVFileName, DBType="Main") {
    
    ## Write to DB or CSV
    if(WriteToDB) {
      if(nrow(DF)) {
        writeDFToGPSDB(DF, Operation, "gps", "potruns_skipped", DBType=DBType, ObjLabel="skipped run")
      }
    } else {
      write.csv(DF, file=CSVFileName, na = "")
    }
    
  }

  ## Eliminate loops in line
  ## ************************

  eliminateLoopsInLine <- function (SPL_Run, UUIDRun, CRS_String=Const_GlobalCrsString) {

    ## Create bogus line
    BogusLine <- readWKT("LINESTRING(0 0, 1 1)")
    BogusLine@proj4string <- CRS(CRS_String)

    ## Split original line to create segments
    Split <- gDifference(SPL_Run, BogusLine)
    NumSplitSeg <- length(Split@lines[[1]]@Lines)

    ## Extract coordinates of start and end points of each segment
    DF_StartEndPoints <- data.frame(Index=c(1:NumSplitSeg),
                                    NumPoints=NA,
                                    StartX=NA,
                                    StartY=NA,
                                    EndX=NA,
                                    EndY=NA,
                                    Next=NA)

    for (Index_Split in 1:NumSplitSeg) {

      Coords <- Split@lines[[1]]@Lines[[Index_Split]]@coords
      NumPoints <- nrow(Coords)

      DF_StartEndPoints$NumPoints[Index_Split] <- NumPoints
      DF_StartEndPoints$StartX[Index_Split] <- Coords[1,1]
      DF_StartEndPoints$StartY[Index_Split] <- Coords[1,2]
      DF_StartEndPoints$EndX[Index_Split] <- Coords[NumPoints,1]
      DF_StartEndPoints$EndY[Index_Split] <- Coords[NumPoints,2]

    }
    rm(Coords, NumPoints, Index_Split)

    ## Determine sequence of segments that always connect the end point with the last segment that has that point as a starting point.
    Index_Segment <- 1
    while (Index_Segment < NumSplitSeg) {

      CurrentSeg_EndX <- DF_StartEndPoints$EndX[Index_Segment]
      CurrentSeg_EndY <- DF_StartEndPoints$EndY[Index_Segment]

      Index_Segment_Next <- max(DF_StartEndPoints$Index[DF_StartEndPoints$StartX==CurrentSeg_EndX & DF_StartEndPoints$StartY==CurrentSeg_EndY & DF_StartEndPoints$Index>Index_Segment])
      if(is.infinite(Index_Segment_Next)){Index_Segment_Next <- NumSplitSeg+1} ## Adds hypothetical link to next for last segment
      DF_StartEndPoints$Next[Index_Segment] <- Index_Segment_Next

      Index_Segment <- Index_Segment_Next

    }

    if(Index_Segment_Next <= NumSplitSeg) {DF_StartEndPoints$Next[NumSplitSeg] <- NumSplitSeg+1}  ## Adds hypothetical link to next for last segment

    ## List of segments to be included
    ListOfSegments <- DF_StartEndPoints$Index[!is.na(DF_StartEndPoints$Next)]

    ## Clean up
    rm(Index_Segment, Index_Segment_Next, NumSplitSeg)
    rm(DF_StartEndPoints, CurrentSeg_EndX, CurrentSeg_EndY)

    ## Piece together coordinates for new line (omits first point of each added segment)
    LineCoord <- Split@lines[[1]]@Lines[[1]]@coords
    for (Index_Seg in 2:length(ListOfSegments)) {
      LineCoord <- rbind(LineCoord, Split@lines[[1]]@Lines[[ListOfSegments[Index_Seg]]]@coords[-1,])
    }
    rm(ListOfSegments, Index_Seg)

    ## Turn merged line coordinates into spatial line
    SPL_Run_Simple <- createSpatialLineFromCoord(LineCoord, UUIDRun, x="x", y="y")
    SPL_Run_Simple@proj4string <- CRS(CRS_String)
    rm(LineCoord)

    ## Return simplified line
    return(SPL_Run_Simple)

  }
  
  ## Write reason for skipped run into dataframe
  writeReasonForSkippedRunIntoDF <- function(DF, gpstracksdays_uuid, run_numpot, reason) {
  
    DF <- rbind(DF, data.frame(uuid=createUuid(),
                               gpstracksdays_uuid=gpstracksdays_uuid,
                               run_numpot=run_numpot,
                               reason=reason))
    return(DF)
  }


## vvvvvvvvvvvvvvvvvvvvv
## 4. GETTING READY ====
## ^^^^^^^^^^^^^^^^^^^^^

  ## Create data frame of progress report and fill with input parameters and constants
  ## *********************************************************************************
  Output_Progress <- data.frame(Comment = character(0),
                                Num = numeric(0),
                                DTCre = numeric(0))

  Output_Progress <- addToProgressDF(Comment="PROCESSING GPS TRACKS")
  Output_Progress <- addToProgressDF(Comment="*********************")
  Output_Progress <- addToProgressDF()

  ## Input parameters
  Output_Progress <- addToProgressDF(Comment="Input parameters")
  Output_Progress <- addToProgressDF(Comment=createParameterString(Operation))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Folder_GPSTracks))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Folder_OutputRoot))
  Output_Progress <- addToProgressDF(Comment=createParameterString(UseElevRaster))
  Output_Progress <- addToProgressDF(Comment=createParameterString(WriteToDB))
  Output_Progress <- addToProgressDF(Comment=createParameterString(WriteChartsToFile))
  if (!is.na(Process_TrackFileNames[1])) {Output_Progress <- addToProgressDF(Comment=createParameterString(Process_TrackFileNames))}
  Output_Progress <- addToProgressDF(Comment=createParameterString(Process_FirstTrackIndex))
  if (!is.na(Process_TrackdayDates)) {Output_Progress <- addToProgressDF(Comment=createParameterString(Process_TrackdayDates))}
  if (NeedToDownSampleGPSTrack) {
    Output_Progress <- addToProgressDF(Comment=createParameterString(NeedToDownSampleGPSTrack))
    Output_Progress <- addToProgressDF(Comment=createParameterString(TargetRecordingInterval))
  }
  Output_Progress <- addToProgressDF(Comment=createParameterString(InclRunUseList))
  Output_Progress <- addToProgressDF(Comment=createParameterString(InclRunUseListFromPrevProcTrackdays))
  Output_Progress <- addToProgressDF()

  ## Constants
  Output_Progress <- addToProgressDF(Comment="Constants")
  Output_Progress <- addToProgressDF(Comment=createParameterString(TimeDiffGMT, "hrs"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(MinNumObsInsideOpAreaForProcessing))
  Output_Progress <- addToProgressDF(Comment=createParameterString(DilutionThreshold))
  Output_Progress <- addToProgressDF(Comment=createParameterString(GPSSpatialAccuracy, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(OpArea_Buffer, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Flight_MinSpeed, "km/h"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Flight_MinDuration, "sec"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Flight_MinAltGain, "sec"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(CatRoad_MaxDist, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(CatRoad_MinDist, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(CatRoad_MinDuration, "sec"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Pause_MinTime, "sec"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Pause_MoveMinTime, "sec"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Pause_DistRadius, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Break_MinTime, "sec"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Break_MoveMinTime, "sec"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Break_DistRadius, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(UpDown_AltDiffThreshold, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(UpDown_MinDuration, "sec"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(UpDown_StartEndAltBuffer, "m"))
  # Output_Progress <- addToProgressDF(Comment=createParameterString(NewRun_DistToPrevObsWithAltDiffReq, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(NewRun_DistToPrevObsWithInclReq_Dist, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(NewRun_DistToPrevObsWithInclReq_Incl, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(NewRun_AltDiffToPrevObs, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(NewRun_DistToPrevObsWithoutInclReq, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(PotRun_MinDuration, "sec"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(PotRun_MinDurationSkiing, "sec"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(PotRun_ShowProcessingDetail))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Run_MinLength, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Run_MinMaxSkiingSpeed, "km/h"))

  Output_Progress <- addToProgressDF(Comment=createParameterString(Simplify_RemoveBightMaxDist, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Simplify_ForcePausePointsReplaceDist, "m"))
  Output_Progress <- addToProgressDF(Comment=createParameterString(Simplify_SimplifyThreshold, "m"))

  Output_Progress <- addToProgressDF()


  ## Extract activities for operation from DB (for extraction rules)
  ## ***************************************************************
  Oper_Activities <- getRecordsFromQuery(Operation, "SELECT * FROM misc.activities", DBType=DBType)
  if (nrow(Oper_Activities)==0) {stop("Information on operation activities MUST be stored in GPS DB!")}

  ## Get information on local coordinate reference system from database
  ## ******************************************************************
  Input_CRSLocal <- getLocalSRDI(Operation)$proj4text
  if (is.na(Input_CRSLocal)) {stop("Information on local CRS MUST be stored in GPS DB!")}

  Output_Progress <- addToProgressDF(Comment=createParameterString(Input_CRSLocal))

  ## Spatial dataframe with operating area
  ## **************************************
  OpAreaUUID <- getRecordsFromQuery(Operation, "SELECT uuid FROM loccat.oparea", DBType=DBType)
  if (nrow(OpAreaUUID)>0) {SPDF_OpArea <- getSpatialObjectsDataFrameFromUUID(Operation=Operation, Schema="loccat", DBTbl="oparea", DBType=DBType, UUID=OpAreaUUID$uuid, ResultCol="name")
  } else {
    stop("No operating area polygons available!")
  }

  ## Progress report dataframe
  Output_Progress <- addToProgressDF(Comment="Loaded SPDF_OpArea - number of polygons", Num=nrow(OpAreaUUID))
  rm(OpAreaUUID)

  ## Adding buffer to SPDF_OPArea
  if(OpArea_Buffer>0) {
    SPDF_OpArea_Trans <- spTransform(SPDF_OpArea, Input_CRSLocal)
    SPDF_OpAreaWithBuffer_Trans <- gBuffer(SPDF_OpArea_Trans, width=OpArea_Buffer)
    SPDF_OpArea <- spTransform(SPDF_OpAreaWithBuffer_Trans, Const_GlobalCrsString)

    Output_Progress <- addToProgressDF(Comment=paste0("Added buffer to operating area: ", OpArea_Buffer, "m"))
    rm(SPDF_OpArea_Trans, SPDF_OpAreaWithBuffer_Trans)
  }


  ## Spatial dataframe with exclusion polygon (roads, towns, lodge, etc.)
  ## ********************************************************************
  ExclAreasUUID <- getRecordsFromQuery(Operation, "SELECT uuid FROM loccat.exclusionareas", DBType=DBType)
  if (nrow(ExclAreasUUID)>0) {
    SPDF_ExclAreas <- getSpatialObjectsDataFrameFromUUID(Operation, Schema="loccat", DBTbl="exclusionareas", UUID=ExclAreasUUID$uuid, DBType=DBType, ResultCol="name")
  }

  ## Progress report dataframe
  Output_Progress <- addToProgressDF(Comment="Loaded SPDF_ExclArea - number of polygons", Num=nrow(ExclAreasUUID))
  rm(ExclAreasUUID)


  ## Spatial dataframe with inclusion polygon (neighbouring area outside of tenure)
  ## ******************************************************************************
  InclAreasUUID <- getRecordsFromQuery(Operation, "SELECT uuid FROM loccat.inclusionareas", DBType=DBType)
  if (nrow(InclAreasUUID)>0) {
    SPDF_InclAreas <- getSpatialObjectsDataFrameFromUUID(Operation, Schema="loccat", DBTbl="inclusionareas", UUID=InclAreasUUID$uuid, DBType=DBType, ResultCol="name")
  }
  
  ## Progress report dataframe
  Output_Progress <- addToProgressDF(Comment="Loaded SPDF_InclAreas - number of polygons", Num=nrow(InclAreasUUID))
  rm(InclAreasUUID)
  
  
  ## Spatial dataframe with cat skiing area
  ## **************************************
  if (Oper_Activities$value[Oper_Activities$name=="Cat"]=="Some") {

    CatAreasUUID <- getRecordsFromQuery(Operation, "SELECT uuid FROM loccat.catareas", DBType=DBType)
    if (nrow(CatAreasUUID)>0) {
      SPDF_CatAreas <- getSpatialObjectsDataFrameFromUUID(Operation, Schema="loccat", DBTbl="catareas", UUID=CatAreasUUID$uuid, DBType=DBType, ResultCol="name")
    }

    ## Progress report dataframe
    Output_Progress <- addToProgressDF(Comment="Loaded SPDF_CatArea - number of polygons", Num=nrow(CatAreasUUID))
    rm(CatAreasUUID)

  }

  ## Spatial dataframe with relevant cat roads
  ## *****************************************
  if (Oper_Activities$value[Oper_Activities$name=="Cat"]=="Some" | Oper_Activities$value[Oper_Activities$name=="Cat"]=="Yes") {

    CatRoadsUUID <- getRecordsFromQuery(Operation, "SELECT uuid FROM loccat.catroads", DBType=DBType)
    if(nrow(CatRoadsUUID)>0) {
      SPDF_CatRoads <- getSpatialObjectsDataFrameFromUUID(Operation, Schema="loccat", DBTbl="catroads", UUID=CatRoadsUUID$uuid, DBType=DBType, ResultCol=c("name", "incl_gps_processing"))
    }

    ## Extract relevant cat roads
    SPDF_CatRoads_ForRunIdentification <- SPDF_CatRoads[SPDF_CatRoads$incl_gps_processing=="Yes",]
    
    ## Progress report dataframe
    Output_Progress <- addToProgressDF(Comment="Loaded SPDF_CatRoads - number of lines", Num=nrow(CatRoadsUUID))
    Output_Progress <- addToProgressDF(Comment="Number of cat roads relevant for run identification", Num=nrow(SPDF_CatRoads_ForRunIdentification))
    rm(CatRoadsUUID)

    ## Creating buffered cat roads
    SPDF_CatRoads_ForRunIdentification_Trans <- spTransform(SPDF_CatRoads_ForRunIdentification, Input_CRSLocal)
    SPDF_CatRoads_ForRunIdentificationBuffer_Trans <- gBuffer(SPDF_CatRoads_ForRunIdentification_Trans, byid = T, width=CatRoad_MaxDist)
    SPDF_CatRoads_ForRunIdentificationBuffer <- spTransform(SPDF_CatRoads_ForRunIdentificationBuffer_Trans, Const_GlobalCrsString)
    
    ## Progress report dataframe
    Output_Progress <- addToProgressDF(Comment=paste0(CatRoad_MaxDist, "m buffer created around relevant cat roads."))
    rm(SPDF_CatRoads_ForRunIdentification_Trans)
    rm(SPDF_CatRoads_ForRunIdentificationBuffer_Trans)

  }

  ## Initialize dataframe for user feedback
  ## **************************************
  Output_Feedback <- data.frame(StartTime=Sys.time(),
                                NumTracks=0,
                                NumTrackDaysProcessed = 0,
                                NumTrackDaysSkipped = 0,
                                NumRunsIdentified=0,
                                EndTime=NA)


  ## Listing of tracks to be processed
  ## *********************************
  ## Regular case
  if(is.na(Process_TrackFileNames[1])) {
    GPSTrackFileList <- list.files(path = Folder_GPSTracks, pattern = ".csv")
    ## When an individual track is targeted
  } else {
    GPSTrackFileList <- Process_TrackFileNames
  }


## vvvvvvvvvvvvvvvvvvvvvvvvvvvv
## 5. PROCESSING OF TRACKS ====
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^

  if (length(GPSTrackFileList)>0) {

    for (Index_Track in Process_FirstTrackIndex:length(GPSTrackFileList)) {

      # Index_Track <- Process_FirstTrackIndex

##### vvvvvvvvvvvvvvvvvvvvvvvvv
##### 5.1 READING OF TRACK ====
##### ^^^^^^^^^^^^^^^^^^^^^^^^^

      ## Reading of file name
      TrackFileName <- GPSTrackFileList[Index_Track]

      ## Reading of raw GPS track file
      DF_Track <- read.csv2 (paste0(Folder_GPSTracks, createOSSpecFolderLink(), TrackFileName), dec=".", skip=6)
      
      ## Eliminate unnecessary columns
      DF_Track <- DF_Track[,c("Date", "Time", "Latitude", "Longitude", "Quality", "NumSat", "Dilution", "Altitude", "Geoid.Separation")]
      

      ## User feedback
      print("")
      print("******************************************************")
      print(paste("*** PROCESSING TRACK", Index_Track, "OF", length(GPSTrackFileList), "-", TrackFileName, "***"))
      print("******************************************************")
      print(Sys.time())

      ## Progress Output DF
      Output_Progress <- addToProgressDF()
      Output_Progress <- addToProgressDF(Comment=paste("PROCESSING TRACK", TrackFileName, "..."))
      Output_Progress <- addToProgressDF(Comment="Original number of observations: ", Num=nrow(DF_Track))
      print(paste("Original number of observations: ", nrow(DF_Track)))



##### vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
##### 5.2 CREATING DATA FRAMES FOR TRACK AND TRACKDAYS INFORMATION ====
##### ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

      ## File name without extension: used for output file names
      TrackFileNameShort <- substring(TrackFileName, 1, nchar(TrackFileName)-4)

      ## Extracting information from track file name
      if(is.na(FileNameThirdPart)) {
        Guide <- NA
        Program <- NA
      } else if(tolower(FileNameThirdPart)=="guide") {
        Guide <- strsplit(TrackFileNameShort, "_")[[1]][3]
        Program <- NA
      } else if (tolower(FileNameThirdPart)=='program') {
        Guide <- NA
        Program <- strsplit(TrackFileNameShort, "_")[[1]][3]
      } else {
        stop("Incorrect value for parameter 'FileNameThirdPart'. Value can only be 'guide', 'program' or NA!")
      }
      
      ## Check whether a records for this track already exists in the DB
      Flag_TrackRecordExistsInDB <- F
      if (OnlyProcessTrackdaysNotInDB | WriteToDB) {
        Query <- paste0("SELECT uuid FROM gps.tracks WHERE raw_file='", TrackFileName, "'")
        DF_TrackUUID <- getRecordsFromQuery(Operation, Query, DBType=DBType)

        if (nrow(DF_TrackUUID)==1) {
          UUIDTrack <- DF_TrackUUID$uuid
          Flag_TrackRecordExistsInDB <- T
        } else if (nrow(DF_TrackUUID)==0) {
          UUIDTrack <- createUuid()
        } else {
          stop("Multiple records for same track in PostgreSQL/PostGIS GPS database!")
        }
        rm(DF_TrackUUID)
      } else {
        UUIDTrack <- createUuid()
      }


      ## Creation Info_Track dataframe
      Info_Track <- data.frame(uuid = UUIDTrack,
                               unit = strsplit(TrackFileName, "_")[[1]][2],
                               raw_file = TrackFileName,
                               datetime_local_start = NA,
                               datetime_local_end = NA,
                               numobs_original = nrow(DF_Track),
                               numobs_final = NA)


##### vvvvvvvvvvvvvvvvvvvvvvvvvvv
##### 5.3 INITIAL PROCESSING ====
##### ^^^^^^^^^^^^^^^^^^^^^^^^^^^

      ## Change names of columns for obspoints table in database
      colnames(DF_Track) <- paste0("gps_", gsub(".", "_", tolower(colnames(DF_Track)), fixed=T))

      ## Create date and time objects for GMT
      DF_Track$date_gmt <- dates(as.character(DF_Track$gps_date), format = c(dates="d.m.y"))
      DF_Track$time_gmt <- times(as.character(DF_Track$gps_time))
      DF_Track$datetime_gmt <- chron (DF_Track$date_gmt, DF_Track$time_gmt)

      ## Eliminate observations that do not have any time records
      NumObsBefore <- nrow(DF_Track)
      DF_Track <- DF_Track[!is.na(DF_Track$datetime_gmt),]
      NumObsAfter <- nrow(DF_Track)
      
      if (NumObsAfter < NumObsBefore) {
        OutputString <- paste0("Eliminated ", NumObsBefore-NumObsAfter , " obs due to missing time value:")
      } else {
        OutputString <- paste0("No records eliminated due to missing time value:")
      }
      
      print(paste(OutputString, nrow(DF_Track)))
      Output_Progress <- addToProgressDF(Comment=OutputString, Num=nrow(DF_Track))
      rm(OutputString)
      rm(NumObsBefore, NumObsAfter)
      
      ## Eliminate sections with duplicate/repeat times
      DF_Track$ToKeep <- T
      ObsDeletedDueToDuplicateTimes <- 0
      
      for (Index_For in 2:nrow(DF_Track)) {
        
        if (DF_Track$ToKeep[Index_For]==T) {
          
          TimeDiff <- DF_Track$datetime_gmt[Index_For] - DF_Track$datetime_gmt[Index_For-1]
          
          if (TimeDiff <= 0) {
            
            LastTime <- DF_Track$datetime_gmt[Index_For-1]
            DF_Track$ToKeep[Index_For] <- F
            ObsDeletedDueToDuplicateTimes <- ObsDeletedDueToDuplicateTimes+1
            Index_While <- 1
            
            while (DF_Track$datetime_gmt[Index_For+Index_While]-LastTime <= 0 & Index_For+Index_While < nrow(DF_Track)) {
              
              DF_Track$ToKeep[Index_For+Index_While] <- F
              Index_While <- Index_While + 1
              ObsDeletedDueToDuplicateTimes <- ObsDeletedDueToDuplicateTimes+1
              
            }
            
          }    
          
        }
        
      }
      
      if (ObsDeletedDueToDuplicateTimes>0) {
        DF_Track <- DF_Track[DF_Track$ToKeep==T,]
        OutputString <- paste0("Eliminated ", ObsDeletedDueToDuplicateTimes , " obs due to duplicate time periods:")
      } else {
        OutputString <- paste0("No duplicate time periods detected:")
      }
        
      print(paste(OutputString, nrow(DF_Track)))
      Output_Progress <- addToProgressDF(Comment=OutputString, Num=nrow(DF_Track))
      rm(OutputString)
      
      rm(ObsDeletedDueToDuplicateTimes, Index_For)
      DF_Track <- deleteDFColumnsByName(DF_Track, "ToKeep")
      if(exists("Index_While")) {rm(Index_While)}
      
      ## Eliminate any duplicate observations
      UniqueDateTime <- unique(DF_Track$datetime_gmt)

      if (length(UniqueDateTime)<nrow(DF_Track)) {
        UniqueObsIndexArray <- numeric(0)
        for(UniqueIndex in 1:length(UniqueDateTime)) {
          UniqueObsIndexArray <- c(UniqueObsIndexArray, min(which(DF_Track$datetime_gmt==UniqueDateTime[UniqueIndex])))
        }

        OutputString <- paste0("Eliminated ", (nrow(DF_Track)-length(UniqueDateTime)) , " duplicate records.")
        DF_Track <- DF_Track[UniqueObsIndexArray,]

        print(paste(OutputString, nrow(DF_Track)))
        Output_Progress <- addToProgressDF(Comment=OutputString, Num=nrow(DF_Track))
        rm(OutputString)

        rm(UniqueObsIndexArray, UniqueIndex)

      }
      rm(UniqueDateTime)

      ## Calculate local time
      DF_Track$datetime_local <- DF_Track$datetime_gmt - TimeDiffGMT/24
      DF_Track$date_local <- dates(DF_Track$datetime_local)
      DF_Track$date_local_esri <- dates(DF_Track$datetime_local)
      DF_Track$time_local <- chronDateTime2Time(DF_Track$datetime_local)

      ## Down sampling data frame to target recording frequency
      if (NeedToDownSampleGPSTrack) {
        RandomIndex <- floor(runif(5, 2, nrow(DF_Track)))
        OriginalRecordingInterval <- round(as.numeric(min(c(DF_Track$time_local[RandomIndex[1]]-DF_Track$time_local[(RandomIndex[1]-1)],
                                                 DF_Track$time_local[RandomIndex[2]]-DF_Track$time_local[(RandomIndex[2]-1)],
                                                 DF_Track$time_local[RandomIndex[3]]-DF_Track$time_local[(RandomIndex[3]-1)],
                                                 DF_Track$time_local[RandomIndex[4]]-DF_Track$time_local[(RandomIndex[4]-1)],
                                                 DF_Track$time_local[RandomIndex[5]]-DF_Track$time_local[(RandomIndex[5]-1)]))*24*60*60), 0)

        if (OriginalRecordingInterval < TargetRecordingInterval) {
          DF_Track <- deleteNth(DF_Track, TargetRecordingInterval/OriginalRecordingInterval)
          ## Progress Output DF
          OutputString <- paste0("After downsampling to increase recording interval from ", as.numeric(OriginalRecordingInterval), " to", TargetRecordingInterval, "sec:")
        } else {
          OutputString <- "No downsampling required:"
        }

        print(paste(OutputString, nrow(DF_Track)))
        Output_Progress <- addToProgressDF(Comment=OutputString, Num=nrow(DF_Track))
        rm(OutputString)

        rm(RandomIndex, OriginalRecordingInterval)

      }

      ## Adding information to Info_Track DF
      Info_Track$numobs_final <- nrow(DF_Track)
      Info_Track$datetime_local_start = min(DF_Track$datetime_local)
      Info_Track$datetime_local_end   = max(DF_Track$datetime_local)


      ## Eliminating observation of low quality
      DF_Track <-  DF_Track[DF_Track$gps_dilution<DilutionThreshold,]

      OutputString <- paste0("Number of points with dilution <", DilutionThreshold, ":")
      print(paste(OutputString, nrow(DF_Track)))
      Output_Progress <- addToProgressDF(Comment=OutputString, Num=nrow(DF_Track))
      rm(OutputString)

      Info_Track$numobs_final = nrow(DF_Track)

      ## Write track info into DB (only if record does not exist yet)
      if(WriteToDB & Flag_TrackRecordExistsInDB==F) {

        writeDFToGPSDB(Info_Track, Operation, "gps", "tracks")

        OutputString <- "Writing Info_Track to PostgreSQL/PostGIS GPS DB."
        print(OutputString)
        Output_Progress <- addToProgressDF(Comment=OutputString)

      } else if (WriteToDB & Flag_TrackRecordExistsInDB==T) {

        OutputString <- "Info_Track record already exists in PostgreSQL DB."
        print(OutputString)
        Output_Progress <- addToProgressDF(Comment=OutputString)

      } else {
        write.csv(Info_Track, file=paste0(Folder_OutputRoot, createOSSpecFolderLink(), "Info_Track_", TrackFileNameShort, ".csv"), na = "")
      }
      rm(Flag_TrackRecordExistsInDB)


      if (nrow(DF_Track)==0) {

        OutputString <- "Track processing completed because no observations after initial processing"
        print(OutputString)
        Output_Progress <- addToProgressDF(Comment=OutputString)

      } else {

##### vvvvvvvvvvvvvvvvvvvvvvvvvvv
##### 5.4 PROCESSING OF DAYS ====
##### ^^^^^^^^^^^^^^^^^^^^^^^^^^^

        ## Overview
        table(DF_Track$date_local)

        ## Create list of dates to be processed (regular and if a specific trackday is processed)
        if (is.na(Process_TrackdayDates)) {
          DateList <- c(min(DF_Track$date_local):max(DF_Track$date_local))
        } else {
          DateList <- chron(Process_TrackdayDates, format="y-m-d")
        }

        ## Loop for days
        for (Index_Date in 1:length(DateList)) {
          # for (Index_Date in 3:3) {

          # Index_Date <- 1

######### vvvvvvvvvvvvvvvvvvvvvvvv
######### 5.4.1 GETTING READY ====
######### ^^^^^^^^^^^^^^^^^^^^^^^^

          ## Create DateString
          DayDate <- chron(DateList[Index_Date])
          DayDateString <- gsub("'", "", convertDateToSQL(DayDate))

          ## Create title and file name
          Chart_Title <- paste0(DayDateString, " - Unit ", Info_Track$unit[1])
          Output_FileNameRoot <- paste(DayDateString, Info_Track$unit[1], sep="_")

          ## Create output folder structure
          DayTrackFolderName <- paste(DayDateString, Info_Track$unit[1], sep="_")
          dir.create(file.path(Folder_OutputRoot, DayTrackFolderName), showWarnings = FALSE)
          dir.create(file.path(paste(Folder_OutputRoot, DayTrackFolderName, sep=createOSSpecFolderLink()), "RunFiles"), showWarnings = FALSE)

          ## Open file for sinking feedback
          sink(paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "ProcessingDetails_", strftime(Sys.time(), "%Y%m%d_%H%M%S"), ".txt"), append=F, split=T)

          ## User feedback and Progress Output DF
          OutputString <- paste("PROCESSING UNIT", Info_Track$unit[1] , "DAY", DayDateString)
          print("")
          print(OutputString)
          print("*********************************")
          print(Sys.time())
          print("")

          Output_Progress <- addToProgressDF()
          Output_Progress <- addToProgressDF(Comment=OutputString)
          rm(OutputString)

          ## Source file
          OutputString <- paste0("Source file: ", TrackFileName)
          print(OutputString)
          Output_Progress <- addToProgressDF(Comment=OutputString)
          rm(OutputString)
          
          ## User feedback and Progress Output DF
          OutputString <- paste0("Created directory ", Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName)
          print(OutputString)
          Output_Progress <- addToProgressDF(Comment=OutputString)
          rm(OutputString)

          ## Deleting Output SPDF
          if (exists("SPDF_Runs")) {rm(SPDF_Runs)}
          if (exists("SPDF_Runs_PausePoints")) {rm(SPDF_Runs_PausePoints)}
          if (exists("DF_PotRuns_Skipped")) {rm(DF_PotRuns_Skipped)}

          ## Checking whether records already exist
          if (WriteToDB | OnlyProcessTrackdaysNotInDB) {
            Query <- paste0("SELECT Count(gps.obspoints.uuid) AS \"Count\" FROM gps.obspoints ",
                            "INNER JOIN gps.tracks_days ON gps.obspoints.gpstracksdays_uuid = gps.tracks_days.uuid ",
                            "INNER JOIN gps.tracks ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                            "WHERE gps.tracks_days.date_local = '", DayDateString, "' AND ",
                            "gps.tracks.unit = '", Info_Track$unit[1], "'")

            DBRunRecordsCheck_NumObsPoints <- getRecordsFromQuery(Operation, Query, DBType=DBType)$Count
            rm(Query)
          } else {
            DBRunRecordsCheck_NumObsPoints <- 0
          }

          if (DBRunRecordsCheck_NumObsPoints>0 & (WriteToDB | OnlyProcessTrackdaysNotInDB)) {

            ## User feedback and Progress Output DF
            OutputString <- paste0("Day skipped because DB already contains ", DBRunRecordsCheck_NumObsPoints, " obs points from Unit ", Info_Track$unit[1]," on ", DayDateString, ".")
            print(OutputString)
            Output_Progress <- addToProgressDF(Comment=OutputString)
            rm(OutputString, DBRunRecordsCheck_NumObsPoints)

            Query <- paste0("SELECT Count(gps.runs.uuid) AS run_num FROM gps.tracks_days ",
                            "INNER JOIN gps.tracks ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                            "INNER JOIN gps.runs ON gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid ",
                            "WHERE gps.tracks_days.date_local = '", DayDateString, "' AND gps.tracks.unit = '", Info_Track$unit[1],"'")

            DBRunRecordsCheck_NumRuns <- getRecordsFromQuery(Operation, Query, DBType=DBType)$run_num
            rm(Query)


            if(DBRunRecordsCheck_NumRuns>0) {

              if (InclRunUseListFromPrevProcTrackdays) {

                print("")
                print("Run use list")
                print("************")

                print(getRunUsageList(Operation, DayDateString, Info_Track$unit[1], DBType=DBType))
                print("")

              } else {

                print("============================================================")
                print(paste0(DBRunRecordsCheck_NumRuns, " run(s) in the database for Unit ", Info_Track$unit[1]," on ", DayDateString, " already."))
                print("============================================================")

              }

            } else {

              print("==========================================================")
              print(paste0("No runs in the database for Unit ", Info_Track$unit[1]," on ", DayDateString, "."))
              print("==========================================================")

            }
            rm(DBRunRecordsCheck_NumRuns)

            ## Update counter
            Output_Feedback$NumTrackDaysSkipped <- Output_Feedback$NumTrackDaysSkipped + 1

          } else {

            ## Delete run check
            rm(DBRunRecordsCheck_NumObsPoints)

            ## Extracting GPS obs for this day
            DF_Day <- DF_Track[DF_Track$date_local==DateList[Index_Date],]

            ## Check whether any observations
            if (nrow(DF_Day)==0) {

              ## User feedback and Progress Output DF
              OutputString <- "No GPS observations available."
              print("==============================")
              print(OutputString)
              print("==============================")
              Output_Progress <- addToProgressDF(Comment=OutputString)
              rm(OutputString)

              ## Check whether any observations
            } else if (nrow(DF_Day)==1) {

              ## User feedback and Progress Output DF
              OutputString <- "Only a single GPS observation available."
              print(OutputString)
              Output_Progress <- addToProgressDF(Comment=OutputString)
              rm(OutputString)

            } else {

              ## User feedback and Progress Output DF
              OutputString <- "Original number of observations:"
              print(paste(OutputString, nrow(DF_Day)))
              Output_Progress <- addToProgressDF(Comment=OutputString, Num=nrow(DF_Day))
              rm(OutputString)

              ## UUID for Trackday
              UUIDTrackDay <- createUuid()

              OutputString <- paste0("Trackday UUID: ", UUIDTrackDay)
              print(OutputString)
              Output_Progress <- addToProgressDF(Comment=OutputString)
              rm(OutputString)

              ## Creating Info_TrackDays
              Info_TrackDays <- data.frame(uuid = UUIDTrackDay,
                                           gpstracks_uuid = UUIDTrack,
                                           date_local = DF_Day$date_local[1],
                                           date_local_esri = DF_Day$date_local[1],
                                           datetime_local_start = DF_Day$datetime_local[1],
                                           datetime_local_end = DF_Day$datetime_local[nrow(DF_Day)],
                                           guide = toupper(Guide),
                                           guides_uuid = NA,
                                           guide_type = DefaultGuideType,
                                           program = tolower(Program))
              
              ## Create dataframe DF_PotRuns_Skipped
              DF_PotRunsSkipped <- data.frame(uuid=character(0),
                                              gpstracksdays_uuid=character(0),
                                              run_numpot=numeric(0),
                                              reason=character(0))


######### vvvvvvvvvvvvvvvvvvvvvVVVVVVVVVVVVV
######### 5.4.2 INITIAL QUALITY CONTROL ====
######### ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

              ## Marking obs within operating area
              DF_Day$marker_inoparea <- isPointInSPDFPolygons(DF_Day$gps_longitude, DF_Day$gps_latitude, SPDF_OpArea)
              table(DF_Day$marker_inoparea)

              ## Marking obs in exclusion areas
              if(exists("SPDF_ExclAreas")) {
                DF_Day$marker_inexclarea <- isPointInSPDFPolygons(DF_Day$gps_longitude, DF_Day$gps_latitude, SPDF_ExclAreas)
              } else {
                DF_Day$marker_inexclarea <- 0
              }
              table(DF_Day$marker_inexclarea)

              ## Marking obs in inclusion areas
              if(exists("SPDF_InclAreas")) {
                DF_Day$marker_ininclarea <- isPointInSPDFPolygons(DF_Day$gps_longitude, DF_Day$gps_latitude, SPDF_InclAreas)
              } else {
                DF_Day$marker_ininclarea <- 0
              }
              table(DF_Day$marker_ininclarea)
              
              ## Check whether any obs in cat area
              if (Oper_Activities$value[Oper_Activities$name=="Cat"]=="Yes") {

                ## DF_Day$marker_incatarea <- 1
                DF_Day$marker_incatarea[(DF_Day$marker_ininclarea + DF_Day$marker_inoparea)>= 1 & DF_Day$marker_inexclarea!=1] <- 1
                Output_Progress <- addToProgressDF(Comment="All points assumed to be in cat area", Num=nrow(DF_Day[DF_Day$marker_incatarea==1,]))
                Flag_CatSkiing <- T

              } else if (Oper_Activities$value[Oper_Activities$name=="Cat"]=="Some") {

                DF_Day$marker_incatarea <- isPointInSPDFPolygons(DF_Day$gps_longitude, DF_Day$gps_latitude, SPDF_CatAreas)
                table(DF_Day$marker_incatarea)

                Output_Progress <- addToProgressDF(Comment="Number of observations in cat area", Num=nrow(DF_Day[DF_Day$marker_incatarea==1,]))
                Flag_CatSkiing <- ifelse(nrow(DF_Day[DF_Day$marker_incatarea==1,])>0, T, F)

              } else {

                DF_Day$marker_incatarea <- 0
                Flag_CatSkiing <- F

              }


######### vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
######### 5.4.3 EXTRACTION OF ELEVATION FROM DEM ====
######### ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

              if (UseElevRaster) {
                DF_Day$dem_altitude <- round(extract(ElevRaster, DF_Day[,c("gps_longitude", "gps_latitude")]))
              } else {
                DF_Day$dem_altitude <- NA
              }

######### vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
######### 5.4.4 CALCULATING OF DISTANCE, SPEED AND BEARING ====
######### ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

              ## Create additional columns
              ## *************************
              DF_Day$timediff_sec <- NA
              DF_Day$dist <- NA
              DF_Day$gps_altdiff <- NA
              DF_Day$dem_altdiff <- NA
              DF_Day$bearing <- NA
              DF_Day$uuid <- NA
              DF_Day$gpstracksdays_uuid <- UUIDTrackDay
              DF_Day$gpsruns_uuid <- NA
              DF_Day$gpsruns_pausepoints_uuid <- NA


              ## Loop for distance and bearing calculation
              ## *****************************************
              for (Index_Obs in 2:nrow(DF_Day)) {

                ## Add unique identifier to each observation point
                DF_Day$uuid[Index_Obs] <- createUuid()

                ## Time difference - to the second
                DF_Day$timediff_sec[Index_Obs] <- round(as.numeric((DF_Day$time_local[Index_Obs]-DF_Day$time_local[Index_Obs-1])*24*60*60),0)

                ## Distance in m
                Point1 <- c(DF_Day$gps_longitude[Index_Obs-1], DF_Day$gps_latitude[Index_Obs-1])
                Point2 <- c(DF_Day$gps_longitude[Index_Obs],   DF_Day$gps_latitude[Index_Obs])
                DF_Day$dist[Index_Obs] <- round(distHaversine(Point1, Point2),1)

                ## Altitude difference in m
                DF_Day$gps_altdiff[Index_Obs] <- round(DF_Day$gps_altitude[Index_Obs] - DF_Day$gps_altitude[Index_Obs-1],1)

                if (UseElevRaster) {
                  DF_Day$dem_altdiff[Index_Obs] <- round(DF_Day$dem_altitude[Index_Obs] - DF_Day$dem_altitude[Index_Obs-1],1)
                } else {
                  DF_Day$dem_altdiff[Index_Obs] <- NA
                }

                ## Bearing
                DF_Day$bearing[Index_Obs] <- round(bearing(Point1, Point2),1)

                ## System feedback
                if (Index_Obs %% 200 == 0) { print(paste("Calculating speeds/bearings for obs ", Index_Obs, "of", nrow(DF_Day), "..."))}
                if (Index_Obs ==nrow(DF_Day)) { print(paste("Calculating speeds/bearings for obs ", Index_Obs, "of", nrow(DF_Day)))}

                ## Check whether there are any negative time differences that might be indicative of duplicate records
                if (DF_Day$timediff_sec[Index_Obs]<0) {
                  stop(paste0("NEGATIVE TIME DIFFERENCE BETWEEN OBSERVATION POINTS: CHECK FILE FOR DUPLICATE OBSERVATION TIMES AROUND ", DF_Day$datetime_gmt[Index_Obs], "."))
                }

              }  ## End of for loop for distance and bearing calculation

              ## Clean up
              rm(Index_Obs)
              rm(Point1, Point2)

              ## Remove first row
              DF_Day <- DF_Day[-1,]
              

              ## Speed calculations
              ## ******************

              ## Horizontal speed in km/h
              DF_Day$hspeed_kmh <- round(DF_Day$dist/DF_Day$timediff_sec/1000*3600, 1)

              ## Vertical speed in m/s
              DF_Day$vspeed_ms <- round(DF_Day$gps_altdiff/DF_Day$timediff_sec, 1)

              ## Progress report
              Output_Progress <- addToProgressDF(Comment="Calculated speeds and bearing", Num=nrow(DF_Day))


######### vvvvvvvvvvvvvvVVvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
######### 5.4.5 CREATING MARKERS FOR RUN IDENTIFICATION ====
######### ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

              # if (nrow(subset(DF_Day, (marker_inoparea + marker_ininclarea >=1) & marker_inexclarea!=1))==0) {
              if (nrow(subset(DF_Day, (marker_inoparea + marker_ininclarea >=1) & marker_inexclarea!=1))<MinNumObsInsideOpAreaForProcessing) {


                ## User feedback and Progress Output DF
                if (nrow(subset(DF_Day, (marker_inoparea + marker_ininclarea >=1) & marker_inexclarea!=1))==0) { 
                  OutputString <- "No observations within op area outside of exclusion area."
                } else {
                  OutputString <- paste("Less than", MinNumObsInsideOpAreaForProcessing, "observations within op area outside of exclusion area.")
                }
                print("=========================================================")
                print(OutputString)
                print("=========================================================")
                Output_Progress <- addToProgressDF(Comment=OutputString)
                rm(OutputString)

                ## Write Info_TrackDays to DB or CSV (same code in other areas)
                writeInfotrackDaysToDBorCSV(Operation, Info_TrackDays, WriteToDB, CSVFileName=paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "Info_TrackDays.csv"), DBType=DBType)

                ## Turn DF_Day into SPDF and write to DB or CSV (same code in other areas)
                DF_Day <- deleteDFColumnsByName(DF_Day, "time_local")
                writeDFDayToDBorCSV(Operation, DF_Day, WriteToDB, CSVFileName=paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "SPDF_ObsPoints.csv"), DBType=DBType)

                ## Update Output_Feedback
                Output_Feedback$NumTrackDaysProcessed <- Output_Feedback$NumTrackDaysProcessed + 1


              } else {

############### 5.4.5.1 Flying ----
############### *******************
                
                if (Oper_Activities$value[Oper_Activities$name=="Heli"]=="Yes") {

                  ## User feedback
                  print(paste0("Creating Markers - Flying sections (", Sys.time(), ")"))

                  # Compare with threshold (only with oparea and outside of exclarea)
                  DF_Day$marker_flying <- ifelse(DF_Day$hspeed_kmh>=Flight_MinSpeed, 1, 0)

                  ## Mark obs outside of oparea and within exclarea as NA
                  DF_Day$marker_flying[(DF_Day$marker_inoparea!=1 & DF_Day$marker_ininclarea!=1) | DF_Day$marker_inexclarea==1] <- NA

                  ## Summarize periods
                  PeriodSummary_Flying1 <- summarizePeriodsInTimeSeries(DF_Day, "marker_flying")

                  ## First progress report
                  Output_Progress <- addToProgressDF(Comment=paste0("Flying marker: ", nrow(PeriodSummary_Flying1[PeriodSummary_Flying1$Value==1,]), " periods with speeds above min flying speed (>=", Flight_MinSpeed, "km/h)"), Num=nrow(DF_Day[DF_Day$marker_flying==1,]))
                  table(DF_Day$marker_flying)

                  if (nrow(PeriodSummary_Flying1) > 1) {

                    ## Eliminate flight and non-flight periods that are too short
                    DF_Day <- eliminateShortPeriods(DF_Day, "marker_flying", PeriodSummary_Flying1, Flight_MinDuration)
                    PeriodSummary_Flying2 <- summarizePeriodsInTimeSeries(DF_Day, "marker_flying")

                    ## Second progress report
                    Output_Progress <- addToProgressDF(Comment=paste0("Flying marker: Eliminate short flying/non-flying periods (<", Flight_MinDuration, "sec)"))
                    Output_Progress <- addToProgressDF(Comment=paste0("Flying marker: ", nrow(PeriodSummary_Flying2[PeriodSummary_Flying2$Value==1,]), " periods with speeds above min flying speed (>=", Flight_MinSpeed, "km/h)"), Num=nrow(DF_Day[DF_Day$marker_flying==1,]))
                    table(DF_Day$marker_flying)

                    if (nrow(PeriodSummary_Flying2) > 1) {

                      ## Examining max elevation for each flight to avoid marking of fast downhill sections as flights
                      PeriodSummary_Flying2$AltStart <- NA
                      PeriodSummary_Flying2$AltMax <- NA

                      for (Index_Period in 1:(nrow(PeriodSummary_Flying2)-1)) {
                        if (PeriodSummary_Flying2$Value[Index_Period]==1) {
                          PeriodSummary_Flying2$AltStart[Index_Period] <- DF_Day$gps_altitude[DF_Day$datetime_local==PeriodSummary_Flying2$StartDateTimeLocal[Index_Period]]
                          PeriodSummary_Flying2$AltMax[Index_Period] <- max(DF_Day$gps_altitude[DF_Day$datetime_local>=PeriodSummary_Flying2$StartDateTimeLocal[Index_Period] & DF_Day$datetime_local<=PeriodSummary_Flying2$EndDateTimeLocal[Index_Period]])
                        }
                      }
                      rm(Index_Period)

                      ## Calculate altitude gain
                      PeriodSummary_Flying2$AltMaxGain <- PeriodSummary_Flying2$AltMax - PeriodSummary_Flying2$AltStart

                      ## Unmark last flight as it is often straigth into the vallue
                      StartTime_LastFlight <- max(PeriodSummary_Flying2$StartDateTimeLocal[PeriodSummary_Flying2$Value==1])
                      PeriodSummary_Flying2$AltMaxGain[PeriodSummary_Flying2$StartDateTimeLocal==StartTime_LastFlight] <- NA

                      ## Eliminate flight sections that did not have a sufficient altitude gain and are shorted than min flight duration
                      PeriodSummary_Flying2a <- subset(PeriodSummary_Flying2, AltMaxGain<Flight_MinAltGain & Duration<Flight_MinDuration)
                      if (nrow(PeriodSummary_Flying2a)>0) {
                        for (Index_Period in 1:nrow(PeriodSummary_Flying2a)) {
                          DF_Day$marker_flying[DF_Day$datetime_local>=PeriodSummary_Flying2a$StartDateTimeLocal[Index_Period] & DF_Day$datetime_local<=PeriodSummary_Flying2a$EndDateTimeLocal[Index_Period]] <- 0
                        }

                        PeriodSummary_Flying3 <- summarizePeriodsInTimeSeries(DF_Day, "marker_flying")

                        ## Third progress report
                        Output_Progress <- addToProgressDF(Comment=paste0("Flying marker: Eliminated flights with <", Flight_MinAltGain, "m elevation gain."))
                        Output_Progress <- addToProgressDF(Comment=paste0("Flying marker: ", nrow(PeriodSummary_Flying3[PeriodSummary_Flying3$Value==1,]), " periods with speeds above min flying speed (>=", Flight_MinSpeed, "km/h)"), Num=nrow(DF_Day[DF_Day$marker_flying==1,]))
                        table(DF_Day$marker_flying)

                      }

                    }  ## End of if condition: "nrow(PeriodSummary_Flying2) > 1"

                  }  ## End of if condition: "nrow(PeriodSummary_Flying1) > 1"

                  ## Clean-up
                  rm(PeriodSummary_Flying1)
                  if (exists("PeriodSummary_Flying2")) {rm(PeriodSummary_Flying2)}
                  if (exists("PeriodSummary_Flying2a")) {rm(PeriodSummary_Flying2a)}
                  if (exists("PeriodSummary_Flying3")) {rm(PeriodSummary_Flying3)}

                } else {
                  
                  ## Default if not heliskiing
                  DF_Day$marker_flying <- 0
                  
                }

############### 5.4.5.2 Up and down ----
############### ************************

                ## User feedback
                print(paste0("Creating Markers - Downhill sections (", Sys.time(), ")"))

                ## Identify up and down sections
                if (UseElevRaster) {
                  DF_Day$marker_downhill <- as.numeric(assignUpAndDownToPoints(DF_Day$dem_altitude, UpDown_AltDiffThreshold, UpValue=0, DownValue=1))
                } else {
                  DF_Day$marker_downhill <- as.numeric(assignUpAndDownToPoints(DF_Day$gps_altitude, UpDown_AltDiffThreshold, UpValue=0, DownValue=1))
                }

                ## Mark obs outside of oparea and within exclarea as NA
                DF_Day$marker_downhill[(DF_Day$marker_inoparea!=1 & DF_Day$marker_ininclarea!=1) | DF_Day$marker_inexclarea==1] <- NA

                ## Summarize periods
                PeriodSummary_UpDown1 <- summarizePeriodsInTimeSeries(DF_Day, "marker_downhill")

                ## First progress report
                Output_Progress <- addToProgressDF(Comment=paste0("Up/down marker: ", nrow(PeriodSummary_UpDown1[PeriodSummary_UpDown1$Value==1,]), "  downhill segments"), Num=nrow(DF_Day[DF_Day$marker_downhill==1,]))
                table(DF_Day$marker_downhill)

                ## Eliminate up- and downhill periods that are too short
                DF_Day <- eliminateShortPeriods(DF_Day, "marker_downhill", PeriodSummary_UpDown1, UpDown_MinDuration)
                PeriodSummary_UpDown2 <- summarizePeriodsInTimeSeries(DF_Day, "marker_downhill")

                ## Second progress report
                Output_Progress <- addToProgressDF(Comment=paste0("Up/down marker: Eliminating short downhill segments (<", UpDown_MinDuration, "sec)"))
                Output_Progress <- addToProgressDF(Comment=paste0("Up/down marker: ", nrow(PeriodSummary_UpDown2[PeriodSummary_UpDown2$Value==1,]), "  downhill segments"), Num=nrow(DF_Day[DF_Day$marker_downhill==1,]))
                table(DF_Day$marker_downhill)

                ## Clean-up
                rm(PeriodSummary_UpDown1, PeriodSummary_UpDown2)

############### 5.4.5.3 Identify short stationary periods: pauses ----
############### ******************************************************

                ## User feedback
                print(paste0("Creating Markers - Shorter pause periods (>", Pause_MinTime, "sec) (", Sys.time(), ")"))

                ## Indentifying pauses
                PeriodIdentify_Pause <- identifyStationaryPeriod(DF_Day, Pause_DistRadius, Pause_MinTime)

                ## Marking of observations
                DF_Day$marker_pause <- 0
                for (Index_Pause in 1:nrow(PeriodIdentify_Pause)) {
                  DF_Day$marker_pause[(DF_Day$datetime_local>=PeriodIdentify_Pause$DateTimeLocalStart[Index_Pause]) & (DF_Day$datetime_local<=PeriodIdentify_Pause$DateTimeLocalEnd[Index_Pause])] <- 1
                }
                rm(Index_Pause)

                ## First progress report
                Output_Progress <- addToProgressDF(Comment=paste0("Pause marker: ", nrow(PeriodIdentify_Pause), " pauses identified (within ", Pause_DistRadius, "m for >", Pause_MinTime, "sec)"), Num=sum(PeriodIdentify_Pause$NumObs))
                table(DF_Day$marker_pause)

                ## Summarize pause periods
                PeriodSummary_Pause1 <- summarizePeriodsInTimeSeries(DF_Day, "marker_pause")

                ## Eliminate moving periods between pauses that are too short
                DF_Day <- eliminateShortPeriods(DF_Day, "marker_pause", PeriodSummary_Pause1, Pause_MoveMinTime)
                PeriodSummary_Pause2 <- summarizePeriodsInTimeSeries(DF_Day, "marker_pause")

                ## Mark obs outside of oparea and within exclarea as NA
                DF_Day$marker_pause[(DF_Day$marker_inoparea!=1 & DF_Day$marker_ininclarea!=1) | DF_Day$marker_inexclarea==1] <- NA

                ## Second progress report
                Output_Progress <- addToProgressDF(Comment=paste0("Pause marker: ", nrow(PeriodSummary_Pause2[PeriodSummary_Pause2$Value==1,]), " pauses after short moves in between (<=", Pause_MoveMinTime, "sec) have been eliminated."), Num=nrow(DF_Day[DF_Day$marker_pause==1,]))
                table(DF_Day$marker_pause)

                ## Clean-up
                rm(PeriodIdentify_Pause, PeriodSummary_Pause1, PeriodSummary_Pause2)

############### 5.4.5.4 Identify longer stationary periods: breaks ----
############### *******************************************************

                ## User feedback
                print(paste0("Creating Markers - Longer break periods (>", Break_MinTime, "sec) (", Sys.time(), ")"))

                ## Identifying breaks
                PeriodIdentify_Break <- identifyStationaryPeriod(DF_Day, Break_DistRadius, Break_MinTime)

                ## Marking of observations
                DF_Day$marker_break <- 0
                for (Index_Break in 1:nrow(PeriodIdentify_Break)) {
                  DF_Day$marker_break[(DF_Day$datetime_local>=PeriodIdentify_Break$DateTimeLocalStart[Index_Break]) & (DF_Day$datetime_local<=PeriodIdentify_Break$DateTimeLocalEnd[Index_Break])] <- 1
                }
                rm(Index_Break)

                ## First progress report
                Output_Progress <- addToProgressDF(Comment=paste0("Break marker: ", nrow(PeriodIdentify_Break), " long breaks identified (within ", Break_DistRadius, "m for more than ", Break_MinTime, "sec)"), Num=sum(PeriodIdentify_Break$NumObs))
                table(DF_Day$marker_break)

                ## Summarize break periods
                PeriodSummary_Break1 <- summarizePeriodsInTimeSeries(DF_Day, "marker_break")

                ## Eliminate moving periods between pauses that are too short
                DF_Day <- eliminateShortPeriods(DF_Day, "marker_break", PeriodSummary_Break1, Break_MoveMinTime)
                PeriodSummary_Break2 <- summarizePeriodsInTimeSeries(DF_Day, "marker_break")

                ## Mark obs outside of oparea and within exclarea as NA
                DF_Day$marker_break[(DF_Day$marker_inoparea!=1 & DF_Day$marker_ininclarea!=1) | DF_Day$marker_inexclarea==1] <- NA

                ## Second progress report
                Output_Progress <- addToProgressDF(Comment=paste0("Break marker: ", nrow(PeriodSummary_Break2[PeriodSummary_Break2$Value==1,]), " longer breaks after short moves in between (<=", Break_MoveMinTime, "sec) have been eliminated."), Num=nrow(DF_Day[DF_Day$marker_break==1,]))
                table(DF_Day$marker_break)

                ## Clean-up
                rm(PeriodIdentify_Break, PeriodSummary_Break1, PeriodSummary_Break2)
                
############### 5.4.5.5 Highlight points with exceptionally large distance to previous obs ----
############### *******************************************************************************

                # User feedback
                print(paste0("Creating Markers - Obs with expectional large distance (", Sys.time(), ")"))

                ## Highlight obs
                # DF_Day$marker_lrgdist <- ifelse(DF_Day$dist >= NewRun_DistToPrevObsWithAltDiffReq, 1, 0)
                DF_Day$marker_lrgdist <- ifelse(DF_Day$dist >= NewRun_DistToPrevObsWithInclReq_Dist, 1, 0)
                DF_Day$marker_lrgdistnoaltdiffreq <- ifelse(DF_Day$dist >= NewRun_DistToPrevObsWithoutInclReq, 1, 0)

                ## Mark obs outside of oparea and within exclarea as NA
                DF_Day$marker_lrgdist[(DF_Day$marker_inoparea!=1 & DF_Day$marker_ininclarea!=1) | DF_Day$marker_inexclarea==1] <- NA
                DF_Day$marker_lrgdistnoaltdiffreq[(DF_Day$marker_inoparea!=1 & DF_Day$marker_ininclarea!=1) | DF_Day$marker_inexclarea==1] <- NA

                ## Progress report
                # Output_Progress <- addToProgressDF(Comment=paste0("Lrg dist marker: Obs with lrg distance to previous obs (>=", NewRun_DistToPrevObsWithAltDiffReq, "m)"), Num=nrow(DF_Day[DF_Day$marker_lrgdist==1,]))
                Output_Progress <- addToProgressDF(Comment=paste0("Lrg dist marker: Obs with lrg distance to previous obs (>=", NewRun_DistToPrevObsWithInclReq_Dist, "m)"), Num=nrow(DF_Day[DF_Day$marker_lrgdist==1,]))
                table(DF_Day$marker_lrgdist)
                
                Output_Progress <- addToProgressDF(Comment=paste0("Lrg dist marker: Obs with lrg distance to previous obs (>=", NewRun_DistToPrevObsWithoutInclReq, "m)"), Num=nrow(DF_Day[DF_Day$marker_lrgdistnoaltdiffreq==1,]))
                table(DF_Day$marker_lrgdistnoaltdiffreq)
                
############### 5.4.5.6 Highlight points with exceptionally large elevation gain to previous obs ----
############### *************************************************************************************

                if (UseElevRaster) {

                  # User feedback
                  print(paste0("Creating Markers - Obs with expectional large elevation gain (DEM) (", Sys.time(), ")"))

                  ## Highlight obs
                  DF_Day$marker_lrgaltdiff <- ifelse(DF_Day$dem_altdiff >= NewRun_AltDiffToPrevObs, 1, 0)

                  ## Mark obs outside of oparea and within exclarea as NA
                  DF_Day$marker_lrgaltdiff[(DF_Day$marker_inoparea!=1 & DF_Day$marker_ininclarea!=1) | DF_Day$marker_inexclarea==1] <- NA

                  ## Progress report
                  Output_Progress <- addToProgressDF(Comment=paste0("Lrg alt diff marker (DEM): Obs with lrg altitude difference to previous obs (>=", NewRun_AltDiffToPrevObs, "m)"), Num=nrow(DF_Day[DF_Day$marker_lrgaltdiff==1,]))
                  table(DF_Day$marker_lrgaltdiff)

                } else {

                  # User feedback
                  print(paste0("Creating Markers - Obs with expectional large elevation gain (GPS) (", Sys.time(), ")"))

                  ## Highlight obs
                  DF_Day$marker_lrgaltdiff <- ifelse(DF_Day$gps_altdiff >= NewRun_AltDiffToPrevObs, 1, 0)

                  ## Mark obs outside of oparea and within exclarea as NA
                  DF_Day$marker_lrgaltdiff[(DF_Day$marker_inoparea!=1 & DF_Day$marker_ininclarea!=1) | DF_Day$marker_inexclarea==1] <- NA

                  ## Progress report
                  Output_Progress <- addToProgressDF(Comment=paste0("Lrg alt diff marker (GPS): Obs with lrg altitude difference to previous obs (>=", NewRun_AltDiffToPrevObs, "m)"), Num=nrow(DF_Day[DF_Day$marker_lrgaltdiff==1,]))
                  table(DF_Day$marker_lrgaltdiff)

                }
                
############### 5.4.5.7 On Cat Road ----
############### *************************

                ## Create columns
                DF_Day$marker_oncatroad <- 0
                DF_Day$marker_oncatroad[(DF_Day$marker_inoparea!=1 & DF_Day$marker_ininclarea!=1)|DF_Day$marker_inexclarea==1] <- NA
                DF_Day$marker_oncatroad_uuid <- NA

                if (Flag_CatSkiing) {

                  ## User feedback
                  print(paste0("Creating Markers - Obs on relevant cat roads (", Sys.time(), ")"))

                  ## Checking whether points fall within buffered cat roads for each road
                  for (Index_CatRoads in 1:nrow(SPDF_CatRoads_ForRunIdentificationBuffer)) {
                    
                    DF_Day$marker_oncatroad <- isPointInSPDFPolygons(DF_Day$gps_longitude, DF_Day$gps_latitude, SPDF_CatRoads_ForRunIdentificationBuffer[Index_CatRoads,])
                    DF_Day$marker_oncatroad_uuid[DF_Day$marker_oncatroad==1] <- SPDF_CatRoads_ForRunIdentificationBuffer@data$uuid[Index_CatRoads]

                  }
                  
                  ## Finalizing marker_oncatroad
                  DF_Day$marker_oncatroad[!is.na(DF_Day$marker_oncatroad_uuid)] <- 1
                  DF_Day$marker_oncatroad[(DF_Day$marker_inoparea!=1 & DF_Day$marker_ininclarea!=1)|DF_Day$marker_inexclarea==1] <- NA
                  DF_Day$marker_oncatroad_uuid[(DF_Day$marker_inoparea!=1 & DF_Day$marker_ininclarea!=1)|DF_Day$marker_inexclarea==1] <- NA
                  
                  ## Summarize periods
                  PeriodSummary_CatRoads1 <- summarizePeriodsInTimeSeries(DF_Day, "marker_oncatroad")
                  
                  #print("************ WRITING CAT ROAD SUMMARY *********************")
                  #write.csv(PeriodSummary_CatRoads1, "C:\\Temp\\PeriodSummary_CatRoads1.csv")

                  ## First progress report
                  OutputString <- paste0("Cat road marker: Points within ", CatRoad_MaxDist, "m of relevant cat roads:")
                  Output_Progress <- addToProgressDF(Comment=OutputString, Num=nrow(DF_Day[DF_Day$marker_oncatroad==1,]))
                  rm(OutputString)

                  ## Eliminate short distances that are too short
                  DF_Day <- eliminateShortDistances(DF_Day, "marker_oncatroad", PeriodSummary_CatRoads1, CatRoad_MinDist)
                  
                  PeriodSummary_CatRoads2 <- summarizePeriodsInTimeSeries(DF_Day, "marker_oncatroad")

                  #print("************ WRITING CAT ROAD SUMMARY *********************")
                  #write.csv(PeriodSummary_CatRoads2, "C:\\Temp\\PeriodSummary_CatRoads2.csv")
                  
                  
                  
                  ## First progress report
                  OutputString <- paste0("Cat road marker: Points within ", CatRoad_MaxDist, "m of relevant cat roads after periods <", CatRoad_MinDuration, " are eliminated:")
                  Output_Progress <- addToProgressDF(Comment=OutputString, Num=nrow(DF_Day[DF_Day$marker_oncatroad==1,]))
                  rm(OutputString)

                  ## Clean-up
                  ## rm(CatRoadDistMatrix, SPDF_CatGPSPoints)
                  rm(PeriodSummary_CatRoads1, PeriodSummary_CatRoads2)

                }
                
############### 5.4.5.8 Overview Plot ----
############### **************************

                if (UseElevRaster){
                  PlotAltCol <- "dem_altitude"
                } else {
                  PlotAltCol <- "gps_altitude"
                }

                if (WriteChartsToFile) {png(paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "Overview_01_Profile.png"), width=Chart_Width, height=Chart_Height)}

                plot(DF_Day$time_local, DF_Day[,PlotAltCol], type="l", xlab="Local time", ylab="Altitude", main=paste0(Chart_Title, " (", PlotAltCol, ") - Markings"))
                points(DF_Day$time_local[DF_Day$marker_downhill==1],   DF_Day[DF_Day$marker_downhill==1,PlotAltCol], col=Col_Skiing, pch=20)
                points(DF_Day$time_local[DF_Day$marker_flying==1],     DF_Day[DF_Day$marker_flying==1,PlotAltCol], col=Col_Flying, pch=20)
                points(DF_Day$time_local[DF_Day$marker_oncatroad==1],  DF_Day[DF_Day$marker_oncatroad==1,PlotAltCol], col=Col_CatRiding, pch=20)
                points(DF_Day$time_local[DF_Day$marker_pause==1],      DF_Day[DF_Day$marker_pause==1,PlotAltCol], col=Col_Pause, pch="*")
                points(DF_Day$time_local[DF_Day$marker_break==1],      DF_Day[DF_Day$marker_break==1,PlotAltCol], col=Col_Break, pch="*")
                points(DF_Day$time_local[DF_Day$marker_lrgdist==1],    DF_Day[DF_Day$marker_lrgdist==1,PlotAltCol], col="black", pch=16)
                points(DF_Day$time_local[DF_Day$marker_lrgdistnoaltdiffreq==1],    DF_Day[DF_Day$marker_lrgdistnoaltdiffreq==1,PlotAltCol], col="black", pch=16)
                points(DF_Day$time_local[DF_Day$marker_lrgaltdiff==1],    DF_Day[DF_Day$marker_lrgaltdiff==1,PlotAltCol], col="brown", pch=16)
                # points(DF_Day$time_local[DF_Day$marker_lrgaltdiff==1], DF_Day[DF_Day$marker_lrgaltdiff==1,PlotAltCol], col="blue", pch=20)

                points(DF_Day$time_local[(DF_Day$marker_inoparea!=1 & DF_Day$marker_ininclarea!=1)], DF_Day[(DF_Day$marker_inoparea!=1 & DF_Day$marker_ininclarea!=1),PlotAltCol], col=Col_Outside, pch=20)
                points(DF_Day$time_local[DF_Day$marker_inexclarea==1], DF_Day[DF_Day$marker_inexclarea==1,PlotAltCol], col=Col_Outside, pch=20)

                if (UseElevRaster){
                  lines(DF_Day$time_local, DF_Day$gps_altitude, lty=2)
                } else {
                  lines(DF_Day$time_local, DF_Day$dem_altitude, lty=2)
                }

                if (WriteChartsToFile) {dev.off()}

                rm(PlotAltCol)
                
############### 5.4.5.9 Create activity column ----
############### ***********************************
                DF_Day$activity <- NA
                
                ## Basic marking
                if(Oper_Activities$value[Oper_Activities$name=="Heli"]=="Yes") {
                  DF_Day$activity[(DF_Day$marker_inoparea==1 | DF_Day$marker_ininclarea!=1) & DF_Day$marker_inexclarea!=1 & DF_Day$marker_flying==1] <- "Flying"
                }

######### vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
######### 5.4.6 IDENTIFYING POTENTIAL RUNS ====
######### ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

                ## Identify potential start and end points
                ## ***************************************

                DF_Day$runmarker_startend <- NA
                DF_Day$runmarker_why <- NA

                if (UseElevRaster){
                  AltDiffCol <- "dem_altdiff"
                } else {
                  AltDiffCol <- "dem_altdiff"
                }

                for (Index_Obs in 2:nrow(DF_Day)) {

                  ## Both points outside of op area/inclusion area or inside exclusion area
                  if (((DF_Day$marker_inoparea[Index_Obs]+DF_Day$marker_ininclarea[Index_Obs])==0 | DF_Day$marker_inexclarea[Index_Obs]==1) & ((DF_Day$marker_inoparea[Index_Obs-1]+DF_Day$marker_ininclarea[Index_Obs-1])==0 | DF_Day$marker_inexclarea[Index_Obs-1]==1)) {

                    ## Do nothing if both rows are outside of operation area or in exclusion area

                    ## Next point outside of op area or inside exclusion area
                  } else if (((DF_Day$marker_inoparea[Index_Obs]+DF_Day$marker_ininclarea[Index_Obs])==0 | DF_Day$marker_inexclarea[Index_Obs]==1) & ((DF_Day$marker_inoparea[Index_Obs-1]+DF_Day$marker_ininclarea[Index_Obs-1])>=1 & DF_Day$marker_inexclarea[Index_Obs-1]==0)) {

                    ## Exiting operating area or entering exclusion area when not flying
                    if (DF_Day$marker_flying[Index_Obs-1]==0 &  ((DF_Day$marker_inoparea[Index_Obs]+DF_Day$marker_ininclarea[Index_Obs])==0 | DF_Day$marker_inexclarea[Index_Obs]==1)) {
                      DF_Day$runmarker_startend[Index_Obs-1] <- "End"
                      DF_Day$runmarker_why[Index_Obs-1] <- "Next obs outside"
                    }

                    ## Both points inside op area and no in exclusion area
                  } else if (((DF_Day$marker_inoparea[Index_Obs]+DF_Day$marker_ininclarea[Index_Obs])>=1 & DF_Day$marker_inexclarea[Index_Obs]==0) & ((DF_Day$marker_inoparea[Index_Obs-1]+DF_Day$marker_ininclarea[Index_Obs-1])>=1 & DF_Day$marker_inexclarea[Index_Obs-1]==0)) {

                    ## Start - Heli: Switch from flying to not flying
                    if (Oper_Activities$value[Oper_Activities$name=="Heli"]=="Yes" & DF_Day$marker_flying[Index_Obs-1]==1 & DF_Day$marker_flying[Index_Obs]==0) {
                      DF_Day$runmarker_startend[Index_Obs] <- "Start"
                      DF_Day$runmarker_why[Index_Obs] <- "Flying"

                      ## End - Heli: Switch from not flying to flying
                    } else if (Oper_Activities$value[Oper_Activities$name=="Heli"]=="Yes" & DF_Day$marker_flying[Index_Obs-1]==0 & DF_Day$marker_flying[Index_Obs]==1) {
                      DF_Day$runmarker_startend[Index_Obs-1] <- "End"
                      DF_Day$runmarker_why[Index_Obs-1] <- "Flying"

                      ## Large distance with positive altitude gain when not flying
                    } else if (DF_Day$marker_flying[Index_Obs]==0 &  DF_Day$marker_lrgdist[Index_Obs]==1 & ((DF_Day[Index_Obs, AltDiffCol]/DF_Day$dist[Index_Obs]) > -tan(NewRun_DistToPrevObsWithInclReq_Incl*pi/180))) {
                      DF_Day$runmarker_startend[Index_Obs-1] <- "End"
                      DF_Day$runmarker_startend[Index_Obs] <- "Start"
                      DF_Day$runmarker_why[Index_Obs-1] <- "Lrg dist"
                      DF_Day$runmarker_why[Index_Obs] <- "Lrg dist"
                      
                      ## Exceptially large distance without altitude gain requirement
                    } else if (DF_Day$marker_lrgdistnoaltdiffreq[Index_Obs]==1) {
                      DF_Day$runmarker_startend[Index_Obs-1] <- "End"
                      DF_Day$runmarker_startend[Index_Obs] <- "Start"
                      DF_Day$runmarker_why[Index_Obs-1] <- "Lrg dist"
                      DF_Day$runmarker_why[Index_Obs] <- "Lrg dist"
                      
                      ## Large altitude gain when not flying
                    } else if (DF_Day$marker_flying[Index_Obs]==0 &  DF_Day$marker_lrgaltdiff[Index_Obs]==1) {
                      DF_Day$runmarker_startend[Index_Obs-1] <- "End"
                      DF_Day$runmarker_startend[Index_Obs] <- "Start"
                      DF_Day$runmarker_why[Index_Obs-1] <- "Lrg dist"
                      DF_Day$runmarker_why[Index_Obs] <- "Lrg dist"
                      
                      ## Start - in cat area: Switch from uphill to downhill
                    } else if (DF_Day$marker_incatarea[Index_Obs]==1 & DF_Day$marker_flying[Index_Obs-1]==0 & DF_Day$marker_flying[Index_Obs]==0 & DF_Day$marker_downhill[Index_Obs-1]==0 & DF_Day$marker_downhill[Index_Obs]==1) {
                      DF_Day$runmarker_startend[Index_Obs] <- "Start"
                      DF_Day$runmarker_why[Index_Obs] <- "Uphill to downhill"

                      ## End - in cat area: Switch from downhill to uphill
                    } else if (DF_Day$marker_incatarea[Index_Obs]==1 & DF_Day$marker_flying[Index_Obs-1]==0 & DF_Day$marker_flying[Index_Obs]==0 & DF_Day$marker_downhill[Index_Obs-1]==1 & DF_Day$marker_downhill[Index_Obs]==0) {
                      DF_Day$runmarker_startend[Index_Obs] <- "End"
                      DF_Day$runmarker_why[Index_Obs] <- "Downhill to uphill"

                      ## End - in cat area: onto cat road
                    } else if (DF_Day$marker_oncatroad[Index_Obs-1]==0 & DF_Day$marker_oncatroad[Index_Obs]==1) {
                      DF_Day$runmarker_startend[Index_Obs] <- "End"
                      DF_Day$runmarker_why[Index_Obs] <- "Onto relevant catroad"

                    }

                  }

                } ## End of for loop for indentification of potential runs
                rm(Index_Obs)


                ## Check whether any run start or end points have been identified
                ## **************************************************************

                DF_StartEndSummary <- DF_Day[!is.na(DF_Day$runmarker_startend),]
                
                if((nrow(DF_StartEndSummary)==0) | (nrow(DF_StartEndSummary[DF_StartEndSummary$runmarker_startend=="Start",])==0)) {

                  ## User feedback and Progress Output DF
                  if (nrow(DF_StartEndSummary)==0) {
                    OutputString <- "No run start or end points identified."
                  } else if (nrow(DF_StartEndSummary[DF_StartEndSummary$runmarker_startend=="Start",])==0){
                    OutputString <- "No run start points identified."
                  }
                  print("======================================")
                  print(OutputString)
                  print("======================================")
                  Output_Progress <- addToProgressDF(Comment=OutputString)
                  rm(OutputString)

                  rm(DF_StartEndSummary)

                  ## Write Info_TrackDays to DB or CSV (same code in other areas)
                  writeInfotrackDaysToDBorCSV(Operation, Info_TrackDays, WriteToDB, CSVFileName=paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "Info_TrackDays.csv"), DBType=DBType)

                  ## Turn DF_Day into SPDF and write to DB or CSV (same code in other areas)
                  DF_Day <- deleteDFColumnsByName(DF_Day, "time_local")
                  writeDFDayToDBorCSV(Operation, DF_Day, WriteToDB, CSVFileName=paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "SPDF_ObsPoints.csv"), DBType=DBType)

                  ## Update Output_Feedback
                  Output_Feedback$NumTrackDaysProcessed <- Output_Feedback$NumTrackDaysProcessed + 1


                } else {

                  ## Ensure that first marker is actually a starting point
                  ## *****************************************************

                  FirstStartIndex <- min(which(DF_StartEndSummary$runmarker_startend=="Start"))
                  if (FirstStartIndex != 1) {

                    for (Index_StartEnd in 1:(FirstStartIndex-1)) {
                      DF_Day$runmarker_startend[DF_Day$uuid==DF_StartEndSummary$uuid[Index_StartEnd]] <- "End (deleted)"
                    }
                    rm(Index_StartEnd)

                    DF_StartEndSummary <- DF_StartEndSummary[c(FirstStartIndex:nrow(DF_StartEndSummary)),]

                    OutputString <- paste0("Removed ", FirstStartIndex-1, " potential end points at beginning of day")
                    print(OutputString)
                    Output_Progress <- addToProgressDF(Comment=OutputString)
                    rm(OutputString)

                  }
                  rm(FirstStartIndex)


                  ## If no end point for the final run has been identified --> Last break within op area or last obs
                  ## ***********************************************************************************************

                  ## Last break
                  if (tail(DF_StartEndSummary$runmarker_startend,1)=="Start") {
                    
                    LastStart_DateTime <- DF_Day$datetime_local[max(which(DF_Day$runmarker_startend=="Start"))]
                    options(warn=-1)
                    Row_LastBreak <- max(which(DF_Day$datetime_local>LastStart_DateTime & DF_Day$marker_break==1 & (DF_Day$marker_inoparea+DF_Day$marker_ininclarea)==1 & DF_Day$marker_inexclarea!=1))
                    options(warn=0)
                    
                    if(is.finite(Row_LastBreak)) {
                      DF_Day$runmarker_startend[Row_LastBreak] <- "End"
                      DF_Day$runmarker_why[Row_LastBreak] <- "Last break"
                      rm(Row_LastBreak)
                      DF_StartEndSummary <- DF_Day[!is.na(DF_Day$runmarker_startend),]
                    }
                  }

                  ## Last obs is still no final end
                  if (tail(DF_StartEndSummary$runmarker_startend,1)=="Start") {
                    DF_Day$runmarker_startend[nrow(DF_Day)] <- "End"
                    DF_Day$runmarker_why[nrow(DF_Day)] <- "Last obs"
                    DF_StartEndSummary <- DF_Day[!is.na(DF_Day$runmarker_startend),]
                  }
                  
                  ## Going through start and end points to ensure meaningful sequence
                  ## ****************************************************************

                  if (nrow(DF_StartEndSummary)>2) {

                    for (Index_StartEnd in 2:(nrow(DF_StartEndSummary)-1)) {

                      StartEnd1 <- DF_StartEndSummary$runmarker_startend[Index_StartEnd-1]
                      StartEnd2 <- DF_StartEndSummary$runmarker_startend[Index_StartEnd]
                      StartEnd3 <- DF_StartEndSummary$runmarker_startend[Index_StartEnd+1]
                      Reason1 <- DF_StartEndSummary$runmarker_why[Index_StartEnd-1]
                      Reason2 <- DF_StartEndSummary$runmarker_why[Index_StartEnd]
                      Reason3 <- DF_StartEndSummary$runmarker_why[Index_StartEnd+1]

                      ## Two starts in a row
                      if (StartEnd1=="Start" & StartEnd2=="Start") {

                        DF_Day$runmarker_startend[DF_Day$uuid==DF_StartEndSummary$uuid[Index_StartEnd-1]] <- "Start (deleted)"
                        DF_StartEndSummary$runmarker_startend[Index_StartEnd-1] <- "Start (deleted)"
                        print(paste0("Deleted unnecessary start point (", tolower(Reason1), ")"))

                        ## Three ends in a row
                      } else if (StartEnd1=="End" & StartEnd2=="End" & StartEnd3=="End") {

                        DF_Day$runmarker_startend[DF_Day$uuid==DF_StartEndSummary$uuid[Index_StartEnd]] <- "End (deleted)"
                        DF_StartEndSummary$runmarker_startend[Index_StartEnd] <- "End (deleted)"
                        DF_Day$runmarker_startend[DF_Day$uuid==DF_StartEndSummary$uuid[Index_StartEnd+1]] <- "End (deleted)"
                        DF_StartEndSummary$runmarker_startend[Index_StartEnd+1] <- "End (deleted)"
                        print(paste0("Deleted unnecessary end points (", tolower(Reason2), ", ", tolower(Reason3),")"))

                        ## Two ends in a row
                      } else if (StartEnd1=="End" & StartEnd2=="End") {

                        DF_Day$runmarker_startend[DF_Day$uuid==DF_StartEndSummary$uuid[Index_StartEnd]] <- "End (deleted)"
                        DF_StartEndSummary$runmarker_startend[Index_StartEnd] <- "End (deleted)"
                        print(paste0("Deleted unnecessary end point (", tolower(Reason2), ")"))

                      }

                    }

                    rm(Index_StartEnd)
                    rm(StartEnd1, StartEnd2, StartEnd3, Reason1, Reason2, Reason3)

                  }  ## End of condition "(nrow(DF_StartEndSummary)>2)"

                  #stop("Debug")


                  ## Overview chart with start and end points
                  ## ****************************************

                  if (UseElevRaster){
                    PlotAltCol <- "dem_altitude"
                  } else {
                    PlotAltCol <- "gps_altitude"
                  }

                  if (WriteChartsToFile) {png(paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "Overview_02_StartEndMarks.png"), width=Chart_Width, height=Chart_Height)}

                  plot(DF_Day$time_local, DF_Day[, PlotAltCol], type="l", xlab="Local time", ylab="Altitude", main=paste0(Chart_Title, " (", PlotAltCol, ") - Start & End Points"))
                  points(DF_StartEndSummary$time_local[DF_StartEndSummary$runmarker_startend=="Start"], DF_StartEndSummary[DF_StartEndSummary$runmarker_startend=="Start", PlotAltCol], pch=19, col=Col_GettingReady)
                  points(DF_StartEndSummary$time_local[DF_StartEndSummary$runmarker_startend=="Start (deleted)"], DF_StartEndSummary[DF_StartEndSummary$runmarker_startend=="Start (deleted)", PlotAltCol], pch=21, col=Col_GettingReady, bg="White")
                  points(DF_StartEndSummary$time_local[DF_StartEndSummary$runmarker_startend=="End"],   DF_StartEndSummary[DF_StartEndSummary$runmarker_startend=="End", PlotAltCol], pch=19, col=Col_PackingUp)
                  points(DF_StartEndSummary$time_local[DF_StartEndSummary$runmarker_startend=="End (deleted)"],   DF_StartEndSummary[DF_StartEndSummary$runmarker_startend=="End (deleted)", PlotAltCol], pch=21, col=Col_PackingUp, bg="White")

                  if (WriteChartsToFile) {dev.off()}

                  rm(PlotAltCol)


                  ## Identify potential runs
                  ## ***********************

                  DF_Day$run_numpot <- NA

                  Num_PotRun <- 1
                  Value <- NA

                  for (Index_Obs in 1:nrow(DF_Day)) {

                    ## Identify start and end
                    if (!is.na(DF_Day$runmarker_startend[Index_Obs])) {
                      if (DF_Day$runmarker_startend[Index_Obs]=="Start") {
                        Value <- Num_PotRun
                      }
                    }

                    ## Mark run
                    DF_Day$run_numpot[Index_Obs] <- Value

                    ## Identify end
                    if (!is.na(DF_Day$runmarker_startend[Index_Obs])) {
                      if (DF_Day$runmarker_startend[Index_Obs]=="End") {
                        Value <- NA
                        Num_PotRun <- Num_PotRun+1
                      }
                    }
                  }  ## End of for loop with Index_Obs
                  rm(Index_Obs, Value)

                  ## Transfer to column that retains original numbering
                  DF_Day$run_numpot_orig <- DF_Day$run_numpot

                  ## Feedback
                  Num_PotRun <- max(DF_Day$run_numpot, na.rm = T)
                  print(paste0("Number of identified potential runs: ", Num_PotRun))
                  Output_Progress <- addToProgressDF(Comment=paste0("Number of identified potential runs: ", Num_PotRun))

                  ## Overview chart with potential runs
                  if (UseElevRaster){
                    PlotAltCol <- "dem_altitude"
                  } else {
                    PlotAltCol <- "gps_altitude"
                  }

                  if (WriteChartsToFile) {png(paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "Overview_03_PotentialRuns.png"), width=Chart_Width, height=Chart_Height)}

                  plot(DF_Day$time_local, DF_Day[, PlotAltCol], type="l", xlab="Local time", ylab="Altitude", main=paste0(Chart_Title, " (", PlotAltCol, ") - Potential runs"))

                  for (Index_PotRun in 1:Num_PotRun) {

                    points(DF_Day$time_local[DF_Day$run_numpot==Index_PotRun], DF_Day[DF_Day$run_numpot==Index_PotRun, PlotAltCol], col=Index_PotRun+1, pch=19)
                    text(mean(DF_Day$time_local[DF_Day$run_numpot==Index_PotRun], na.rm=T), mean(DF_Day[DF_Day$run_numpot==Index_PotRun, PlotAltCol], na.rm=T), Index_PotRun, font=2)

                  }

                  if (WriteChartsToFile) {dev.off()}

                  rm(PlotAltCol)


######### vvvvvvvvvvvvvvvvvvvvvvvvvv
######### 5.4.7 PROCESSING RUNS ====
######### ^^^^^^^^^^^^^^^^^^^^^^^^^^

                  ## Initializing flag that any runs are processed
                  Flag_AnyRunsProcessed <- F

                  for (Index_PotRun in 1:Num_PotRun) {
                    # for (Index_PotRun in 12:12) {

                    # Index_PotRun <- 14

                    ## Feedback
                    ## ********
                    print(paste("PROCESSING POTENTIAL RUN", Index_PotRun, "..."))
                    OutputStringPre <- paste0("Pot Run ", Index_PotRun, " - ")

                    if (nrow(subset(DF_Day, run_numpot==Index_PotRun))==0) {

                      ## Feedback
                      OutputString <- paste0(OutputStringPre, "Number of initial obs zero.")
                      print(OutputString)
                      Output_Progress <- addToProgressDF(Comment=OutputString)
                      DF_PotRunsSkipped <- writeReasonForSkippedRunIntoDF(DF_PotRunsSkipped, UUIDTrackDay, Index_PotRun, OutputString)
                      rm(OutputString)

                      OutputString <- paste0(OutputStringPre, "RUN NOT PROCESSED FURTHER!")
                      print(OutputString)
                      Output_Progress <- addToProgressDF(Comment=OutputString)
                      rm(OutputString)

                    } else {

                      ## Create UUID
                      ## ***********
                      UUIDRun <- createUuid()
                      OutputString <- paste0(OutputStringPre, "Run uuid: ", UUIDRun)
                      print(OutputString)
                      Output_Progress <- addToProgressDF(Comment=OutputString)
                      rm(OutputString)


                      ## Adjusting start and end points
                      ## ******************************

                      Start_Index <- min(which(DF_Day$run_numpot==Index_PotRun))
                      Start_Reason <- DF_Day$runmarker_why[Start_Index]
                      Start_Break <- ifelse(DF_Day$marker_break[Start_Index]==1, T, F)
                      Start_Pause <- ifelse(DF_Day$marker_pause[Start_Index]==1, T, F)
                      End_Index <- max(which(DF_Day$run_numpot==Index_PotRun))
                      End_Reason <- DF_Day$runmarker_why[End_Index]
                      End_Break <- ifelse(DF_Day$marker_break[End_Index]==1, T, F)
                      End_Pause <- ifelse(DF_Day$marker_pause[End_Index]==1, T, F)

                      OutputString <- paste0(OutputStringPre, "Initial observations: ", Start_Index, "-", End_Index, " (", End_Index-Start_Index+1, " obs)")
                      print(OutputString)
                      Output_Progress <- addToProgressDF(Comment=OutputString)
                      rm(OutputString)


                      ## Longest continuous downhill section
                      Run_PeriodSummary_UpDown <- summarizePeriodsInTimeSeries(subset(DF_Day, run_numpot==Index_PotRun), "marker_downhill")
                      Run_PeriodSummary_DownOnly <- Run_PeriodSummary_UpDown[Run_PeriodSummary_UpDown$Value==1,]

                      if(nrow(Run_PeriodSummary_DownOnly)==0) {

                        OutputString <- paste0(OutputStringPre, "Longest continuous downhill section: No downhill section identified!")
                        print(OutputString)
                        Output_Progress <- addToProgressDF(Comment=OutputString)
                        DF_PotRunsSkipped <- writeReasonForSkippedRunIntoDF(DF_PotRunsSkipped, UUIDTrackDay, Index_PotRun, OutputString)
                        rm(OutputString)

                        OutputString <- paste0(OutputStringPre, "RUN NOT PROCESSED FURTHER!")
                        print(OutputString)
                        Output_Progress <- addToProgressDF(Comment=OutputString)
                        rm(OutputString)

                      } else {

                        ## Find indexes for start and end point
                        ContDownhill_Start_Index <- which(DF_Day$datetime_local==Run_PeriodSummary_DownOnly$StartDateTimeLocal[which.max(Run_PeriodSummary_DownOnly$Duration)])
                        ContDownhill_End_Index   <- which(DF_Day$datetime_local==Run_PeriodSummary_DownOnly$EndDateTimeLocal[which.max(Run_PeriodSummary_DownOnly$Duration)])

                        if (ContDownhill_Start_Index > Start_Index) {

                          Index_StartBackwards <- ContDownhill_Start_Index
                          StartIdentified <- F

                          ## Find earliest point within elevation margin
                          while (Index_StartBackwards>Start_Index & StartIdentified==F) {

                            if(DF_Day$dem_altitude[Index_StartBackwards] < DF_Day$dem_altitude[ContDownhill_Start_Index]-UpDown_StartEndAltBuffer) {

                              DF_Day$run_numpot[c(Start_Index:Index_StartBackwards)] <- NA
                              DF_Day$activity[c(Start_Index:Index_StartBackwards)] <- DF_Day$activity[Start_Index-1]
                              DF_Day$runmarker_why[Start_Index] <- NA
                              DF_Day$runmarker_why[Index_StartBackwards+1] <- "Uphill to downhill"

                              Start_Index <- min(which(DF_Day$run_numpot==Index_PotRun))
                              Start_Reason <- DF_Day$runmarker_why[Start_Index]
                              Start_Break <- ifelse(DF_Day$marker_break[Start_Index]==1, T, F)
                              Start_Pause <- ifelse(DF_Day$marker_pause[Start_Index]==1, T, F)

                              StartIdentified <- T

                            }
                            Index_StartBackwards <- Index_StartBackwards-1

                          }
                          rm(Index_StartBackwards, StartIdentified)

                        }

                        if (ContDownhill_End_Index < End_Index) {

                          Index_EndForwardwards <- ContDownhill_End_Index
                          EndIdentified <- F

                          ## Find last point within elevation margin
                          while (Index_EndForwardwards<End_Index & EndIdentified==F) {
                            
                            if(!is.na(DF_Day$dem_altitude[Index_EndForwardwards]) & !is.na(DF_Day$dem_altitude[ContDownhill_End_Index])) {
                              if(DF_Day$dem_altitude[Index_EndForwardwards] > DF_Day$dem_altitude[ContDownhill_End_Index]+UpDown_StartEndAltBuffer) {
  
                                DF_Day$run_numpot[c(Index_EndForwardwards:End_Index)] <- NA
                                DF_Day$activity[c(Index_EndForwardwards:End_Index)] <- DF_Day$activity[End_Index+1]
                                DF_Day$runmarker_why[End_Index] <- NA
                                DF_Day$runmarker_why[Index_EndForwardwards-1] <- "Downhill to uphill"
  
                                End_Index <- max(which(DF_Day$run_numpot==Index_PotRun))
                                End_Reason <- DF_Day$runmarker_why[End_Index]
                                End_Break <- ifelse(DF_Day$marker_break[End_Index]==1, T, F)
                                End_Pause <- ifelse(DF_Day$marker_pause[End_Index]==1, T, F)
  
                                EndIdentified <- T
  
                              }
                            }
                            Index_EndForwardwards <- Index_EndForwardwards+1

                          }
                          rm(Index_EndForwardwards, EndIdentified)

                        }

                        OutputString <- paste0(OutputStringPre, "Longest continuous downhill section within ", UpDown_StartEndAltBuffer, " elevation metre buffer: ", Start_Index, "-", End_Index, " (", End_Index-Start_Index+1, " obs)")
                        print(OutputString)
                        Output_Progress <- addToProgressDF(Comment=OutputString)
                        rm(OutputString)

                        OutputString <- paste0(OutputStringPre, "Original start point (", Start_Index, "): ", DF_Day$time_local[Start_Index], " @ ", DF_Day$dem_altitude[Start_Index], " m asl (", Start_Reason, ")")
                        print(OutputString)
                        Output_Progress <- addToProgressDF(Comment=OutputString)
                        rm(OutputString)

                        OutputString <- paste0(OutputStringPre, "Original end point (", End_Index, "): ", DF_Day$time_local[End_Index], " @ ", DF_Day$dem_altitude[End_Index], " m asl (", End_Reason, ")")
                        print(OutputString)
                        Output_Progress <- addToProgressDF(Comment=OutputString)
                        rm(OutputString)

                        rm(Run_PeriodSummary_UpDown, Run_PeriodSummary_DownOnly)


                        ## Starting point
                        ## **************
                        if (Start_Reason=="Flying") {
                          
                          ## NEW #################
                          ## NEW #################
                          
                          ## Search next/expand break/pause point in forward direction
                          ## Max distance backwards: NA
                          ## Max distance forwards: unlimited (entire run)
                          Adjustment <- adjustRunToNextPreviousBreakPause(DF_Day, PotRunNum=Index_PotRun, IndexStart=Start_Index, IndexEnd=End_Index, StartOrEnd="Start", PreviousOrNextOrBoth="Next", MaxDistPrevious = 0, MaxDistNext = 100000)
                          OutputString <- "Replaced"
                          
                          ## NEW #################
                          ## NEW #################
                          
#                           ## Find next break or pause
#                           NextBreak_Index <- NextStartOrEnd(DF_Day, "marker_break", Start_Index, MaxIter=End_Index-Start_Index,  BeforeOrAfter = "After", StartOrEnd = "Start")
#                           NextPause_Index <- NextStartOrEnd(DF_Day, "marker_pause", Start_Index, MaxIter=End_Index-Start_Index,  BeforeOrAfter = "After", StartOrEnd = "Start")
# 
#                           options(warn=-1)
#                           Start_Index_New <- min(NextBreak_Index, NextPause_Index, na.rm = T)
#                           options(warn=0)
# 
#                           ## No next break/pause found
#                           if (is.na(NextBreak_Index) & is.na(NextPause_Index)) {
#                             DF_Day$run_numpot[c(Start_Index:End_Index)] <- NA
#                             DF_Day$activity[c(Start_Index:End_Index)] <- "Flying"
#                             OutputString <- paste0(OutputStringPre, "No (start) break/pause found --> potential run deleted.")
# 
#                             ## Next pause/break found
#                           } else {
#                             DF_Day$run_numpot[c(Start_Index:(Start_Index_New-1))] <- NA
#                             DF_Day$activity[c(Start_Index:(Start_Index_New-1))] <- "Flying"
# 
#                             if (!is.na(NextBreak_Index) & (Start_Index_New==NextBreak_Index)) {
#                               OutputString <- paste0(OutputStringPre, "Moved start ", (Start_Index_New-Start_Index), " obs later (break).")
#                             } else if (!is.na(NextPause_Index) & (Start_Index_New==NextPause_Index)) {
#                               OutputString <- paste0(OutputStringPre, "Moved start ", (Start_Index_New-Start_Index), " obs later (pause).")
#                             } else {
#                               stop("Error: Start - flying - next pause/break found")
#                             }
# 
#                           }
#                           rm(Start_Index_New, NextBreak_Index, NextPause_Index)


                        } else if (Start_Reason=="Uphill to downhill") {
                          
                          ## NEW #################
                          ## NEW #################
                          
                          ## Search next/expand break/pause point in both directions
                          ## Max distance backwards: 200m
                          ## Max distance forwards: unlimited (entire run)
                          Adjustment <- adjustRunToNextPreviousBreakPause(DF_Day, PotRunNum=Index_PotRun, IndexStart=Start_Index, IndexEnd=End_Index, StartOrEnd="Start", PreviousOrNextOrBoth="Both", MaxDistPrevious = 200, MaxDistNext = 100000)
                          OutputString <- "Replaced"
                          
                          ## NEW #################
                          ## NEW #################
                          
#                           ## Start point is already a break --> expand break
#                           if(Start_Break) {
# 
#                             Start_Index_New <- NextStartOrEnd(DF_Day, "marker_break", Start_Index, BeforeOrAfter = "Before", StartOrEnd = "Start")
#                             DF_Day$run_numpot[c(Start_Index_New:Start_Index)] <- Index_PotRun
# 
#                             OutputString <- paste0(OutputStringPre, "Moved start ", (Start_Index-Start_Index_New), " obs earlier (break).")
#                             rm(Start_Index_New)
# 
#                             ## Start point is already a pause --> expand pause
#                           } else if (Start_Pause) {
# 
#                             Start_Index_New <- NextStartOrEnd(DF_Day, "marker_pause", Start_Index, BeforeOrAfter = "Before", StartOrEnd = "Start")
#                             DF_Day$run_numpot[c(Start_Index_New:Start_Index)] <- Index_PotRun
# 
#                             OutputString <- paste0(OutputStringPre, "Moved start ", (Start_Index-Start_Index_New), " obs earlier (pause).")
#                             rm(Start_Index_New)
# 
#                             ## Start point not a pause or break --> Find next break/pause
#                           } else {
#                             
#                             # OutputString <- paste0(OutputStringPre, "Start: Uphill to downhill - no break/pause - did nothing")
# 
#                             NextBreak_Index <- NextStartOrEnd(DF_Day, "marker_break", Start_Index, MaxIter=End_Index-Start_Index,  BeforeOrAfter = "After", StartOrEnd = "Start")
#                             NextPause_Index <- NextStartOrEnd(DF_Day, "marker_pause", Start_Index, MaxIter=End_Index-Start_Index,  BeforeOrAfter = "After", StartOrEnd = "Start")
# 
#                             options(warn=-1)
#                             Start_Index_New <- min(NextBreak_Index, NextPause_Index, na.rm = T)
#                             options(warn=0)
# 
#                             ## No next break/pause found
#                             if (is.na(NextBreak_Index) & is.na(NextPause_Index)) {
#                               DF_Day$run_numpot[c(Start_Index:End_Index)] <- NA
#                               DF_Day$activity[c(Start_Index:End_Index)] <- DF_Day$activity[Start_Index-1]
#                               OutputString <- paste0(OutputStringPre, "No (start) break/pause found --> potential run deleted.")
# 
#                               ## Next break
#                             } else if  (!is.na(NextBreak_Index) & Start_Index_New==NextBreak_Index) {
#                               DF_Day$run_numpot[c(Start_Index:(Start_Index_New-1))] <- NA
#                               DF_Day$activity[c(Start_Index:(Start_Index_New-1))] <- DF_Day$activity[Start_Index-1]
# 
#                               OutputString <- paste0(OutputStringPre, "Moved start ", (Start_Index_New-Start_Index), " obs later (break).")
#                               # OutputString <- paste0(OutputStringPre, "Did nothing with the start")
# 
#                               ## Next pause
#                             } else if (!is.na(NextPause_Index) & Start_Index_New==NextPause_Index) {
#                               DF_Day$run_numpot[c(Start_Index:(Start_Index_New-1))] <- NA
#                               DF_Day$activity[c(Start_Index:(Start_Index_New-1))] <- DF_Day$activity[Start_Index-1]
# 
#                               OutputString <- paste0(OutputStringPre, "Moved start ", (Start_Index_New-Start_Index), " obs later (pause).")
#                               # OutputString <- paste0(OutputStringPre, "Did nothing with the start")
# 
#                               ## Error
#                             } else {
# 
#                               stop(paste0(OutputStringPre, "Error: Start - lrg distance - next break/pause"))
# 
#                             }
# 
#                             rm(Start_Index_New, NextBreak_Index, NextPause_Index)
# 
#                           }

                        } else if (Start_Reason=="Lrg dist") {

                          ## NEW #################
                          ## NEW #################
                          
                          ## Search next/expand break/pause point in forward directions
                          ## Max distance backwards: NA
                          ## Max distance forwards: unlimited (entire run)
                          Adjustment <- adjustRunToNextPreviousBreakPause(DF_Day, PotRunNum=Index_PotRun, IndexStart=Start_Index, IndexEnd=End_Index, StartOrEnd="Start", PreviousOrNextOrBoth="Next", MaxDistPrevious = 0, MaxDistNext = 100000)
                          OutputString <- "Replaced"
                          
                          ## NEW #################
                          ## NEW #################
                          
                          
#                           if(Start_Pause | Start_Break) {
# 
#                             ## Do nothing if new run starts with break or pause
#                             OutputString <- paste0(OutputStringPre, "Start: Large distance - break/pause - did nothing")
# 
#                           } else {
# 
#                             # OutputString <- paste0(OutputStringPre, "Start: Large distance - no break/pause - did nothing")
# 
#                             ## Find next break or pause
#                             NextBreak_Index <- NextStartOrEnd(DF_Day, "marker_break", Start_Index, MaxIter=End_Index-Start_Index,  BeforeOrAfter = "After", StartOrEnd = "Start")
#                             NextPause_Index <- NextStartOrEnd(DF_Day, "marker_pause", Start_Index, MaxIter=End_Index-Start_Index,  BeforeOrAfter = "After", StartOrEnd = "Start")
# 
#                             options(warn=-1)
#                             Start_Index_New <- min(NextBreak_Index, NextPause_Index, na.rm = T)
#                             options(warn=0)
# 
#                             ## No next break/pause found
#                             if (is.na(NextBreak_Index) & is.na(NextPause_Index)) {
#                               DF_Day$run_numpot[c(Start_Index:End_Index)] <- NA
#                               DF_Day$activity[c(Start_Index:End_Index)] <- DF_Day$activity[Start_Index-1]
#                               OutputString <- paste0(OutputStringPre, "No (start) break/pause found --> potential run deleted.")
# 
#                               ## Next break
#                             } else if  (!is.na(NextBreak_Index) & Start_Index_New==NextBreak_Index) {
#                               DF_Day$run_numpot[c(Start_Index:(Start_Index_New-1))] <- NA
#                               DF_Day$activity[c(Start_Index:(Start_Index_New-1))] <- DF_Day$activity[Start_Index-1]
# 
#                               OutputString <- paste0(OutputStringPre, "Moved start ", (Start_Index_New-Start_Index), " obs later (break).")
# 
#                               ## Next pause
#                             } else if (!is.na(NextPause_Index) & Start_Index_New==NextPause_Index) {
#                               DF_Day$run_numpot[c(Start_Index:(Start_Index_New-1))] <- NA
#                               DF_Day$activity[c(Start_Index:(Start_Index_New-1))] <- DF_Day$activity[Start_Index-1]
# 
#                               OutputString <- paste0(OutputStringPre, "Moved start ", (Start_Index_New-Start_Index), " obs later (pause).")
# 
#                               ## Error
#                             } else {
# 
#                               stop(paste0(OutputStringPre, "Error: Start - lrg distance - next break/pause"))
# 
#                             }
# 
#                             rm(Start_Index_New, NextBreak_Index, NextPause_Index)
# 
#                           }
 
                        } else {

                          stop(paste0(OutputStringPre, "Undefined start point reason."))

                        }
                        
                        ## Transfer
                        DF_Day <- Adjustment$DataFrame
                        Start_Index <- Adjustment$AdjustedIndexStart
 
#                         print(OutputString)
#                         Output_Progress <- addToProgressDF(Comment=OutputString)
#                         rm(OutputString)


                        ## End point
                        ## *********
                        
                        if (End_Reason=="Flying") {
                          
                          ## NEW #################
                          ## NEW #################
                          
                          ## Search previous/expand break/pause point in backwards directions
                          ## Max distance backwards: unlimited (entire run)
                          ## Max distance forwards: NA
                          Adjustment <- adjustRunToNextPreviousBreakPause(DF_Day, PotRunNum=Index_PotRun, IndexStart=Start_Index, IndexEnd=End_Index, StartOrEnd="End", PreviousOrNextOrBoth="Previous", MaxDistPrevious = 100000, MaxDistNext = 0)
                          OutputString <- "Replaced"
                          
                          ## NEW #################
                          ## NEW #################
                          

#                           ## Find last previous break/pause
#                           PrevBreak_Index <- NextStartOrEnd(DF_Day, "marker_break", End_Index, MaxIter=End_Index-Start_Index, BeforeOrAfter = "Before", StartOrEnd = "End")
#                           PrevPause_Index <- NextStartOrEnd(DF_Day, "marker_pause", End_Index, MaxIter=End_Index-Start_Index, BeforeOrAfter = "Before", StartOrEnd = "End")
# 
#                           options(warn=-1)
#                           End_Index_New <- max(PrevBreak_Index, PrevPause_Index, na.rm=T)
#                           options(warn=0)
# 
#                           if (is.na(PrevBreak_Index) & is.na(PrevPause_Index)) {
#                             DF_Day$run_numpot[c(Start_Index:End_Index)] <- NA
#                             DF_Day$activity[c(Start_Index:End_Index)] <- DF_Day$activity[Start_Index-1]
#                             OutputString <- paste0(OutputStringPre, "No (end) break/pause found --> potential run deleted.")
#                           } else {
#                             DF_Day$run_numpot[c((End_Index_New+1):End_Index)] <- NA
#                             DF_Day$activity[c((End_Index_New+1):End_Index)] <- "Flying"
# 
#                             if(!is.na(PrevBreak_Index) & (End_Index_New==PrevBreak_Index)) {
#                               OutputString <- paste0(OutputStringPre, "Moved end ", (End_Index-End_Index_New+1), " obs earlier (break).")
#                             } else if (!is.na(PrevPause_Index) & (End_Index_New==PrevPause_Index)) {
#                               OutputString <- paste0(OutputStringPre, "Moved end ", (End_Index-End_Index_New+1), " obs earlier (pause).")
#                             } else {
#                               stop(paste0(OutputStringPre, "Error: End - flying - previous break/pause"))
#                             }
#                           }
#                           rm(End_Index_New, PrevBreak_Index, PrevPause_Index)


                        } else if (End_Reason=="Downhill to uphill") {
                          
                          ## NEW #################
                          ## NEW #################
                          
                          ## Search/expand break/pause point in both directions
                          ## Max distance backwards: unlimited (entire run)
                          ## Max distance forwards: 200 m
                          Adjustment <- adjustRunToNextPreviousBreakPause(DF_Day, PotRunNum=Index_PotRun, IndexStart=Start_Index, IndexEnd=End_Index, StartOrEnd="End", PreviousOrNextOrBoth="Both", MaxDistPrevious = 100000, MaxDistNext = 200)
                          OutputString <- "Replaced"
                          
                          ## NEW #################
                          ## NEW #################
                          
#                           if ((is.na(End_Break) & is.na(End_Pause))|(!End_Break & !End_Pause)) {
#                             
#                             # OutputString <- paste0(OutputStringPre, "End: Downhill to uphill - no break/pause - did nothing")
#                             
#                             PrevBreak_Index <- NextStartOrEnd(DF_Day, "marker_break", End_Index, MaxIter=End_Index-Start_Index, BeforeOrAfter = "Before", StartOrEnd = "End")
#                             PrevPause_Index <- NextStartOrEnd(DF_Day, "marker_pause", End_Index, MaxIter=End_Index-Start_Index, BeforeOrAfter = "Before", StartOrEnd = "End")
#                             
#                             options(warn=-1)
#                             End_Index_New <- max(PrevBreak_Index, PrevPause_Index, na.rm=T)
#                             options(warn=0)
#                             
#                             if (is.na(PrevBreak_Index) & is.na(PrevPause_Index)) {
#                               DF_Day$run_numpot[c(Start_Index:End_Index)] <- NA
#                               DF_Day$activity[c(Start_Index:End_Index)] <- DF_Day$activity[Start_Index-1]
#                               OutputString <- paste0(OutputStringPre, "No (end) break/pause found --> potential run deleted.")
#                               
#                             } else if (!is.na(PrevBreak_Index) & End_Index_New==PrevBreak_Index){
#                               DF_Day$run_numpot[c((End_Index_New+1):End_Index)] <- NA
#                               DF_Day$activity[c((End_Index_New+1):End_Index)] <- DF_Day$activity[End_Index+1]
#                               OutputString <- paste0(OutputStringPre, "Moved end ", (End_Index-End_Index_New+1), " obs earlier (break).")
#                               
#                             } else if (!is.na(PrevPause_Index) & End_Index_New==PrevPause_Index) {
#                               DF_Day$activity[c((End_Index_New+1):End_Index)] <- DF_Day$activity[End_Index+1]
#                               DF_Day$run_numpot[c((End_Index_New+1):End_Index)] <- NA
#                               OutputString <- paste0(OutputStringPre, "Moved end ", (End_Index-End_Index_New+1), " obs earlier (pause).")
#                               
#                             } else {
#                               stop(paste0(OutputStringPre, "Error: End - downhill - previous break/pause"))
#                             }
#                             
#                             rm(End_Index_New, PrevBreak_Index, PrevPause_Index)
#                             
#                           ## End point is already a break --> expand break
#                           } else if(End_Break) {
# 
#                             End_Index_New <- NextStartOrEnd(DF_Day, "marker_break", End_Index, BeforeOrAfter = "After", StartOrEnd = "End")
#                             DF_Day$run_numpot[c(End_Index:End_Index_New)] <- Index_PotRun
#                             OutputString <- paste0(OutputStringPre, "Moved end ", (End_Index_New-End_Index), " obs later (break).")
#                             rm(End_Index_New)
# 
#                             ## End point is already a pause --> expand pause
#                           } else if (End_Pause) {
# 
#                             End_Index_New <- NextStartOrEnd(DF_Day, "marker_pause", End_Index, BeforeOrAfter = "After", StartOrEnd = "End")
#                             DF_Day$run_numpot[c(End_Index:End_Index_New)] <- Index_PotRun
#                             OutputString <- paste0(OutputStringPre, "Moved end ", (End_Index_New-End_Index), " obs later (pause).")
#                             rm(End_Index_New)
# 
#                             ## End point not a pause or break --> Find previous break
#                           }


                        } else if (End_Reason=="Onto relevant catroad") {
                          
                          ## NEW #################
                          ## NEW #################
                          
                          ## Search/expand break/pause points in both directions
                          ## Max distance backwards: unlimited (entire run)
                          ## Max distance forwards: 500 m
                          Adjustment <- adjustRunToNextPreviousBreakPause(DF_Day, PotRunNum=Index_PotRun, IndexStart=Start_Index, IndexEnd=End_Index, StartOrEnd="End", PreviousOrNextOrBoth="Both", MaxDistPrevious = 100000, MaxDistNext = 500)
                          OutputString <- "Replaced"
                          
                          ## NEW #################
                          ## NEW #################
                          
                          
#                           ## End point is already a break --> expand break to end
#                           if(End_Break) {
#                             
#                             End_Index_New <- NextStartOrEnd(DF_Day, "marker_break", End_Index, BeforeOrAfter = "After", StartOrEnd = "End")
#                             DF_Day$run_numpot[c(End_Index:End_Index_New)] <- Index_PotRun
#                             OutputString <-paste0(OutputStringPre, "Moved end ", (End_Index_New-End_Index), " obs later (break).")
#                             rm(End_Index_New)
# 
#                             ## End point is already a pause --> expand pause to end
#                           } else if (End_Pause) {
#                             
#                             End_Index_New <- NextStartOrEnd(DF_Day, "marker_pause", End_Index, BeforeOrAfter = "After", StartOrEnd = "End")
#                             DF_Day$run_numpot[c(End_Index:End_Index_New)] <- Index_PotRun
#                             OutputString <-paste0(OutputStringPre, "Moved end ", (End_Index_New-End_Index), " obs later (pause).")
#                             rm(End_Index_New)
# 
#                             ## End point not a pause or break --> Find previous or next pause/break (whatever is closer)
#                           } else {
#                             
#                             End_Index_PrevBreak_End <- NextStartOrEnd(DF_Day, "marker_break", End_Index, MaxIter=End_Index-Start_Index, BeforeOrAfter = "Before", StartOrEnd = "End")
#                             End_Index_PrevPause_End <- NextStartOrEnd(DF_Day, "marker_pause", End_Index, MaxIter=End_Index-Start_Index, BeforeOrAfter = "Before", StartOrEnd = "End")
# 
#                             options(warn=-1)
#                             End_Index_Prev_End <- max(End_Index_PrevBreak_End, End_Index_PrevPause_End, na.rm=T)
#                             options(warn=0)
#                             
#                             End_Index_NextBreak_Start <- NextStartOrEnd(DF_Day, "marker_break", End_Index, MaxIter=max(which(!is.na(DF_Day$marker_break)))-End_Index, BeforeOrAfter = "After", StartOrEnd = "Start")
#                             End_Index_NextPause_Start <- NextStartOrEnd(DF_Day, "marker_pause", End_Index, MaxIter=max(which(!is.na(DF_Day$marker_pause)))-End_Index, BeforeOrAfter = "After", StartOrEnd = "Start")
#                             
#                             options(warn=-1)
#                             End_Index_Next_Start <- min(End_Index_NextBreak_Start, End_Index_NextPause_Start, na.rm=T)
#                             options(warn=0)
#                             
#                             if (is.finite(End_Index_Prev_End) & is.finite(End_Index_Next_Start)) {
# 
#                               SecToPrev_End   <- DF_Day$datetime_local[End_Index] - DF_Day$datetime_local[End_Index_Prev_End]
#                               SecToNext_Start <- DF_Day$datetime_local[End_Index_Next_Start] - DF_Day$datetime_local[End_Index]
# 
#                               if (SecToPrev_End < SecToNext_Start) {
# 
#                                 DF_Day$run_numpot[c((End_Index_Prev_End+1):End_Index)] <- NA
#                                 DF_Day$activity[c((End_Index_Prev_End+1):End_Index)] <- DF_Day$activity[End_Index+1]
#                                 OutputString <-paste0(OutputStringPre, "Moved end ", (End_Index-(End_Index_Prev_End-1)), " obs earlier (break).")
# 
#                               } else {
# 
#                                 DF_Day$run_numpot[c(End_Index:End_Index_Next_Start)] <- Index_PotRun
#                                 OutputString <-paste0(OutputStringPre, "Moved end ", (End_Index_Next_Start-End_Index), " obs later (break).")
# 
#                               }
# 
#                               rm(SecToPrev_End, SecToNext_Start)
# 
#                             } else if (is.finite(End_Index_Prev_End)) {
# 
#                               DF_Day$run_numpot[c((End_Index_Prev_End+1):End_Index)] <- NA
#                               DF_Day$activity[c((End_Index_Prev_End+1):End_Index)] <- DF_Day$activity[End_Index+1]
#                               OutputString <-paste0(OutputStringPre, "Moved end ", (End_Index-(End_Index_Prev_End-1)), " obs earlier (break).")
# 
#                             } else if (is.finite(End_Index_Next_Start)) {
# 
#                               DF_Day$run_numpot[c(End_Index:End_Index_Next_Start)] <- Index_PotRun
#                               OutputString <-paste0(OutputStringPre, "Moved end ", (End_Index_Next_Start-End_Index), " obs later (break).")
# 
#                             } else {
# 
#                               OutputString <-paste0(OutputStringPre, "No (end) break/pause point found because run likely already deleted.")
#                               # stop(paste0(OutputStringPre, "Error: End - catroad - previous or next break/pause"))
# 
#                             }
# 
#                             rm(End_Index_PrevBreak_End, End_Index_PrevPause_End, End_Index_Prev_End)
#                             rm(End_Index_NextBreak_Start, End_Index_NextPause_Start, End_Index_Next_Start)
# 
#                           }


                        } else if (End_Reason=="Lrg dist" | End_Reason=="Next obs outside") {
                          
                          ## NEW #################
                          ## NEW #################
                          
                          ## Search/expand break/pause point in backwards directions
                          ## Max distance backwards: unlimited (entire run)
                          ## Max distance forwards: NA
                          Adjustment <- adjustRunToNextPreviousBreakPause(DF_Day, PotRunNum=Index_PotRun, IndexStart=Start_Index, IndexEnd=End_Index, StartOrEnd="End", PreviousOrNextOrBoth="Previous", MaxDistPrevious = 100000, MaxDistNext = 0)
                          OutputString <- "Replaced"
                          
                          ## NEW #################
                          ## NEW #################

#                           if(End_Pause | End_Break) {
# 
#                             ## Do nothing if new run starts with break or pause
#                             OutputString <- paste0(OutputStringPre, "End: Large distance - break/pause - did nothing")
# 
#                           } else {
# 
#                             ## Find next break or pause
#                             PrevBreak_Index <- NextStartOrEnd(DF_Day, "marker_break", End_Index, MaxIter=End_Index-Start_Index, BeforeOrAfter = "Before", StartOrEnd = "End")
#                             PrevPause_Index <- NextStartOrEnd(DF_Day, "marker_pause", End_Index, MaxIter=End_Index-Start_Index, BeforeOrAfter = "Before", StartOrEnd = "End")
# 
#                             options(warn=-1)
#                             End_Index_New <- max(PrevBreak_Index, PrevPause_Index, na.rm=T)
#                             options(warn=0)
# 
#                             if (is.na(PrevBreak_Index) & is.na(PrevPause_Index)) {
#                               DF_Day$run_numpot[c(Start_Index:End_Index)] <- NA
#                               DF_Day$activity[c(Start_Index:End_Index)] <- DF_Day$activity[End_Index+1]
#                               OutputString <- paste0(OutputStringPre, "No (end) break/pause found --> potential run deleted.")
# 
#                             } else if (!is.na(PrevBreak_Index) & End_Index_New==PrevBreak_Index){
#                               DF_Day$run_numpot[c((End_Index_New+1):End_Index)] <- NA
#                               DF_Day$activity[c((End_Index_New+1):End_Index)] <- DF_Day$activity[End_Index+1]
#                               OutputString <- paste0(OutputStringPre, "Moved end ", (End_Index-End_Index_New+1), " obs earlier (break).")
# 
#                             } else if (!is.na(PrevPause_Index) & End_Index_New==PrevPause_Index) {
#                               DF_Day$run_numpot[c((End_Index_New+1):End_Index)] <- NA
#                               DF_Day$activity[c((End_Index_New+1):End_Index)] <- DF_Day$activity[End_Index+1]
#                               OutputString <- paste0(OutputStringPre, "Moved end ", (End_Index-End_Index_New+1), " obs earlier (pause).")
# 
#                             } else {
#                               stop(paste0(OutputStringPre, "Error: End - downhill - previous break/pause"))
#                             }
# 
#                             rm(End_Index_New, PrevBreak_Index, PrevPause_Index)
# 
#                           }

                        } else if (End_Reason=="Last break") {

                          OutputString <- paste0(OutputStringPre, "End not treated (last break).")
                          print(OutputString)
                          Output_Progress <- addToProgressDF(Comment=OutputString)
                          rm(OutputString)

                        } else if (End_Reason=="Last obs") {

                          OutputString <- paste0(OutputStringPre, "End not treated (last obs).")
                          print(OutputString)
                          Output_Progress <- addToProgressDF(Comment=OutputString)
                          rm(OutputString)

                        } else {

                          stop(paste0(OutputStringPre, "Undefined end point reason: ", End_Reason))

                        }
                        
                        ## Transfer
                        DF_Day <- Adjustment$DataFrame

#                         print(OutputString)
#                         Output_Progress <- addToProgressDF(Comment=OutputString)
#                         rm(OutputString)

                        ## Clean-up
                        rm(Adjustment)
                        rm(Start_Index, Start_Break, Start_Pause)
                        rm(End_Index, End_Break, End_Pause)
                        ## Start_Reason and End_Reason kept for SPDF_Runs


                        ## Create dataframe for potential run
                        ## ***********************************
                        DF_PotRun <- subset(DF_Day, run_numpot==Index_PotRun)

                        if (nrow(DF_PotRun)==0) {
                          PotRun_DurationSec <- 0
                        } else {
                          PotRun_DurationSec <- (DF_PotRun$datetime_local[nrow(DF_PotRun)]-DF_PotRun$datetime_local[1])*24*60*60
                        }

                        # if (nrow(DF_PotRun) < NumObsRunMin) {

                        if (PotRun_DurationSec < PotRun_MinDuration) {

                          ## Feedback
                          OutputString <- paste0(OutputStringPre, "Duration of potential run below min threshold (", PotRun_MinDuration, " sec): ", PotRun_DurationSec, " sec")
                          print(OutputString)
                          Output_Progress <- addToProgressDF(Comment=OutputString)
                          DF_PotRunsSkipped <- writeReasonForSkippedRunIntoDF(DF_PotRunsSkipped, UUIDTrackDay, Index_PotRun, OutputString)
                          rm(OutputString)

                          OutputString <- paste0(OutputStringPre, "RUN NOT PROCESSED FURTHER!")
                          print(OutputString)
                          Output_Progress <- addToProgressDF(Comment=OutputString)
                          rm(OutputString)

                        } else {

                          ## Feedback
                          OutputString <- paste0(OutputStringPre, "Duration of potential run above min threshold (", PotRun_MinDuration, " sec): ", PotRun_DurationSec, " sec")
                          print(OutputString)
                          Output_Progress <- addToProgressDF(Comment=OutputString)
                          rm(OutputString)

                          ## Initial overview plot
                          if (UseElevRaster){
                            PlotAltCol <- "dem_altitude"
                          } else {
                            PlotAltCol <- "gps_altitude"
                          }

                          RunChart_FileNameStart <- paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "RunFiles", createOSSpecFolderLink(), "PotRun_", ifelse(Index_PotRun<10, "0", ""), Index_PotRun, "_")

                          if (WriteChartsToFile) {png(paste0(RunChart_FileNameStart, "01_WithinProfile.png"), width=Chart_Width, height=Chart_Height)}

                          plot(DF_Day$time_local, DF_Day[,PlotAltCol], main=paste0(Chart_Title, " - Pot Run ", Index_PotRun, " (", PlotAltCol, ")"), type="l", xlab="Local Time", ylab="Altitude")
                          points(DF_PotRun$time_local, DF_PotRun[,PlotAltCol], col="red", pch=20)

                          if (WriteChartsToFile) {dev.off()}

                          rm(PlotAltCol)


                          ## Adding UUID for run and marking periods with activity
                          ## *****************************************************

                          ## Add UUID for Run
                          DF_PotRun$gpsruns_uuid <- UUIDRun

                          ## Skiing
                          DF_PotRun$activity <- "Skiing"

                          ## Waiting
                          DF_PotRun$activity[DF_PotRun$marker_pause==1 | DF_PotRun$marker_break==1] <- "Waiting"


                          ## Marking first and last waiting period as Getting ready and Packing up
                          ## *********************************************************************
                          PeriodSummary_Activity <- summarizePeriodsInTimeSeries(DF_PotRun, "activity")
                          PeriodSummary_Waiting <- subset(PeriodSummary_Activity, Value=="Waiting")
                          
                          if (nrow(PeriodSummary_Waiting)==0) {
                            
                            ## Feedback
                            OutputString <- paste0(OutputStringPre, "No waiting points found!")
                            print(OutputString)
                            Output_Progress <- addToProgressDF(Comment=OutputString)
                            DF_PotRunsSkipped <- writeReasonForSkippedRunIntoDF(DF_PotRunsSkipped, UUIDTrackDay, Index_PotRun, OutputString)
                            rm(OutputString)
                            rm(Check_SkiDuration)
                            
                            OutputString <- paste0(OutputStringPre, "RUN NOT PROCESSED FURTHER!")
                            print(OutputString)
                            Output_Progress <- addToProgressDF(Comment=OutputString)
                            rm(OutputString)
                            
                            
                          } else {
                          
                            ## For 'Lrg dist' start only if duration is long enough for break
                            if(Start_Reason!="Lrg dist" | PeriodSummary_Waiting$Duration[1]>Break_MinTime) {
                              DF_PotRun$activity[DF_PotRun$datetime_local>= PeriodSummary_Waiting$StartDateTimeLocal[1] & DF_PotRun$datetime_local <= PeriodSummary_Waiting$EndDateTimeLocal[1]] <- "GettingReady"
                            }
  
                            ## For 'Lrg dist' end only if duration is long enough for break
                            if(End_Reason!="Lrg dist" | tail(PeriodSummary_Waiting$Duration, 1)>Break_MinTime) {
                              DF_PotRun$activity[DF_PotRun$datetime_local>= tail(PeriodSummary_Waiting$StartDateTimeLocal,1) & DF_PotRun$datetime_local <= tail(PeriodSummary_Waiting$EndDateTimeLocal, 1)] <- "PackingUp"
                            }
  
                            rm(PeriodSummary_Activity)
                            rm(PeriodSummary_Waiting)
  
  
                            ## Check whether duration of skiing long enough
                            ## ********************************************
  
                            Check_SkiDuration <- sum(DF_PotRun$timediff_sec[DF_PotRun$activity=="Skiing"])
  
                            if (Check_SkiDuration < PotRun_MinDurationSkiing) {
  
                              ## Feedback
                              OutputString <- paste0(OutputStringPre, "Duration of skiing below min threshold (", PotRun_MinDurationSkiing, " sec): ", Check_SkiDuration, " sec")
                              print(OutputString)
                              Output_Progress <- addToProgressDF(Comment=OutputString)
                              DF_PotRunsSkipped <- writeReasonForSkippedRunIntoDF(DF_PotRunsSkipped, UUIDTrackDay, Index_PotRun, OutputString)
                              rm(OutputString)
                              rm(Check_SkiDuration)
  
                              OutputString <- paste0(OutputStringPre, "RUN NOT PROCESSED FURTHER!")
                              print(OutputString)
                              Output_Progress <- addToProgressDF(Comment=OutputString)
                              rm(OutputString)
  
  
                            } else {
  
                              ## Feedback
                              OutputString <- paste0(OutputStringPre, "Duration of skiing above min threshold (", PotRun_MinDurationSkiing, " sec): ", Check_SkiDuration, " sec")
                              print(OutputString)
                              Output_Progress <- addToProgressDF(Comment=OutputString)
                              rm(OutputString)
                              rm(Check_SkiDuration)
  
  
                              ## Check whether max skiing speed above threshold
                              ## **********************************************
  
                              SkiingSpdAvg <- round(mean(DF_PotRun$hspeed_kmh[DF_PotRun$activity=="Skiing"]),1)
                              SkiingSpdMax <- max(DF_PotRun$hspeed_kmh[DF_PotRun$activity=="Skiing"], na.rm = T)
  
                              if (SkiingSpdMax<Run_MinMaxSkiingSpeed) {
  
                                OutputString <- paste0(OutputStringPre, "Max skiing speed below min threshold (", Run_MinMaxSkiingSpeed, " km/h): ", SkiingSpdMax, " km/h")
                                print(OutputString)
                                Output_Progress <- addToProgressDF(Comment=OutputString)
                                DF_PotRunsSkipped <- writeReasonForSkippedRunIntoDF(DF_PotRunsSkipped, UUIDTrackDay, Index_PotRun, OutputString)
                                rm(OutputString)
  
                                OutputString <- paste0(OutputStringPre, "RUN NOT PROCESSED FURTHER!")
                                print(OutputString)
                                Output_Progress <- addToProgressDF(Comment=OutputString)
                                rm(OutputString)
  
  
                              } else {
  
                                ## Feedback
                                OutputString <- paste0(OutputStringPre, "Max skiing speed above min threshold (", Run_MinMaxSkiingSpeed, " km/h): ", SkiingSpdMax, " km/h")
                                print(OutputString)
                                Output_Progress <- addToProgressDF(Comment=OutputString)
                                rm(OutputString)
                                
                                
                                ## Check percentage of run on cat road
                                ## ***********************************
                                
                                PercentObsOnCatRoad <- sum(DF_PotRun$marker_oncatroad)/nrow(DF_PotRun)
                                
                                if (PercentObsOnCatRoad > CatRoad_MaxPercentageOfRun) {
                                  
                                  OutputString <- paste0(OutputStringPre, "Percentage of observations on cat roads above threshold (", round(CatRoad_MaxPercentageOfRun*100), "%): ", round(PercentObsOnCatRoad*100), "%")
                                  print(OutputString)
                                  Output_Progress <- addToProgressDF(Comment=OutputString)
                                  DF_PotRunsSkipped <- writeReasonForSkippedRunIntoDF(DF_PotRunsSkipped, UUIDTrackDay, Index_PotRun, OutputString)
                                  rm(OutputString)
                                  
                                  OutputString <- paste0(OutputStringPre, "RUN NOT PROCESSED FURTHER!")
                                  print(OutputString)
                                  Output_Progress <- addToProgressDF(Comment=OutputString)
                                  rm(OutputString)
                                  
                                } else {
  
                                  ## Feedback
                                  if (Flag_CatSkiing) {
                                    OutputString <- paste0(OutputStringPre, "Percentage of observations on cat roads below threshold (", round(CatRoad_MaxPercentageOfRun*100), "%): ", round(PercentObsOnCatRoad*100), "%")
                                    print(OutputString)
                                    Output_Progress <- addToProgressDF(Comment=OutputString)
                                    rm(OutputString)
                                  }
                                
                                  
                                  ## Set flag that any runs are processed
                                  ## ************************************
                                  Flag_AnyRunsProcessed <- T
    
    
                                  ## Transfer activity and Run UUID back to DF_Day
                                  ## *********************************************
                                  for (Index_Points in 1:nrow(DF_PotRun)) {
                                    DF_Day$activity[DF_Day$uuid==DF_PotRun$uuid[Index_Points]] <- DF_PotRun$activity[Index_Points]
                                    DF_Day$gpsruns_uuid[DF_Day$uuid==DF_PotRun$uuid[Index_Points]] <- UUIDRun
                                  }
    
    
                                  ## Plotting of run
                                  ## ***************
    
                                  if (WriteChartsToFile) {png(paste0(RunChart_FileNameStart, "02_MapProfile.png"), width=Chart_Width, height=Chart_Height)}
    
                                  par(mfrow=c(1,2), oma=c(0,0,1.5,0))
    
                                  # Map
                                  plot(DF_PotRun$gps_longitude, DF_PotRun$gps_latitude, xlab="Lon", ylab="Lat", main="Map",  type="l", cex.lab=0.75, cex.main=0.75, cex.axis=0.5)
                                  points(DF_PotRun$gps_longitude[DF_PotRun$activity=="Waiting"], DF_PotRun$gps_latitude[DF_PotRun$activity=="Waiting"], col=Col_Pause, pch=20)
                                  points(DF_PotRun$gps_longitude[DF_PotRun$marker_break==1], DF_PotRun$gps_latitude[DF_PotRun$marker_break==1], col=Col_Break, pch=20)
                                  points(DF_PotRun$gps_longitude[DF_PotRun$activity=="GettingReady"], DF_PotRun$gps_latitude[DF_PotRun$activity=="GettingReady"], col=Col_GettingReady, pch=20)
                                  points(DF_PotRun$gps_longitude[DF_PotRun$activity=="Skiing"], DF_PotRun$gps_latitude[DF_PotRun$activity=="Skiing"], col=Col_Skiing, pch=20)
                                  points(DF_PotRun$gps_longitude[DF_PotRun$activity=="PackingUp"], DF_PotRun$gps_latitude[DF_PotRun$activity=="PackingUp"], col=Col_PackingUp, pch=20)
                                  points(DF_PotRun$gps_longitude[DF_PotRun$activity=="Flying"], DF_PotRun$gps_latitude[DF_PotRun$activity=="Flying"], col=Col_Flying, pch=20)
    
                                  ## Profile
                                  if (UseElevRaster){
                                    PlotAltCol <- "dem_altitude"
                                  } else {
                                    PlotAltCol <- "gps_altitude"
                                  }
    
                                  plot(DF_PotRun$time_local, DF_PotRun[,PlotAltCol], xlab="Local time", ylab="Altitude", main=paste0("Profile (", PlotAltCol, ")"), type="l", cex.lab=0.75, cex.main=0.75, cex.axis=0.5)
                                  points(DF_PotRun$time_local[DF_PotRun$activity=="Waiting"], DF_PotRun[DF_PotRun$activity=="Waiting", PlotAltCol], col=Col_Pause, pch=20)
                                  points(DF_PotRun$time_local[DF_PotRun$marker_break==1], DF_PotRun[DF_PotRun$marker_break==1, PlotAltCol], col=Col_Break, pch=20)
                                  points(DF_PotRun$time_local[DF_PotRun$activity=="GettingReady"], DF_PotRun[DF_PotRun$activity=="GettingReady", PlotAltCol], col=Col_GettingReady, pch=20)
                                  points(DF_PotRun$time_local[DF_PotRun$activity=="Skiing"], DF_PotRun[DF_PotRun$activity=="Skiing", PlotAltCol], col=Col_Skiing, pch=20)
                                  points(DF_PotRun$time_local[DF_PotRun$activity=="PackingUp"], DF_PotRun[DF_PotRun$activity=="PackingUp", PlotAltCol], col=Col_PackingUp, pch=20)
                                  points(DF_PotRun$time_local[DF_PotRun$activity=="Flying"], DF_PotRun[DF_PotRun$activity=="Flying", PlotAltCol], col=Col_Flying, pch=20)
                                  if(UseElevRaster) {
                                    lines(DF_PotRun$time_local, DF_PotRun$gps_altitude, lty=2)
                                  } else {
                                    lines(DF_PotRun$time_local, DF_PotRun$dem_altitude, lty=2)
                                  }
    
                                  mtext(paste(Chart_Title, "- Pot Run", Index_PotRun), outer=T, cex=1, font=2)
    
                                  par(mfrow=c(1,1), oma=c(0,0,0,0))
    
                                  if (WriteChartsToFile) {dev.off()}
    
                                  rm(PlotAltCol)
    
    
                                  ## Creating DF for run with collapsed waiting points
                                  ## *************************************************
    
                                  ## Create output dataframe format
                                  DF_RunPointsTemp <- DF_PotRun[,c("uuid", "gpsruns_uuid", "datetime_local", "gps_longitude", "gps_latitude", "gps_altitude", "dem_altitude", "timediff_sec", "activity")]
    
                                  ## Copy structure of DF
                                  DF_RunPoints <- DF_RunPointsTemp[0,]
                                  DF_RunPointsWaiting <- DF_RunPointsTemp[0,]
    
                                  ## Fill table with original skiing points and collapsed waiting points
                                  Index_Obs <- 1
    
                                  while (Index_Obs <= nrow(DF_RunPointsTemp)) {
    
                                    if (DF_RunPointsTemp$activity[Index_Obs]=="Skiing") {
    
                                      # print(paste(Index_Obs, DF_RunPointsTemp$activity[Index_Obs]))
                                      DF_RunPoints <- rbind(DF_RunPoints, DF_RunPointsTemp[Index_Obs,])
                                      Index_Obs <- Index_Obs+1
    
                                    } else if (DF_RunPointsTemp$activity[Index_Obs]=="GettingReady" | DF_RunPointsTemp$activity[Index_Obs]=="Waiting" | DF_RunPointsTemp$activity[Index_Obs]=="PackingUp") {
    
                                      DF_RunPointsWaiting <- DF_RunPointsTemp[Index_Obs,]
                                      Activity <- DF_RunPointsTemp$activity[Index_Obs]
                                      Index_Obs_Sub <- 1
    
                                      while ((DF_RunPointsTemp$activity[Index_Obs+Index_Obs_Sub]==Activity) & ((Index_Obs + Index_Obs_Sub) <= nrow(DF_RunPointsTemp))) {
    
                                        # print(paste(Index_Obs+Index_Obs_Sub, DF_RunPointsTemp$activity[Index_Obs+Index_Obs_Sub]))
                                        DF_RunPointsWaiting <- rbind(DF_RunPointsWaiting, DF_RunPointsTemp[Index_Obs+Index_Obs_Sub,])
                                        Index_Obs_Sub <- Index_Obs_Sub+1
    
                                      }
    
                                      ## Add collapsed point to list of run points
    
                                      UUIDPausePoint <- createUuid()
                                      DF_RunPoints <- rbind(DF_RunPoints,
                                                            data.frame(uuid=UUIDPausePoint,
                                                                       gpsruns_uuid=DF_RunPointsWaiting$gpsruns_uuid[1],
                                                                       datetime_local=DF_RunPointsWaiting$datetime_local[nrow(DF_RunPointsWaiting)],   ## Date of last point; consistent with time of regular points
                                                                       gps_longitude=mean(DF_RunPointsWaiting$gps_longitude),
                                                                       gps_latitude=mean(DF_RunPointsWaiting$gps_latitude),
                                                                       gps_altitude=round(mean(DF_RunPointsWaiting$gps_altitude),0),
                                                                       dem_altitude=round(mean(DF_RunPointsWaiting$dem_altitude),0),
                                                                       timediff_sec=sum(DF_RunPointsWaiting$timediff_sec),
                                                                       activity=Activity))
    
                                      ## Transfer PausePoint UUID back to DF_Day
                                      for (Index_Points in 1:nrow(DF_RunPointsWaiting)){
                                        DF_Day$gpsruns_pausepoints_uuid[DF_Day$uuid==DF_RunPointsWaiting$uuid[Index_Points]] <- UUIDPausePoint
                                      }
    
                                      ## Update Index
                                      Index_Obs <- Index_Obs+Index_Obs_Sub ## The +1 is included in the Index_Obs_Sub
    
                                    } else {
                                      Index_Obs <- Index_Obs+1
                                    }
    
                                  }
    
                                  ## Clean up
                                  rm(Index_Obs, Index_Obs_Sub, Activity, DF_RunPointsTemp, DF_RunPointsWaiting)
    
                                  ## Calcuate distance between points for run quality measure
                                  DF_RunPoints$dist <- NA
                                  for (Index_Points in 2:nrow(DF_RunPoints)) {
                                    Point1 <- c(DF_RunPoints$gps_longitude[Index_Points-1], DF_RunPoints$gps_latitude[Index_Points-1])
                                    Point2 <- c(DF_RunPoints$gps_longitude[Index_Points],   DF_RunPoints$gps_latitude[Index_Points])
                                    DF_RunPoints$dist[Index_Points] <- round(distHaversine(Point1, Point2),1)
                                  }
                                  rm(Point1, Point2, Index_Points)
    
                                  ## Feedback
                                  OutputString <- paste0(OutputStringPre, "Original number of obs for run line")
                                  print(paste0(OutputString, ": ", nrow(DF_RunPoints)))
                                  Output_Progress <- addToProgressDF(Comment=OutputString, Num=nrow(DF_RunPoints))
                                  rm(OutputString)
    
    
    ############### vvvvvvvvvvvvvvvvvvVVVvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
    ############### 5.4.7.1 CONVERTING PAUSE POINTS TO SPDF_RUN_PAUSEPOINTS ====
    ############### ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    
                                  ## Create dataframe with only pause points (i.e., not skiing)
                                  DF_RunPausePoints <- subset(DF_RunPoints, activity!="Skiing")
                                  row.names(DF_RunPausePoints) <- DF_RunPausePoints$uuid
    
                                  ## Create spatial points
                                  SPP_RunPausePoints <- createSpatialPointsFromCoord(DF_RunPausePoints, x="gps_longitude", y="gps_latitude")
    
                                  ## Eliminate irrelevant columns from dataframe
                                  DF_RunPausePoints <- deleteDFColumnsByName(DF_RunPausePoints, c("gps_longitude", "gps_latitude", "dist"))
    
                                  ## Create SPDF
                                  if (exists("SPDF_Runs_PausePoints")) {
                                    SPDF_Runs_PausePoints <- spRbind(SPDF_Runs_PausePoints, SpatialPointsDataFrame(SPP_RunPausePoints, DF_RunPausePoints))
                                  } else {
                                    SPDF_Runs_PausePoints <- SpatialPointsDataFrame(SPP_RunPausePoints, DF_RunPausePoints)
                                  }
    
                                  ## Clean up
                                  # rm(SPP_RunPausePoints)
                                  rm(DF_RunPausePoints)
    
    
    ############### vvvvvvvvvvvvvvvvvvVVVvvvvvvvvvvvvvvvvvvvvvvv
    ############### 5.4.7.2 CONVERTING POINTS INTO SPDF_RUN ====
    ############### ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    
                                  ## Convert points into SPDF
                                  ## ************************
                                  SPL_Run_Original <- createSpatialLineFromCoord(DF_RunPoints, UUIDRun, x="gps_longitude", y="gps_latitude")
    
    
                                  ## Simplifiying geometry 1: Eliminating loops
                                  ## ******************************************
                                  IsSimple <- gIsSimple(SPL_Run_Original)
    
                                  ## Feedback
                                  OutputString <- paste0(OutputStringPre, "Original run line is simple: ", as.character(IsSimple))
                                  print(OutputString)
                                  Output_Progress <- addToProgressDF(Comment=OutputString)
                                  rm(OutputString)
    
                                  if (IsSimple) {
    
                                    SPL_Run_SimpleV01 <- SPL_Run_Original
    
                                  } else {
    
                                    SPL_Run_SimpleV01 <- eliminateLoopsInLine(SPL_Run_Original, UUIDRun)
    
                                    ## Feedback
                                    OutputString <- paste0(OutputStringPre, "Loop removal: ", nrow(SPL_Run_Original@lines[[1]]@Lines[[1]]@coords)-nrow(SPL_Run_SimpleV01@lines[[1]]@Lines[[1]]@coords), " points removed")
                                    print(OutputString)
                                    Output_Progress <- addToProgressDF(Comment=OutputString)
    
                                  }
    
                                  rm(IsSimple)
    
    
                                  ## Simplify geometry 2: Eliminating bights (U-shaped sections where the closed section is closer than a threshold)
                                  ## ***************************************************************************************************************
    
    
                                  if(!is.na(Simplify_RemoveBightMaxDist)) {
    
                                    ## Get list of point coordinates
                                    In_Coords <- SPL_Run_SimpleV01@lines[[1]]@Lines[[1]]@coords
                                    In_Coords <- data.frame(Index=c(1:nrow(In_Coords)),
                                                            x=In_Coords[,1],
                                                            y=In_Coords[,2],
                                                            Dist=NA,
                                                            Next=NA)
                                    NumPoints <- nrow(In_Coords)
    
                                    ## Loops through all points to determin closest point within max link distance
                                    Index_Point <- 1
                                    while (Index_Point < NumPoints) {
    
                                      ## Determine distance from point to all subsequent points
                                      ActivePoint <- c(In_Coords$x[Index_Point], In_Coords$y[Index_Point])
                                      DistanceToActive <- distHaversine(ActivePoint, In_Coords[c((Index_Point+1):NumPoints),c("x", "y")])
                                      MinDist <- min(DistanceToActive)
    
                                      ## Only merge if distance is smaller than maximum merge distance
                                      if (MinDist<Simplify_RemoveBightMaxDist) {
                                        In_Coords$Dist[Index_Point] <- MinDist
                                        In_Coords$Next[Index_Point] <- Index_Point + which.min(DistanceToActive)
    
                                      } else {
                                        In_Coords$Dist[Index_Point] <- DistanceToActive[1]
                                        In_Coords$Next[Index_Point] <- Index_Point + 1
                                      }
    
                                      Index_Point <- In_Coords$Next[Index_Point]
    
                                      # Clean up
                                      rm(ActivePoint, DistanceToActive, MinDist)
    
                                    }
                                    rm(Index_Point)
    
                                    ## Filling Next field for last point (hypothetical next point)
                                    In_Coords$Next[NumPoints] <- NumPoints+1
    
                                    ## Extract final list of points
                                    Out_Coord <- In_Coords[!is.na(In_Coords$Next), c("x", "y")]
    
                                    ## Turn final line coordinates into spatial line
                                    SPL_Run_SimpleV02 <- createSpatialLineFromCoord(Out_Coord, UUIDRun, x="x", y="y")
    
                                    ## Feedback
                                    OutputString <- paste0(OutputStringPre, "Bight removal: ", nrow(SPL_Run_SimpleV01@lines[[1]]@Lines[[1]]@coords)-nrow(SPL_Run_SimpleV02@lines[[1]]@Lines[[1]]@coords), " points removed")
                                    print(OutputString)
                                    Output_Progress <- addToProgressDF(Comment=OutputString)
    
                                    ## Clean up
                                    rm(In_Coords, Out_Coord, NumPoints)
    
                                  } else {
    
                                    SPL_Run_SimpleV02 <- SPL_Run_SimpleV01
    
                                  }
    
    
                                  ## Simplify geometry 3: Force run line through pause points
                                  ## ********************************************************
    
    
                                  if (!is.na(Simplify_ForcePausePointsReplaceDist)) {
    
                                    ## Forcing line coordinates through pause points
                                    Out_Coord <- forceLineThroughPoints(SPL_Run_SimpleV02, SPP_RunPausePoints, Simplify_ForcePausePointsReplaceDist)
    
                                    ## Turn final line coordinates into spatial line
                                    SPL_Run_SimpleV03 <- createSpatialLineFromCoord(Out_Coord, UUIDRun, x="x", y="y")
    
                                    ## Feedback
                                    OutputString <- paste0(OutputStringPre, "Pause point forcing: ", nrow(SPL_Run_SimpleV03@lines[[1]]@Lines[[1]]@coords)-nrow(SPL_Run_SimpleV02@lines[[1]]@Lines[[1]]@coords), " points added")
                                    print(OutputString)
                                    Output_Progress <- addToProgressDF(Comment=OutputString)
    
                                    ## Clean up
                                    rm(Out_Coord, SPP_RunPausePoints)
    
                                  } else {
    
                                    SPL_Run_SimpleV03 <- SPL_Run_SimpleV02
    
                                  }
    
    
                                  ## Simplify geometry 4: Simplify with gSimplify from rgeos
                                  ## *******************************************************
    
    
                                  if(!is.na(Simplify_SimplifyThreshold)) {
    
                                    ## Simplification
                                    SPL_Run_SimpleV03_Trans <- spTransform(SPL_Run_SimpleV03, Input_CRSLocal)
                                    SPL_Run_SimpleV03_Trans_Simple <- gSimplify(SPL_Run_SimpleV03_Trans, Simplify_SimplifyThreshold)
                                    SPL_Run_SimpleV04 <- spTransform(SPL_Run_SimpleV03_Trans_Simple, Const_GlobalCrsString)
                                    rm(SPL_Run_SimpleV03_Trans, SPL_Run_SimpleV03_Trans_Simple)
    
                                    ## Feedback
                                    OutputString <- paste0(OutputStringPre, "Simplification: ", nrow(SPL_Run_SimpleV03@lines[[1]]@Lines[[1]]@coords)-nrow(SPL_Run_SimpleV04@lines[[1]]@Lines[[1]]@coords) , " points removed")
                                    print(OutputString)
                                    Output_Progress <- addToProgressDF(Comment=OutputString)
                                    rm(OutputString)
    
                                  } else {
    
                                    SPL_Run_SimpleV04 <- SPL_Run_SimpleV03
    
                                  }
    
    
                                  ## Finish simplification sequence with final check for IsSimple
                                  ## ************************************************************
    
                                  if (gIsSimple(SPL_Run_SimpleV04)) {
    
                                    SPL_Run_Final <- SPL_Run_SimpleV04
    
                                  } else {
    
                                    SPL_Run_Final <- eliminateLoopsInLine(SPL_Run_SimpleV04, UUIDRun)
    
                                    OutputString <- paste0(OutputStringPre, "Second round of simplification.")
                                    print(OutputString)
                                    Output_Progress <- addToProgressDF(Comment=OutputString)
                                    rm(OutputString)
    
                                  }
    
    
                                  ## Check whether line long enough
                                  ## ******************************
                                  RunDistanceFinal <- round(SpatialLinesLengths(spTransform(SPL_Run_Final, CRS(Input_CRSLocal))),0)
    
                                  if (RunDistanceFinal < Run_MinLength) {
    
                                    ## Feedback
                                    OutputString <- paste0(OutputStringPre, "Final length of run line below min threshold (", Run_MinLength, " m): ", RunDistanceFinal, " m")
                                    print(OutputString)
                                    Output_Progress <- addToProgressDF(Comment=OutputString)
                                    DF_PotRunsSkipped <- writeReasonForSkippedRunIntoDF(DF_PotRunsSkipped, UUIDTrackDay, Index_PotRun, OutputString)
                                    rm(OutputString, RunDistanceFinal)
    
                                    ## Delete pause points
                                    SPDF_Runs_PausePoints <- SPDF_Runs_PausePoints[SPDF_Runs_PausePoints$gpsruns_uuid!=UUIDRun,]
                                    if (nrow(SPDF_Runs_PausePoints)==0) {rm(SPDF_Runs_PausePoints)}
    
                                    ## Delete run and pause_point uuid reference in DF_Day
                                    DF_Day$gpsruns_pausepoints_uuid[DF_Day$gpsruns_uuid==UUIDRun] <- NA
                                    DF_Day$gpsruns_uuid[DF_Day$gpsruns_uuid==UUIDRun] <- NA
    
                                    ## Delete activity assignments
                                    DF_Day$activity[DF_Day$run_numpot_orig==Index_PotRun & DF_Day$marker_incatarea==1] <- "CatRiding"
                                    DF_Day$activity[DF_Day$run_numpot_orig==Index_PotRun & DF_Day$marker_incatarea==0] <- "Flying"
    
                                    ## Feedback
                                    OutputString <- paste0(OutputStringPre, "RUN NOT PROCESSED FURTHER!")
                                    print(OutputString)
                                    Output_Progress <- addToProgressDF(Comment=OutputString)
                                    rm(OutputString)
    
                                  } else {
    
                                    ## Feedback: Length of run line
                                    OutputString <- paste0(OutputStringPre, "Final length of run line above min threshold (", Run_MinLength, " m): ", RunDistanceFinal, " m")
                                    print(OutputString)
                                    Output_Progress <- addToProgressDF(Comment=OutputString)
                                    rm(OutputString, RunDistanceFinal)
    
                                    ## Feedback: Simple geometry
                                    OutputString <- paste0(OutputStringPre, "Final run line is simple: ", as.character(gIsSimple(SPL_Run_Final)))
                                    print(OutputString)
                                    Output_Progress <- addToProgressDF(Comment=OutputString)
                                    rm(OutputString)
    
                                    OutputString <- paste0(OutputStringPre, "RUN SUCCESSFULLY EXTRACTED!")
                                    print(OutputString)
                                    Output_Progress <- addToProgressDF(Comment=OutputString)
                                    rm(OutputString)
    
    
                                    ## Plotting
                                    ## ********
    
                                    if (WriteChartsToFile) {png(paste0(RunChart_FileNameStart, "03_TrueMap.png"), width=Chart_Width, height=Chart_Height)}
    
                                    par(mar=c(2.1,2.1,4.1,2.1))
                                    plot(SPL_Run_Original, main=paste(Chart_Title, "- Pot Run", Index_PotRun), col="red")
                                    lines(SPL_Run_SimpleV01, col="purple")
                                    lines(SPL_Run_SimpleV02, col="orange")
                                    lines(SPL_Run_SimpleV03, col="brown")
                                    lines(SPL_Run_SimpleV04, col="green")
                                    lines(SPL_Run_Final, col="black")
                                    points(DF_RunPoints$gps_longitude[DF_RunPoints$activity=="GettingReady"], DF_RunPoints$gps_latitude[DF_RunPoints$activity=="GettingReady"], col=Col_GettingReady, pch=19)
                                    points(DF_RunPoints$gps_longitude[DF_RunPoints$activity=="Waiting"], DF_RunPoints$gps_latitude[DF_RunPoints$activity=="Waiting"], col=Col_Pause, pch=19)
                                    points(DF_RunPoints$gps_longitude[DF_RunPoints$activity=="Waiting" & DF_RunPoints$timediff_sec>=Break_MinTime], DF_RunPoints$gps_latitude[DF_RunPoints$activity=="Waiting" & DF_RunPoints$timediff_sec>=Break_MinTime], col=Col_Break, pch=19)
                                    points(DF_RunPoints$gps_longitude[DF_RunPoints$activity=="PackingUp"], DF_RunPoints$gps_latitude[DF_RunPoints$activity=="PackingUp"], col=Col_PackingUp, pch=19)
                                    box()
    
                                    if(Flag_CatSkiing) {
                                      lines(SPDF_CatRoads, col="dark red", lty=2)
                                      lines(SPDF_CatRoads_ForRunIdentification, col="dark red", lty=1)
                                    }
    
                                    par(mar=c(5.1,4.1,4.1,2.1))
    
                                    if (WriteChartsToFile) {dev.off()}
    
    
                                    ## Add data to run lines and convert into SPDF
                                    ## *******************************************
    
                                    SPL_Run_Data <- data.frame(uuid=UUIDRun,
                                                               gpstracksdays_uuid=UUIDTrackDay,
                                                               date_local=dates(DF_RunPoints$datetime_local[1]),
                                                               date_local_esri=dates(DF_RunPoints$datetime_local[1]),
                                                               runnum=NA,
                                                               runnum_rev=NA,
                                                               runnumpot=Index_PotRun,
                                                               datetime_local_start=DF_RunPoints$datetime_local[1],
                                                               datetime_local_end=DF_RunPoints$datetime_local[nrow(DF_RunPoints)],
                                                               numpoints_orig=nrow(DF_RunPoints),
                                                               numpoints_afterloop=nrow(SPL_Run_SimpleV01@lines[[1]]@Lines[[1]]@coords),
                                                               numpoints_afterbight=ifelse(!is.na(Simplify_RemoveBightMaxDist), nrow(SPL_Run_SimpleV02@lines[[1]]@Lines[[1]]@coords), NA),
                                                               numpoints_afterpauseforce=ifelse(!is.na(Simplify_ForcePausePointsReplaceDist), nrow(SPL_Run_SimpleV03@lines[[1]]@Lines[[1]]@coords), NA),
                                                               numpoints_aftersimplify=ifelse(!is.na(Simplify_SimplifyThreshold), nrow(SPL_Run_SimpleV04@lines[[1]]@Lines[[1]]@coords), NA),
                                                               numpoints_final=nrow(SPL_Run_Final@lines[[1]]@Lines[[1]]@coords),
                                                               markerstart_why=Start_Reason,
                                                               markerend_why=End_Reason,
                                                               pointdist_avg=round(mean(DF_RunPoints$dist, na.rm=T),1),
                                                               pointdist_max=max(DF_RunPoints$dist, na.rm=T),
                                                               skispd_avg=SkiingSpdAvg,
                                                               skispd_max=SkiingSpdMax,
                                                               timediff_sec=NA,
                                                               length_m=NA,
                                                               dem_altitude_max=round(max(DF_RunPoints$dem_altitude, na.rm=T),0),
                                                               dem_altitude_min=round(min(DF_RunPoints$dem_altitude, na.rm=T),0),
                                                               gps_altitude_max=round(max(DF_RunPoints$gps_altitude, na.rm=T),0),
                                                               gps_altitude_min=round(min(DF_RunPoints$gps_altitude, na.rm=T),0),
                                                               flag_delete="No",
                                                               comment=NA)
    
                                    row.names(SPL_Run_Data) <- UUIDRun
    
                                    if (exists("SPDF_Runs")) {
                                      SPDF_Runs <- spRbind(SPDF_Runs, SpatialLinesDataFrame(SPL_Run_Final, SPL_Run_Data))
                                    } else {
                                      SPDF_Runs <- SpatialLinesDataFrame(SPL_Run_Final, SPL_Run_Data)
                                    }
    
                                    ## Clean-up
                                    ## ********
                                    rm(Start_Reason, End_Reason)
                                    rm(SPL_Run_Final, SPL_Run_Data)
                                    rm(SPL_Run_Original, SPL_Run_SimpleV01, SPL_Run_SimpleV02, SPL_Run_SimpleV03, SPL_Run_SimpleV04)
                                    # rm(DF_PotRun, DF_RunPoints)
    
    
                                    ## Update counter
                                    ## **************
                                    Output_Feedback$NumRunsIdentified <- Output_Feedback$NumRunsIdentified+1
    
    
                                  } ## End of if-else condition: "(nrow(SPL_Run_Final@lines[[1]]@Lines[[1]]@coords)<20)"
                                  
                                } ## End of if-else condition: "(PercentObsOnCatRoad > CatRoad_MaxPercentageOfRun)"
  
                              } ## End of if-else condition: "(SkiingSpdMax<Run_MinMaxSkiingSpeed)"
  
                            }  ## End of if-else condition: "(nrow(subset(DF_PotRun, activity=="Skiing"))==0)"
                            
                          } ## End of if-condition for waiting points.

                        } ## End of if-condition: "(nrow(DF_PotRun) < NumObsRunMin)"

                      } ## End of if-else condition: "nrow(Run_PeriodSummary_DownOnly)==0"

                    } ## End of if-condition: "(nrow(DF_Day[DF_Day$run_numpot==Index_PotRun])==0)"

                  } ## End of for loop for Index_PotRun (processing potential runs)

                  ## Clean-up
                  # rm (Index_PotRun)
                  rm(Num_PotRun, OutputStringPre)


                  ## Complete run information in SPDF_Runs
                  ## *************************************

                  if(Flag_AnyRunsProcessed & exists("SPDF_Runs")) {

                    ## Adding correct run number (forward and backwards)
                    SPDF_Runs@data$runnum <- c(1:nrow(SPDF_Runs@data))
                    SPDF_Runs@data$runnum_rev <- c(nrow(SPDF_Runs@data):1)

                    ## Calcuating duration
                    SPDF_Runs@data$timediff_sec <- (SPDF_Runs@data$datetime_local_end-SPDF_Runs@data$datetime_local_start)*24*60*60

                    ## Calculate length
                    SPDF_Runs@data$length_m <- round(gLength(spTransform(SPDF_Runs, CRS(Input_CRSLocal)), byid=T),0)

                  }


                  ## Complete obs point information in DF_Day
                  ## ****************************************

                  ## Add cat riding as activity
                  DF_Day$activity[(DF_Day$marker_incatarea==1 & is.na(DF_Day$activity))] <- "CatRiding"


                  ## User feedback
                  ## *************
                  if (Flag_AnyRunsProcessed & exists("SPDF_Runs")) {
                    OutputString <- paste0("NUMBER OF EXTRACTED RUNS: ", nrow(SPDF_Runs@data))
                  } else {
                    OutputString <- paste0("NUMBER OF EXTRACTED RUNS:  0")
                  }
                  print("============================")
                  print(OutputString)
                  print("============================")
                  Output_Progress <- addToProgressDF(Comment=OutputString)
                  rm(OutputString)


######### vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
######### 5.4.8 WRITING TO DB OR CSV ====
######### ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

                  ## Info_TrackDays
                  writeInfotrackDaysToDBorCSV(Operation, Info_TrackDays, WriteToDB, CSVFileName=paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "Info_TrackDays.csv"), DBType=DBType)

                  ## Potential runs skipped
                  writePotRunSkippedToDBorCSV(Operation, DF_PotRunsSkipped, WriteToDB, CSVFileName=paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "PotRunsSkipped.csv"), DBType=DBType)
                  
                  ## SPDF_Runs
                  if(Flag_AnyRunsProcessed & exists("SPDF_Runs")) {
                    if(WriteToDB) {
                      writeSPDFToGPSDB(SPDF_Runs, Operation, "gps", "runs", DBType=DBType, ObjLabel="runs")
                    } else {
                      writeSPDFToCSV(SPDF_Runs, paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "SPDF_Runs.csv"), na = "")
                    }
                  }

                  ## SPDF_Runs_PausePoints
                  if(Flag_AnyRunsProcessed & exists("SPDF_Runs_PausePoints")) {
                    if(WriteToDB) {
                      writeSPDFToGPSDB(SPDF_Runs_PausePoints, Operation, "gps", "runs_pausepoints", DBType=DBType, ObjLabel="pause points", ShowCounter=20)
                    } else {
                      writeSPDFToCSV(SPDF_Runs_PausePoints, paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "SPDF_Runs_PausePoints.csv"), na = "")

                    }
                  }

                  ## Turn DF_Day into SPDF and write to DB or CSV
                  DF_Day <- deleteDFColumnsByName(DF_Day, "time_local")
                  writeDFDayToDBorCSV(Operation, DF_Day, WriteToDB, CSVFileName=paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "SPDF_ObsPoints.csv"), DBType=DBType)


                  ## User feedback
                  if(WriteToDB) {
                    if(Flag_AnyRunsProcessed & exists("SPDF_Runs")) {
                      OutputString <- "SPDF_ObsPoints, SPDF_Runs and SPDF_Runs_PausePoints written to DB."
                    } else {
                      OutputString <- "SPDF_ObsPoints written to DB."
                    }
                  } else {
                    if(Flag_AnyRunsProcessed & exists("SPDF_Runs")) {
                      OutputString <- "SPDF_ObsPoints, SPDF_Runs and SPDF_Runs_PausePoints written to CSV."
                    } else {
                      OutputString <- "SPDF_ObsPoints written to CSV."
                    }
                  }
                  print(OutputString)
                  Output_Progress <- addToProgressDF(Comment=OutputString)
                  rm(OutputString)

                  ## Clean-up
                  rm(DF_StartEndSummary)


                  ## Overview plot
                  ## *************
                  if(Flag_AnyRunsProcessed & exists("SPDF_Runs")) {

                    if (WriteChartsToFile) {png(paste0(Folder_OutputRoot, createOSSpecFolderLink(), DayTrackFolderName, createOSSpecFolderLink(), "Overview_04_OverviewMap.png"), width=Chart_Width, height=Chart_Height)}

                    par(mar=c(2.1,2.1,4.1,2.1))
                    plot(SPDF_Runs, main=paste(Chart_Title, "- Overview Map"))
                    text(gCentroid(SPDF_Runs, byid=T), c(1:nrow(SPDF_Runs)), font=2, cex=0.75, col="blue")
                    box()
                    if(Flag_CatSkiing) {
                      lines(SPDF_CatRoads, col="dark red", lty=2)
                      lines(SPDF_CatRoads_ForRunIdentification, col="dark red", lty=1)
                    }
                    par(mar=c(5.1,4.1,4.1,2.1))

                    if (WriteChartsToFile) {dev.off()}
                  }


                  ## Create run list
                  ## ***************
                  if(Flag_AnyRunsProcessed & exists("SPDF_Runs")) {

                    if (WriteToDB & InclRunUseList) {
                      print("")
                      print("Run use list")
                      print("************")

                      print(getRunUsageList(Operation, DayDateString, Info_Track$unit[1]))
                      print("")
                    }
                  }

                } ## End of if-else condition: "nrow(DF_StartEndSummary)==0"


              } ## End of if-else condition: "nrow(DF_Day[DF_Day$marker_inoparea==1 & DF_Day$marker_inexclarea!=1,])<1"

              ## Update counter
              Output_Feedback$NumTrackDaysProcessed <- Output_Feedback$NumTrackDaysProcessed + 1


            } ## End of if-else condition: "if (nrow(DF_Day)==0)"


          }  ## End of if-else condition: "if (DBRunRecordsCheck_NumObsPoints>0 & (WriteToDB | OnlyProcessTrackdaysNotInDB))"


          ## Close sink output file
          print(Sys.time())
          sink()


        }  ## End of for loop Index_Date

        ## Clean-up
        # rm(Index_Date)

      } ## End of if-else condition that checks whether there are any GPS records in the track after the initial processing


      ## User feedback
      ## *************
      Output_Feedback$NumTracks <- Output_Feedback$NumTracks + 1


    } ## End of for loop with Index_Track

    ## Clean-up
    # rm(Index_Track)
    # rm(DF_Track)


  } ## End of if condition length(GPSTrackFileList)>0


## vvvvvvvvvvvvvvvvvvvv
## 6. FINISHING UP ====
## ^^^^^^^^^^^^^^^^^^^^

  ## Progress output
  ## ***************
  Output_Progress <- addToProgressDF()
  Output_Progress <- addToProgressDF(Comment="SUMMARY")
  Output_Progress <- addToProgressDF(Comment="*******")
  Output_Progress <- addToProgressDF(Comment="Number of tracks processed", Num=Output_Feedback$NumTracks)
  Output_Progress <- addToProgressDF(Comment="Number of track-days processed", Num=Output_Feedback$NumTrackDaysProcessed)
  Output_Progress <- addToProgressDF(Comment="Number of track-days skipped", Num=Output_Feedback$NumTrackDaysSkipped)
  Output_Progress <- addToProgressDF(Comment="Number of runs identified", Num=Output_Feedback$NumRunsIdentified)


  ## Write Output_Progress file to CSV
  ## *********************************
  write.csv(Output_Progress, file=paste0(Folder_OutputRoot, createOSSpecFolderLink(), "Output_Progress_", strftime(Sys.time(), "%Y%m%d_%H%M%S"), ".csv"), na = "")

  ## Final User feedback
  ## *******************
  Output_Feedback$EndTime <- Sys.time()
  print(paste("Last Track Index:", Index_Track, "of", length(GPSTrackFileList)))

  return(Output_Feedback)

}
